#!/usr/bin/env python
#UCAS molecular simulation, Summer 2016
#3D Visual Python Routines
#Written by Ryan Cocking & Robert Forrest

from visual import *
import itp_main as main
import itp_params as params
import numpy as npy
import time

#----------------------------#
#          DRAWING           #
#----------------------------#

#Draw particle container
def build_box(x,y,z=(0.,0.)):
    # Front face
    bounding_box = [curve(pos=[(x[0],y[0],z[0]), (x[1],y[0],z[0]),
                (x[1],y[1],z[0]), (x[0],y[1],z[0]), 
                (x[0],y[0],z[0])])]
    if (params.Ndim == 3):
        # Back face
        bounding_box.append(curve(pos=[(x[0],y[0],z[1]), (x[1],y[0],z[1]),
                   (x[1],y[1],z[1]), (x[0],y[1],z[1]), 
                   (x[0],y[0],z[1])]))
        # Connecting lines
        bounding_box.append(curve(pos=[(x[0],y[0],z[0]), (x[0],y[0],z[1])]))
        bounding_box.append(curve(pos=[(x[1],y[0],z[0]), (x[1],y[0],z[1])]))
        bounding_box.append(curve(pos=[(x[0],y[1],z[0]), (x[0],y[1],z[1])]))
        bounding_box.append(curve(pos=[(x[1],y[1],z[0]), (x[1],y[1],z[1])]))
    
    return bounding_box

#Initialise visuals
def build_visuals():
    global scene, fps, bounding_box

    #Create the scene
    
    scene = display(title='Illustrating Theoretical Physics', x=360, y=30, width=700, height=700, center = [params.box_width*0.5 for i in xrange(params.Ndim)],range=(1.1*(params.box_width/2.0)))
    scene.lights        = []
    scene.ambient       = color.white
    fps                 = 0

    bounding_box = build_box(*([0,params.box_width] for i in xrange(params.Ndim)))
    if (params.Ndim == 2):
        scene.userspin      = False
        scene.userzoom      = False     
    return None

def build_atoms(pos):
    global atoms
    # Create array of atoms
    atoms       = npy.asarray([sphere(pos=vector(atom), radius=0.5, color=color.blue) for atom in pos])
    
def update_scene():
    global bounding_box
    global scene
    
    #Create the bounding box
    bounding_box = build_box(*((0,params.box_width) for i in xrange(params.Ndim)))
    #Center and range the scene
    scene.center = [params.box_width*0.5 for i in xrange(params.Ndim)]
    scene.range         = 0.55*params.box_width

# Update positions and colours of atoms
def update_visuals(q,q2=None):
    global atoms
    global scene
    global N
    global continuing
    global bonds, bond_line, fixed
    global speed_color
    global fps
    
    # Initialise bond lines
    bond_line           = []
    
    # Initial values for fps counter
    frame_times         = []
    start_t             = time.time()
    
    continuing = True
    while continuing:
        #Limit framerate so that sim doesnt run too fast
        rate(30)
        
        # Get values from main thread
        pos     = q.get()
        v       = q.get()
        if (q2):
            #Handle moving individual atoms with the mouse
            picked_atom_index       = -1
            dragging_force          = npy.zeros(2)
            dragging                = False
            drag_line               = None
            already_updated         = False

            # If there is a mouse click
            if scene.mouse.events:
                m1 = scene.mouse.getevent()
                if m1.drag and m1.pick!=None:
                    # Find atom clicked on
                    picked_atom_index       = npy.where(atoms==m1.pick)
                    picked_atom_index       = picked_atom_index[0][0]
                    dragging                = True
    
            #While an atom is being dragged
            while dragging:
                rate(30)
                
                collected_from_queue        = False
                # If needed, collect data from main thread
                if(pos==None):
                    collected_from_queue    = True
                    pos                     = q.get()
                    v                       = q.get()
                else:
                    q.task_done()
                    q.task_done()
                
                # project mouse pos onto xy plane
                new_pos     = scene.mouse.project(normal=(0,0,1))
    
                # Draw dragging line
                if(drag_line!= None):
                    drag_line.visible = False
                    del drag_line
                drag_line = curve(pos=[vector(pos[picked_atom_index]), new_pos], radius=0.05)
                
                # Calculate displacement between atom and mouse pos
                displacement        = npy.asarray(new_pos)[0:2] - pos[picked_atom_index]
                # Place dragging data into queue, to be collected by main thread
                q2.put(dragging)
                q2.put(picked_atom_index)
                q2.put(displacement)
                
                # Check if dragging has stopped
                if scene.mouse.events:
                    m1 = scene.mouse.getevent()
                    if m1.drop:
                        main.picked_atom_index      = -1
                        main.displacement           = npy.zeros(2)
                        main.dragging               = False
                        dragging                    = False
                        drag_line.visible           = False
                        del drag_line
                
                #Update positions and colours
                already_updated     = True
                update_positions_colors(pos, v, atoms, bonds, bond_line, fixed)
                
                pos                 = None
                T                   = None
                
                if(collected_from_queue==True):
                    q.task_done()
                    q.task_done()
            
            # Make sure that visuals havent already been updated in dragging loop
            if(already_updated==False):
                update_positions_colors(pos, v, atoms, bonds, bond_line, fixed)
                q2.put(dragging)
                
                q.task_done()
                q.task_done()
        else:
            update_positions_colors(pos, v, atoms, bonds, bond_line, fixed)
            q.task_done()
            q.task_done()
        
        # Calculate frames per second
        end_t           = time.time()
        time_taken      = end_t - start_t
        start_t         = end_t
        frame_times.append(time_taken)
        frame_times     = frame_times[-20:]
        fps             = len(frame_times) / sum(frame_times)
        
def update_positions_colors(pos, v, atoms, bonds, bond_line, fixed):
    #Update positions and colours
    for i in xrange(params.N):        
        atoms[i].pos    = pos[i]
        
        #Produce a value proportional to atom speed, used for colouring
        speed           = npy.sqrt(npy.dot(v[i],v[i]))
        normalised_v    = (speed*params.N)/(params.speed_colour)
            
        #Colour the atoms based on their speed
        atoms[i].color  = (normalised_v,0,1.-normalised_v)
        
    # Fixed atoms are gray
    for fix in fixed:
        atoms[fix].color = color.gray(0.5)
        
    #Create bond lines
    for i in xrange(len(bonds)):
        if(bond_line[i]!=0):
            bond_line[i].visible = False
            
        bond_line[i] = curve(pos=[vector(pos[bonds[i][0]]), vector(pos[bonds[i][1]])], radius=0.06)
 
    
# Remove all visual objects
def destroy_objects():
    global atoms
    global bond_line
    global bounding_box

    for bond in bond_line:
        bond.visible = False
    for atom in atoms:
        atom.visible = False
    for line in bounding_box:
        line.visible = False

