//////////////////////////////////////////////////////////////////////////////////////////////////
//
//  GAC Schema starts here

//  A tuple and pa are intended to be immutable, although this is hard to enforce.
//  Could be done by copying the arrays passed in, and making the attributes final private, but copying is inefficient.

//  tuple is a complete assignment to the variables in the constraint. Recorded as values only.

// 12/6/06 modified to change initial length of lists etc.

package queso.constraints;

import java.util.*;
import gnu.trove.*;
import queso.core.*;


final class linktuple
{  // a doubly linked list of tuples with fast remove
    final int id; // identifier for this list; index into reference arrays in tuple objects.
    
    private tuple first=null;
    private tuple last=null;
    
    linktuple(int id)
    {
        this.id=id;
    }
    
    public tuple first()
    {
        return first;
    }
    
    boolean in(tuple tau)
    {
        assert really_in(tau)==tau.in_list[id];
        return tau.in_list[id];
    }
    
    boolean really_in(tuple tau)
    {
        tuple current=first;
        while(current!=null)
        {
            if(current.equals(tau))
            {
                assert current==tau;
                return true;
            }
            current=current.next[id];
        }
        return false;
    }
    
    public void insert_end(tuple tau)
    {
        assert !in(tau);
        
        if(first!=null)
        {  // if list not empty
            last.next[id]=tau;
            tau.prev[id]=last;
            tau.next[id]=null;
            last=tau;
        }
        else  // list empty
        {
            first=tau;
            last=tau;
            tau.prev[id]=null;
            tau.next[id]=null;
        }
        
        tau.in_list[id]=true;
        assert in(tau);
    }
    
    public tuple next(tuple tau)
    {
        return tau.next[id];
    }
    
    boolean contains(tuple tau)
    {
        tuple current=first;
        while(current!=null)
        {
            assert current.in_list[id];
            if(tau==current)
                return true;
            
            current=current.next[id];
        }
        return false;
    }
    
    void check(tuple tau)
    {
        // check if tau's in_list matches its in listness.
        if (contains(tau))
        {
            assert tau.in_list[id];
        }
        else
        {
            assert !tau.in_list[id];
        }
    }
    
    // constant-time removal -- this is the important feature.
    public void remove(tuple tau)
    {   // remove tau
        // assuming tau really is in the list.
        
        assert in(tau);
        
        //System.out.println(tau.prev[id]+""+tau.next[id]);
        
        if(tau.prev[id]!=null)
        {
            tau.prev[id].next[id]=tau.next[id];
        }
        else
        {
            first=tau.next[id];
        }
        
        if(tau.next[id]!=null)
        {
            tau.next[id].prev[id]=tau.prev[id];
        }
        else
        {
            last=tau.prev[id];
        }
        
        tau.in_list[id]=false;
        assert !in(tau);
        //assert first==null || last!=null;
    }
    
    public void restore(tuple tau)
    {
        assert !in(tau);
        
        if(tau.prev[id]!=null)
        {
            tau.prev[id].next[id]=tau;
        }
        else
        {
            first=tau;
        }
        
        if(tau.next[id]!=null)
        {
            tau.next[id].prev[id]=tau;
        }
        else
        {
            last=tau;
        }
        tau.in_list[id]=true;
        
        assert in(tau);
    }
    
    public String toString()
    {
        String st="id:"+id+", ";
        tuple pointer=first;
        while(pointer!=null)
        {
            st+=pointer+", ";
            pointer=pointer.next[id];
        }
        return st;
    }
}

// pa (partial assignment) is a partial assignment to the variables in the constraint.
// also intended to be immutable.

final class pa
{
    pa next;
    pa prev;
    
    boolean in_list=false;
    
    // index into the supports_current array.
    int scindex=-1;
    
    pa(int [] vars, int [] vals)
    {
        this.vars=vars;
        this.vals=vals;   // assumes same length
    }
    
    final int [] vars;
    final int [] vals;
    
    public String toString()
    {
        String st="";
        for(int i=0; i<vars.length; i++)
            st+=vars[i]+" = "+vals[i]+", ";
        return st;
    }
    
    public int hashCode()   // this is probably rubbish
    {
        int total=0;
        for(int i=0; i<vars.length; i++)
        {
            total = (total+vars[i]) << 1;
            total = (total+vals[i]) << 1;
        }
        return total;
    }
    
    public boolean equals(Object pa2)
    {
        if(pa2 instanceof pa)
        {
            pa partial=(pa) pa2;
            if(partial.vars.length!=this.vars.length)
            {
                return false;
            }
            else
            {
                for(int i=0; i<this.vars.length; i++)
                {
                    if(this.vars[i]!=partial.vars[i])
                    {
                        return false;
                    }
                    if(this.vals[i]!=partial.vals[i])
                    {
                        return false;
                    }
                }
                return true;
            }
        }
        else
        {
            return false;
        }
    }
}

final class linkpa
{  // a doubly linked list of pa's with fast remove
    pa first;
    pa last;
    
    linkpa()
    {
    
    }
    
    public void insert_end(pa tau)
    {
        assert !tau.in_list;
        if(first!=null)
        {  // if list not empty
            last.next=tau;
            tau.prev=last;
            tau.next=null;
            last=tau;
        }
        else
        {
            first=tau;
            last=tau;
            tau.prev=null;
            tau.next=null;
        }
        tau.in_list=true;
    }
    
    public boolean in(pa partial)
    {
        pa current=first;
        while(current!=null)
        {
            if(current.equals(partial))
            {
                assert current==partial;
                return true;
            }
            current=current.next;
        }
        return false;
    }
    
    public pa pop()
    {
        // pop the first element off the list, or return null.
        
        pa popitem=first;
        
        if(popitem!=null)
        {
            assert popitem.in_list;
            popitem.in_list=false;
            first=first.next;  // may be null
            if(first!=null) first.prev=null;
        }
        return popitem;
    }
    
    // constant-time removal -- this is the important feature.
    public void remove(pa tau)
    {   // remove tau
        // assuming tau really is in the list.
        assert tau.in_list;
        assert in(tau);
        if(tau.prev!=null)  // not first
        {
            tau.prev.next=tau.next;
        }
        else
        {
            first=tau.next;
        }
        
        if(tau.next!=null)  // not last
        {
            tau.next.prev=tau.prev;
        }
        else
        {   // last element
            assert last==tau;
            last=tau.prev;
        }
        tau.in_list=false;
        assert !in(tau);
    }
    
    // this is dangerous and should not be used, because
    // the list pointers are overwritten after a pa is removed from
    // literals_by_support.
    public void restore(pa tau)
    {
        System.out.println("Restoring: "+tau);
        assert !tau.in_list;
        assert !in(tau);
        if(tau.prev!=null)
        {
            tau.prev.next=tau;
        }
        else
        {
            assert tau.next==first;
            first=tau;
        }
        
        if(tau.next!=null)
        {
            tau.next.prev=tau;
        }
        else
        {
            last=tau;
        }
        tau.in_list=true;
        assert in(tau);
    }
    
    public boolean empty()
    {
        return first==null;
    }
    
    public String toString()
    {
        String st="";
        pa pointer=first;
        while(pointer!=null)
        {
            st+=pointer+", ";
            pointer=pointer.next;
        }
        return st;
    }
}

final class supports_current_backtrack extends backtrack
{
    ArrayList<tuple> tuples;
    ArrayList<pa> partials;
    TIntArrayList removals;  // is it a removal or addition? 0= addition 1=removal
    
    final gac_schema constraint;
    
    supports_current_backtrack(gac_schema cons)
    {
        constraint=cons;
        
        tuples= new ArrayList<tuple>(1000);
        partials= new ArrayList<pa>(1000);
        removals=new TIntArrayList(1000);
    }
    
    final void add_addition(pa partial, tuple tau)
    {
        //System.out.println("adding addition tuple "+tau+" and partial "+partial);
        tuples.add(tau);
        partials.add(partial);
        removals.add(0);  // 0 for false, 1 for true
        backtrack_increment();
    }
    
    final void add_removal(pa partial, tuple tau)
    {
        //System.out.println("adding removal tuple "+tau+" and partial "+partial);
        tuples.add(tau);
        partials.add(partial);
        removals.add(1);  // 0 for false, 1 for true
        backtrack_increment();
    }
    
    public void backtrack_item()
    {
        int index=partials.size()-1;
        pa partial=partials.remove(index);
        tuple tau=tuples.remove(index);
        int removal=removals.remove(index);
        
        if(removal==1)
        {
            constraint.supports_current.get(partial.scindex).restore(tau);
        }
        else
        {
            assert removal==0;
            constraint.supports_current.get(partial.scindex).remove(tau);
        }
    }
}

final class literals_by_support_backtrack extends backtrack
{
    ArrayList<tuple> tuples;
    ArrayList<pa> partials;
    TIntArrayList removals;
    
    gac_schema constraint;
    
    literals_by_support_backtrack(gac_schema cons)
    {
        constraint=cons;
        
        tuples= new ArrayList<tuple>(1000); // 10000
        partials= new ArrayList<pa>(1000);
        removals=new TIntArrayList(1000);
    }
    
    void add_addition(tuple tau, pa partial)
    {
        // tau -> partial was added to the literals_by_support
        assert ((linkpa)constraint.literals_by_support.get(tau)).in(partial); // it has really been added.
        assert partial.in_list;
        tuples.add(tau);
        partials.add(partial);
        removals.add(0);  // 0 for false, 1 for true
        backtrack_increment();
    }
    
    void add_removal(tuple tau, pa partial)
    {
        // tau -> partial was removed from literals_by_support
        assert !((linkpa)constraint.literals_by_support.get(tau)).in(partial); // it has really been removed.
        assert !partial.in_list;
        tuples.add(tau);
        partials.add(partial);
        removals.add(1);  // 0 for false, 1 for true
        backtrack_increment();
    }
    
    public void backtrack_item()
    {
        int index=partials.size()-1;
        tuple t=tuples.remove(index);
        pa p=partials.remove(index);
        int isremoval=removals.remove(index);
        
        if(isremoval==1)
        {
            // this was a removal, so make it an addition.
            
            linkpa list=(linkpa)constraint.literals_by_support.get(t);
            //list.restore(p);
            if(list==null)
            {
                list=new linkpa();
                constraint.literals_by_support.put(t, list);
            }
            
            list.insert_end(p); // can't assume that its list pointers have not been messed up.
            // This is because, typically, when a pa is removed from one list (for one supporting tuple)
            // it is immediately inserted into another for the replacement support.
        }
        else
        {
            assert isremoval==0;
            // this was an addition, so make a removal.
            //System.out.println("Removing "+p+" from list "+t);
            linkpa list=(linkpa)constraint.literals_by_support.get(t);
            list.remove(p);
            if(list.empty()) // potential bug.
            {
                constraint.literals_by_support.remove(t);
            }
        }
    }
}

final class last_support_backtrack extends backtrack
{
    // last_support maps a partial to a tuple.
    
    ArrayList<tuple> tuples;
    ArrayList<pa> partials;
    // These actions obviously need to be undone in reverse sequence.
    
    final gac_schema constraint;
    
    last_support_backtrack(gac_schema cons)
    {
        constraint=cons;
        
        tuples= new ArrayList<tuple>(1000);
        partials= new ArrayList<pa>(1000);
    }
    
    void add_action(pa partial, tuple tau)
    {
        //System.out.println("adding addition tuple "+tau+" and partial "+partial);
        tuples.add(tau);
        partials.add(partial);
        backtrack_increment();
    }
    
    public void backtrack_item()
    {   // backtrack one item.
        int index=partials.size()-1;
        tuple t=tuples.remove(index);
        pa p=partials.remove(index);
        constraint.last_support.set(p.scindex, t);
    }
}


// this should not be public; it is so because of a hack in the constraint queue.
public abstract class gac_schema extends constraint implements fd_propagate, stateful
{
    //THashMap supports_current;   // <pa, linktuple>currently used, allowed tuples. supports_current[partial] = tuple linked list
    final ArrayList<linktuple> supports_current;  // indexed like this: supports_current[pa.scindex] for a linked list of current supports.
    THashMap literals_by_support;     // <tuple, linkpa>  partial assignments supported by a tuple. literals_by_support[tuple]= PA linked list
    //THashMap last_support;       // <pa, tuple> lastc i.e. the previous found support for this PA. 
    final ArrayList<tuple> last_support;
    final int [] num_uni_inside;  // the number of universal variables quantified inside, within this constraint. Indexed by variable number.
    
    final supports_current_backtrack supports_current_b;
    final literals_by_support_backtrack literals_by_support_b;
    final last_support_backtrack last_support_b;
    
    final ArrayList<pa> [][] pa_store;  // pa_store[var][val] is the arraylist of pa's for var and val, generated at the root node.
    final int numpa;  // the number of pa's in the system.
    
    final mid_domain [] variables;
    
    // this is for timing seekNextSupport calls, should really have a getter but meanwhile it's public.
    public stopwatch2 sw;
    public long snscount=0;
    
    final boolean timeseeknextsupport=true;
    
    //int maxdomsize;
    
    gac_schema(mid_domain[] variables, qcsp prob)
    {
        super(prob);
        
        this.variables=variables;
        check_variables(variables);
        
        num_uni_inside = new int[variables.length];
        if(timeseeknextsupport) sw=new stopwatch2();
        
        int num_universals=0;
        
        //maxdomsize=0;
        
        for(int i=0; i<variables.length; i++)
        {
            if(variables[i].quant())
            {
                // add one to all the outside ones
                for(int j=0; j<i; j++)
                    num_uni_inside[j]++;
            }
            
            //if(variables[i].hash.length>maxdomsize)
                //maxdomsize=variables[i].hash.length;
            
            if(variables[i].quant())
                num_universals++;
        }
        
        //supports_current=new THashMap(10000); //<pa, linktuple>
        supports_current=new ArrayList<linktuple>();   // pa.scindex -> linktuple
        
        literals_by_support=new THashMap(1000); //<tuple, LinkedList<pa>>  // 10000
        //last_support=new THashMap(10000); //<pa, tuple>
        last_support=new ArrayList<tuple>();
        
        // and the backtrack objects
        supports_current_b=new supports_current_backtrack(this);
        literals_by_support_b=new literals_by_support_backtrack(this);
        last_support_b=new last_support_backtrack(this);
        
        // pa store
        pa_store=new ArrayList[variables.length][];
        for(int varindex=0; varindex<variables.length; varindex++)
        {
            pa_store[varindex]=new ArrayList[variables[varindex].domsize()];
            for(int val=variables[varindex].lowerbound(); val<=variables[varindex].upperbound(); val++)
            {
                if(variables[varindex].is_present(val))
                {
                    pa_store[varindex][val+variables[varindex].offset()]=new ArrayList<pa>();
                }
            }
        }
        
        // now an empty pa_store has been built. Populate it.
        pamasterlist=new ArrayList<pa>();
        
        for(int var=0; var<variables.length; var++)
        {
            //System.out.println("Processing variable "+variables[var].name);
            //System.out.println("num_uni_inside:"+num_uni_inside[var]);
            
            int [] pavars= new int[num_uni_inside[var]+1];
            int [] pavals= new int[num_uni_inside[var]+1];
            
            for(int val=variables[var].lowerbound(); val<=variables[var].upperbound(); val++)
            {
                if(variables[var].is_present(val))
                {
                    pavars[0]=var;
                    pavals[0]=val;
                    
                    make_pa_list(pamasterlist, var+1, 1, pavars, pavals);
                }
            }
        }
        numpa=pamasterlist.size();
    }
    ArrayList<pa> pamasterlist; // this is a temporary storage to hold all the pa's between init and establish().
    
    public void add_backtrack_level()
    {
        // Intended to be called by the search algorithm whenever we go to a new node.
        supports_current_b.add_backtrack_level();
        literals_by_support_b.add_backtrack_level();
        last_support_b.add_backtrack_level();
        // and for the other data structures.
    }
    
    public void backtrack()
    {
        // Intended to be called when the search algorithm backtracks.
        supports_current_b.backtrack();
        literals_by_support_b.backtrack();
        last_support_b.backtrack();
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    public boolean establish()
    {
        // set up the current supports, and literals by support
        // Whatever happens here and in find_support is not backtrackable
        
        for(int i=0; i<variables.length; i++)
        {
            variables[i].add_wakeup(this);
        }
        
    partialloop:
        for(pa partial : pamasterlist)
        {
            if(supports_current.get(partial.scindex)!=null)
            {
                continue;  // go to next partial.
            }
            else
            {
                // Seek the first support for this value.
                //System.out.println("Seeking first support");
                // check validity of partial
                for(int i=0; i<partial.vars.length; i++) // start at the start this time.
                {
                    int vari=partial.vars[i];
                    int vali=partial.vals[i];
                    if(!variables[vari].is_present(vali))
                    {
                        continue partialloop;
                    }
                }
                // partial valid here.
                //System.out.println("Seeking support for "+partial);
                if(timeseeknextsupport) {sw.start(); snscount++;}
                tuple support=seekNextSupport(partial, null);
                if(timeseeknextsupport) sw.end();
                
                // record this in lastc / last_support
                //if(!(this instanceof gac_schema_positive)) 
                    last_support.set(partial.scindex, support);
                
                if(support==null)
                {   if(variables[partial.vars[0]].is_present(partial.vals[0]))
                    {
                        if(!variables[partial.vars[0]].exclude(partial.vals[0], null))
                        {
                            //  null because removals must be queued for this constraint.
                            return false;
                        }
                    }
                }
                else
                {
                    // add support to all supported partial assignments
                    
                    pa [] palist=find_supported_pa(support);
                    
                    for(int i=0; i<palist.length; i++)
                    {
                        if(supports_current.get(palist[i].scindex)!=null)
                        {
                            supports_current.get(palist[i].scindex).insert_end(support);
                        }
                        else
                        {
                            //System.out.println("Making new linktuple with id:"+palist[i].vars[0]);
                            linktuple temptuplelist=new linktuple(palist[i].vars[0]);   //careful here if adapting for SQGAC
                            temptuplelist.insert_end(support);
                            supports_current.set(palist[i].scindex, temptuplelist);
                            
                            // Since support is the first support for palist[i], add support -> temppa to literals_by_support
                            
                            if(literals_by_support.containsKey(support))
                            {
                                ((linkpa)literals_by_support.get(support)).insert_end(palist[i]);
                            }
                            else
                            {
                                linkpa temppalist= new linkpa();
                                temppalist.insert_end(palist[i]);
                                literals_by_support.put(support, temppalist);
                            }
                        }
                    }
                }
            }
        }
        pamasterlist=null;
        // if removals have been made during this initial phase, the
        // constraint queue should now have the appropriate calls to
        // propagate queued up.
        
        return true;
    }
    
    private void make_pa_list(ArrayList<pa> palist, int currentvar, int papos, int [] pavars, int [] pavals)
    {   // simultaneously builds pa_store and a master list, palist containing all pa's.
        if(currentvar==variables.length)
        {
            // pa is complete
            
            // make a copy so that tau will not be changed in later recursions
            int [] pavarscopy= (int [])pavars.clone();  // System.arraycopy may be faster.
            int [] pavalscopy= (int [])pavals.clone();
            
            pa tau=new pa(pavarscopy, pavalscopy);  // the only allowed new pa(..) in the program.
            
            // add to the permanent store
            
            pa_store[pavars[0]][pavals[0]+variables[pavars[0]].offset()].add(tau);
            tau.scindex=supports_current.size();
            supports_current.add(null);   // add a slot in the appropriate place.
            last_support.add(null);  // add a slot
            
            palist.add(tau);
        }
        
        // recursive case
        
        else if(variables[currentvar].quant())
        {
            // currentvar is universal so we need to branch.
            
            //System.out.println(pavars.length+" "+currentvar+" "+papos);
            pavars[papos]=currentvar;
            
            //System.out.println("Number of values to branch on:"+ variables[currentvar].hash.length);
            
            for(int val=variables[currentvar].lowerbound(); val<=variables[currentvar].upperbound();  val++)
            {
                if(variables[currentvar].is_present(val))
                {
                    // branch for val
                    //System.out.println("Branching for value "+val);
                    pavals[papos]=val;
                    make_pa_list(palist, currentvar+1, papos+1, pavars, pavals);
                }
            }
        }
        else
        {
            // currentvar is existential, so just increment currentvar
            make_pa_list(palist, currentvar+1, papos, pavars, pavals);
        }
    }
    
    /*private boolean find_support(int currentvar, int papos, int [] pavars, int [] pavals)
    {
        // currentvar is 0,1,2... for indexing the arrays
        
        // This function can only be used the first time, when an
        // assignment has no linked list of supports.
        
        // base case 
        
        if(currentvar==variables.length)
        {
            // pa is complete
            
            // make a copy so that tau will not be changed in later recursions
            int [] pavarscopy= (int [])pavars.clone();  // System.arraycopy may be faster.
            int [] pavalscopy= (int [])pavals.clone();
            
            pa tau=new pa(pavarscopy, pavalscopy);
            
            // add to the permanent store
            
            pa_store[pavars[0]][pavals[0]].add(tau);
            tau.scindex=supports_current.size();
            supports_current.add(null);   // add a slot in the appropriate place.
            
            // Check it is not already supported
            
            //System.out.println("Checking partial assignment:");
            //System.out.println(tau);
            
            if(supports_current.get(tau.scindex)!=null)
            {
                //System.out.println("Already supported");
                return true;
            }
            else
            {
                // Seek the first support for this value.
                //System.out.println("Seeking first support");
                tuple support=seekNextSupport(tau, null);
                
                // check it
                //if(this instanceof gac_schema_predicate)
                //{
                //    System.out.println("Checking support:"+support+":"+((gac_schema_predicate) this).pred.predicate(support));
                //}
                
                // record this in lastc / last_support
                if(!(this instanceof gac_schema_positive)) last_support.put(tau, support);
                
                if(support==null)
                {
                    return false;
                }
                else
                {
                    // add support to all supported partial assignments
                    
                    pa [] palist=find_supported_pa(support);
                    
                    for(int i=0; i<palist.length; i++)
                    {
                        if(supports_current.get(palist[i].scindex)!=null)
                        {
                            supports_current.get(palist[i].scindex).insert_end(support);
                        }
                        else
                        {
                            //System.out.println("Making new linktuple with id:"+palist[i].vars[0]);
                            linktuple temptuplelist=new linktuple(palist[i].vars[0]);   //careful here if adapting for SQGAC
                            temptuplelist.insert_end(support);
                            supports_current.set(palist[i].scindex, temptuplelist);
                            
                            // Since support is the first support for temppa, add support -> temppa to literals_by_support
                            
                            if(literals_by_support.containsKey(support))
                            {
                                ((linkpa)literals_by_support.get(support)).insert_end(palist[i]);
                            }
                            else
                            {
                                linkpa temppalist= new linkpa();
                                temppalist.insert_end(palist[i]);
                                literals_by_support.put(support, temppalist);
                            }
                        }
                    }
                    
                    return true;
                }
            }
        }
        
        // recursive case
        
        if(variables[currentvar].quant())
        {
            // currentvar is universal so we need to branch.
            
            //System.out.println(pavars.length+" "+currentvar+" "+papos);
            pavars[papos]=currentvar;
            
            //System.out.println("Number of values to branch on:"+ variables[currentvar].hash.length);
            
            for(int val=variables[currentvar].lowerbound(); val<=variables[currentvar].upperbound();  val++)
            {
                if(variables[currentvar].is_present(val))
                {
                    // branch for val
                    //System.out.println("Branching for value "+val);
                    pavals[papos]=val;
                    if( !find_support(currentvar+1, papos+1, pavars, pavals))
                        return false;
                }
            }
            
            return true;
        }
        else
        {
            // currentvar is existential, so just increment currentvar
            return find_support(currentvar+1, papos, pavars, pavals);
        }
    }*/
    
    // fail or succeed at doing some propagation.
    
    public boolean propagate(int problem_var, int val)
    {
        // problem_var is an index into the problem variable array.
        int var=variables[0].id();
        boolean found=false;
        for(int i=0; i<variables.length; i++)
        {
            if (problem_var==variables[i].id())
            {
                var=i;
                found=true;
                break;
            }
        }
        assert found;
        assert !variables[var].is_present(val); // at least if this is true, it can do no harm.
        // var, val is the pair removed.
        
        //System.out.println("Propagate called for "+problem.variables.get(problem_var)+", "+val);
        
        // The old code for constructing all pa's including var,val.
        //int[] tempvars=new int[num_uni_inside[var]+1];
        //int[] tempvals=new int[num_uni_inside[var]+1];
        //tempvars[0]=var;
        //tempvals[0]=val;
        //ArrayList<pa> list=new ArrayList<pa>();
        //generate_pa(list, var+1, 1, tempvars, tempvals);*/
        
        ArrayList<pa> list=pa_store[var][val+variables[var].offset()];
        
        /*if(variables[var].quant())
        {
            // then it could be in pa's starting with any outer var as well
            for(int var2=0; var2<var; var2++)
            {
                for(int val=variables[var2].lowerbound(); val<=variables[var2].upperbound(); val++)
                {
                    
                }
            }
            
            for(ArrayList<pa> list2
        }*/
        
        // for each pa which contains (var, val) as its first entry
        
    supportsloop:
        for(pa supports_current_key : list)
        {
            /*for(int i=1; i<supports_current_key.vars.length; i++) // start with the first inner universal.
            {
                int vari=supports_current_key.vars[i];
                int vali=supports_current_key.vals[i];
                if(!variables[vari].is_present(vali))
                {
                    continue supportsloop; // if the pa, supported by an invalidated tuple, is no longer valid, go to the next one.
                }
            }*/
            
            // Make a copy because it could change.
            
            ArrayList<tuple> sc = new ArrayList<tuple>();
            //System.out.println(supports_current_key);
            
            {linktuple temp = supports_current.get(supports_current_key.scindex);
            if(temp==null) continue supportsloop; // if the pa has no entry in supports_current, go to the next one.
            tuple tau=temp.first();
            while(tau!=null)
            {
                sc.add(tau);
                tau=temp.next(tau);
            }}
            
            // for each tuple containing (var, val) which is currently used as a 
            // first support somewhere (since it is in supports_current)
            
            for(tuple tau : sc)
            {
                // tau is no longer valid.
                assert !valid(tau): "partial:"+supports_current_key+" valid tuple:"+tau;
                // for each (z,c) in tau remove tau from supports_current(z,c,...)
                
                pa [] palist=find_supported_pa(tau);
                
                //System.out.println("Retiring tuple tau:"+tau);
                
                for(pa partial : palist)
                {
                    assert supports_current.get(partial.scindex)!=null;
                    
                    //System.out.println("supported pa:"+partial);
                    //System.out.println("tau:"+tau);
                    linktuple lt=supports_current.get(partial.scindex);
                    
                    if(lt.in(tau))
                    {
                        lt.remove(tau);
                        supports_current_b.add_removal(partial, tau);
                    }
                }
                
                // Look up the pa's which had tau as their first support.
                // Copy to avoid any interference problems.
                ArrayList<pa> taufirstsupport = new ArrayList<pa>();
                
                {linkpa partiallist=(linkpa)literals_by_support.get(tau);
                pa partial=partiallist.pop();
                while(partial!=null)
                {
                    taufirstsupport.add(partial);
                    literals_by_support_b.add_removal(tau, partial);
                    partial=partiallist.pop();
                }}
                
            partialloop:
                for(pa partial : taufirstsupport)
                {
                    // Check this partial is still valid against current domains.
                    for(int i=0; i<partial.vars.length; i++)
                    {
                        if(!variables[partial.vars[i]].is_present(partial.vals[i]))  // if pruned
                        {
                            continue partialloop; // skip to next partial.
                        }
                    }
                    
                    // if partial is valid then seek support for it.
                    tuple sigma=seekInferableSupport(partial);
                    if(sigma!=null)
                    {
                        // add partial to literals_by_support(sigma)
                        // because sigma has become the primary support for partial.
                        //System.out.println("Adding support by seekInferableSupport");
                        ((linkpa)literals_by_support.get(sigma)).insert_end(partial);
                        literals_by_support_b.add_addition(sigma, partial);
                        assert satisfies(sigma, partial);
                        assert valid(sigma);
                    }
                    else
                    {
                        //if(!(this instanceof gac_schema_positive)) 
                        if(timeseeknextsupport) {sw.start(); snscount++;}
                            sigma=seekNextSupport(partial, last_support.get(partial.scindex));
                        if(timeseeknextsupport) sw.end();
                        //else sigma=seekNextSupport(partial, null); // cheaper call for gac_schema_positive
                        
                        if(sigma!=null)
                        {
                            // add partial in literals_by_support.
                            if(literals_by_support.containsKey(sigma))
                            {
                                ((linkpa)literals_by_support.get(sigma)).insert_end(partial);
                            }
                            else
                            {
                                linkpa temppalist=new linkpa();
                                temppalist.insert_end(partial);
                                literals_by_support.put(sigma, temppalist);
                            }
                            // store backtracking info
                            literals_by_support_b.add_addition(sigma, partial);
                            assert satisfies(sigma, partial);
                            
                            // update last_support
                            // first record backtracking info
                            
                            //if(!(this instanceof gac_schema_positive))
                            //{
                                last_support_b.add_action(partial, last_support.get(partial.scindex));
                                last_support.set(partial.scindex, sigma);
                            //}
                            
                            // for all pa supported by sigma, add sigma in supports_current(pa)
                            palist=find_supported_pa(sigma);
                            for(int i=0; i<palist.length; i++)
                            {
                                assert satisfies(sigma, palist[i]);
                                supports_current.get(palist[i].scindex).insert_end(sigma);
                                supports_current_b.add_addition(palist[i], sigma);
                            }
                        }
                        else
                        {
                            //System.out.println("Attempting to prune "+variables[partial.vars[0]]+" value "+partial.vals[0]);
                            
                            if(!variables[partial.vars[0]].exclude(partial.vals[0], null))
                            {
                                return false;  //fail
                            }
                        }
                    }
                }
            }
        }
        
        // check if there are any tuples containing var,val still in the system
        /*for(linktuple tl : supports_current)
        {
            if(tl!=null)
            {
                tuple current=tl.first();
                while(current!=null)
                {
                    if(current.vals[var]==val)
                    {
                        System.out.println("Called for "+variables[var]+" and val "+val);
                        System.out.println(this);
                        System.out.println("Found invalid tuple "+current);
                        
                        for(int i=0; i<variables.length; i++)
                        {
                            System.out.println("variable "+variables[i]+" vals "+variables[i].domain());
                            
                        }
                        
                        assert false;
                        //return true;
                    }
                    current=current.next[tl.id];
                }
            }
        }*/
        return true;
    }
    
    // remove this debugging function
    boolean satisfies(tuple sigma, pa partial)
    {
        for(int i=0; i<partial.vars.length; i++)
        {
            if(sigma.vals[partial.vars[i]]!=partial.vals[i])
                return false;
        }
        return true;
    }
    
    abstract tuple seekNextSupport(pa partial, tuple currentsupport);
    
    private tuple seekInferableSupport(pa partial)
    {
        linktuple i = (linktuple)supports_current.get(partial.scindex);
        tuple sup=i.first();
        
        while(sup!=null)
        {
            // test if sup is a support given current domains.
            
            boolean supporting=true;
            for(int j=0; j<sup.vals.length; j++)
            {
                if(!variables[j].is_present(sup.vals[j]))  // if pruned
                {
                    supporting=false;
                    break;
                }
            }
            
            if(supporting)
            {
                return sup;
            }
            else
            {
                i.remove(sup);
                supports_current_b.add_removal(partial, sup);
            }
            
            // find next value
            sup=i.next(sup);
            assert sup==null || i.in(sup);
            assert sup==i.first();
        }
        
        return null;
    }
    
    // this version should not be as quick as the one below, worst case.
    private pa [] find_supported_pa(tuple sigma)
    {
        // find all the pa's supported by sigma.
        // Iterate from the innermost to the outermost literal, retaining universals as you go.
        
        pa[] palist = new pa[variables.length];
        
        for(int i=0; i<variables.length; i++)
        {
            int [] tempvars = new int[num_uni_inside[i]+1];
            int [] tempvals = new int[num_uni_inside[i]+1];
            
            tempvars[0]=i;
            tempvals[0]=sigma.vals[i];
            
            int papointer=1;
            
            for(int j=i+1; j<variables.length; j++)
            {
                if(variables[j].quant())
                {
                    tempvars[papointer]=j;
                    tempvals[papointer]=sigma.vals[j];
                    papointer++;
                }
            }
            
            assert papointer==tempvars.length;
            
            //palist[i]= new pa(tempvars, tempvals);
            // instead of making a new pa, find it in pa_store
            // n log(something) binary search -- is there a better way?? Store PA's in a tree / trie for O(n) lookup.
            ArrayList<pa> tosearch=pa_store[tempvars[0]][tempvals[0]+variables[tempvars[0]].offset()];
            int top=tosearch.size()-1;
            int bottom=0;
            int index=top>>1;
            while(true)
            {
                pa currentpa=tosearch.get(index);
                
                boolean match=true;
                boolean lessthan=false;  // currentpa < tempvals
                // by definition the variables in the pa's are the same.
                for(int j=0; j<tempvals.length; j++)
                {
                    if(currentpa.vals[j]<tempvals[j])
                    {
                        match=false;
                        lessthan=true;
                        break;
                    }
                    else if(currentpa.vals[j]>tempvals[j])
                    {
                        match=false;
                        lessthan=false;
                        break;
                    }
                }
                if(match)
                {
                    palist[i]=currentpa;
                    break;
                }
                if(lessthan)
                {   // i.e. the currentpa is less than the one we are searching for
                    // increase the index.
                    bottom=index+1;
                }
                else
                {   // i.e. the currentpa is greater than the one we are searching for
                    // decrease the index.
                    top=index-1;
                }
                index=bottom+( (top-bottom)>>1 );
            }
        }
        return palist;
    }
    
    /*pa[] find_supported_pa(tuple sigma)
    {
        // find all the pa's supported by sigma.
        // Iterate from the innermost to the outermost literal, retaining universals as you go.
        
        pa[] palist = new pa[variables.length];
        
        int retained=sigma.vals.length;  // fill in the arrays from the end.
        
        int [] retained_vars = new int[variables.length];
        int [] retained_vals = new int[variables.length];
        
        for(int i=variables.length-1; i>=0; i--)
        {
            int [] tempvars = new int[variables.length-retained+1];
            int [] tempvals = new int[variables.length-retained+1];
            
            tempvars[0]=i;
            tempvals[0]=sigma.vals[i];
            
            // copy the retained universal assignments (if any)
            for(int j=retained; j<variables.length; j++)
            {
                tempvars[j-retained+1]=retained_vars[j];
                tempvals[j-retained+1]=retained_vals[j];
            }
            
            palist[i]= new pa(tempvars, tempvals);
            
            if(variables[i].quant())
            {
                // if the ith variable is universal then add it on the end of retained_array
                retained--;
                retained_vars[retained]=i;
                retained_vals[retained]=sigma.vals[i];
            }
        }
        return palist;
    }*/
    
    public String toString()
    {
        String st="gac_schema state:\n";
        
        /*st+="Keys in supports_current:\n";
        
        Set<Map.Entry<pa,linktuple>> mapset=supports_current.entrySet();
        
        for(Map.Entry<pa,linktuple> temp : mapset)
        {
            linktuple tempvalue=temp.getValue();
            pa temppa=temp.getKey();
            
            for(int j=0; j<temppa.vars.length; j++)
                st+=temppa.vars[j]+" = "+temppa.vals[j]+", ";
            
            st+=":";
            
            st+=tempvalue;
            
            st+="\n";
        }*/
        
        Set<Map.Entry<tuple, linkpa>> mapset2=literals_by_support.entrySet();
        st+="Keys in literals_by_support:\n";
        for(Map.Entry<tuple, linkpa> temp : mapset2)
        {
            linkpa tempvalue=temp.getValue();
            tuple tempkey=temp.getKey();
            
            st+=tempkey+":";
            
            pa a=tempvalue.first;
            while(a!=null)
            {
                st+=a+"; ";
                a=a.next;
            }
            
            st+="\n";
        }
        
        /*Set<Map.Entry<pa, tuple>> mapset3 = last_support.entrySet();
        st+="Keys in last_support\n";
        for(Map.Entry<pa, tuple> temp : mapset3)
        {
            tuple tempvalue=temp.getValue();
            pa tempkey=temp.getKey();
            
            st+=tempkey+":"+tempvalue+"\n";
        }*/
        
        return st;
    }
    
    abstract public boolean entailed();
    
    boolean valid(tuple tau)
    {
        if(tau==null)
            return false;   // surely a null tuple is not a valid tuple??
        return tuples.valid(tau.vals, variables);
    }
    
    
    // the following are required by the positive and nightingale instantiations.
    
    void generate_pa(ArrayList<pa> list, int currentvar, int index, int[] tempvars, int [] tempvals)
    {
        // base case
        if(currentvar==variables.length)
        {
            // Instead of making a new pa, look it up in the pa_store.
            
            ArrayList<pa> tosearch=pa_store[tempvars[0]][tempvals[0]+variables[tempvars[0]].offset()];
            int top=tosearch.size()-1;
            int bottom=0;
            int index2=top>>1;
            while(true)
            {
                pa currentpa=tosearch.get(index2);
                
                boolean match=true;
                boolean lessthan=false;  // currentpa < tempvals
                // by definition the variables in the pa's are the same.
                for(int j=0; j<tempvals.length; j++)
                {
                    if(currentpa.vals[j]<tempvals[j])
                    {
                        match=false;
                        lessthan=true;
                        break;
                    }
                    else if(currentpa.vals[j]>tempvals[j])
                    {
                        match=false;
                        lessthan=false;
                        break;
                    }
                }
                if(match)
                {
                    list.add(currentpa);
                    break;
                }
                if(lessthan)
                {   // i.e. the currentpa is less than the one we are searching for
                    // increase the index.
                    bottom=index2+1;
                }
                else
                {   // i.e. the currentpa is greater than the one we are searching for
                    // decrease the index.
                    top=index2-1;
                }
                index2=bottom+( (top-bottom)>>1 );
            }
            
            
        }
        // recursive case
        else
        {
            if(variables[currentvar].quant())
            {
                for(int i=variables[currentvar].lowerbound(); i<=variables[currentvar].upperbound(); i++)
                {
                    if(variables[currentvar].is_present(i))
                    {// if not pruned
                        tempvars[index]=currentvar;
                        tempvals[index]=i;
                        generate_pa(list, currentvar+1, index+1, tempvars, tempvals);
                    }
                }
            }
            else
            {
                generate_pa(list, currentvar+1, index, tempvars, tempvals);
            }
        }
    }
    
    void splittuples(int [][][] goods, int [][] alltuples) //<pa, goodlistwrapper>
    {
        for(int var=0; var<variables.length; var++)
        {
            // assumes tuples sorted in increasing lex order.
            //System.out.println("Variable "+var);
            
            for(int val=variables[var].lowerbound(); val<=variables[var].upperbound(); val++)
            {
                //System.out.println("Value "+val);
                
                int[] tempvars = new int[num_uni_inside[var]+1];
                int[] tempvals = new int[num_uni_inside[var]+1];
                tempvars[0]=var;
                tempvals[0]=val;
                ArrayList<pa> palist = new ArrayList<pa>();
                
                generate_pa(palist, var+1, 1, tempvars, tempvals); // returns tuples in lex order
                
                for(pa partial : palist)
                {
                    ArrayList<int []> localgoods = new ArrayList<int []>();
                    
                    //System.out.println("Partial: "+partial);
                    
                    // iterate through the table.
                    for(int tupleindex=0; tupleindex<alltuples.length; tupleindex++)
                    {
                        boolean matching=true;
                        for(int i=0; i<partial.vars.length; i++)
                        {
                            int [] tup=alltuples[tupleindex];
                            if(tup[partial.vars[i]]!=partial.vals[i])
                            {
                                matching=false;
                                break;
                            }
                        }
                        if(matching)
                        {
                            //System.out.println("Adding "+alltuples[tupleindex][0]+alltuples[tupleindex][1]+alltuples[tupleindex][2]+alltuples[tupleindex][3]);
                            localgoods.add(alltuples[tupleindex]);
                        }
                    }
                    
                    // store the tuple list in goods.
                    int [][] goodlist=localgoods.toArray(new int[0][]);
                    
                    goods[partial.scindex]=goodlist;
                }
            }
        }
    }
}

// ---------------------------------------------------------------------------------------------------
// instantiations

