// A constraint which finds the max of a vector of variables.

package queso.constraints;

import java.util.*;
import gnu.math.*;

import queso.core.*;

public class max_constraint extends constraint implements make_ac
{
    // implements the rule-based sum constraint with three rules per
    // variable : prune upper (lower) bound and 'sufficient inner variance'
    // Uses variables whose bounds are represented by integers
    mid_domain[] variables;
    
    public max_constraint(mid_domain [] variables, int maxindex, qcsp problem)
    {
        super(problem);
        this.variables=variables;
        check_variables(variables);
        this.maxindex=maxindex;
    }
    
    final int maxindex;   // the index of the variable that is the max of the others.
    
    public boolean establish()
    {
        for(mid_domain temp : variables)
        {
            temp.add_wakeup(this);
        }
        
        return make_ac();
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    public String toString()
    {
        String st="max(";
        for(int i=0; i<variables.length; i++)
        {
            if(i!=maxindex) st+=(variables[i].quant()?"A":"E")+variables[i]+" ";
        }
        st+=")=";
        st+=(variables[maxindex].quant()?"A":"E")+variables[maxindex];
        
        return st;
    }
    
    public boolean make_ac()
    {
        int newlowerbound=(maxindex==0)?variables[1].lowerbound():variables[0].lowerbound();  // new lower bound for maxindex variable
        int newupperbound=(maxindex==0)?variables[1].upperbound():variables[0].upperbound();  // new upper bound for maxindex variable
        
        // find the maximum lower bound and apply to maxindex
        // find the maximum upper bound and apply to maxindex
        
        for(int i=0; i<variables.length; i++)
        {
            if(i!=maxindex)
            {
                int thisupperbound=variables[i].upperbound();
                int thislowerbound=variables[i].lowerbound();
                
                if(thisupperbound>newupperbound)
                {
                    newupperbound=thisupperbound;
                }
                if(thislowerbound>newlowerbound)
                {
                    newlowerbound=thislowerbound;
                }
            }
        }
        
        if(newupperbound<variables[maxindex].upperbound())
        {
            if(!variables[maxindex].exclude_upper(newupperbound, this))
            {
                return false;
            }
        }
        
        if(newlowerbound>variables[maxindex].lowerbound())
        {
            if(!variables[maxindex].exclude_lower(newlowerbound, this))
            {
                return false;
            }
        }
        
        // now propagate back from maxindex to the other variables.
        
        if(newupperbound>variables[maxindex].upperbound())
        {
            for(int i=0; i<variables.length; i++)
            {
                if(i!=maxindex && variables[i].upperbound()>variables[maxindex].upperbound())
                {   // trim it
                    if(!variables[i].exclude_upper(variables[maxindex].upperbound(), this))
                    {
                        return false;
                    }
                }
            }
        }
        
        return true;
    }
    
    public boolean entailed()
    {
        // if all vars unit
        for(int i=0; i<variables.length; i++)
        {
            if(!variables[i].unit())
            {
                return false;
            }
        }
        return true;
    }
}

class max_constraint_test
{
    public static void main(String[] args)
    {
        mid_domain [] variables= new mid_domain[4];
        qcsp prob = new qcsp();
        
        variables[0]=new existential(-10, 10, prob, "a");
        variables[1]=new existential(-10, 10, prob, "b");
        variables[2]=new existential(0, 3, prob, "c");
        variables[3]=new existential(0, 3, prob, "d");
        
        int weights[]= {2,-2,3,3};
        
        sum_constraint c1= new sum_constraint(variables, weights, prob);
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
    }
}

class max_constraint_test3
{
    // checks if solving a QCSP involving a single max constraint
    // produces the same result as propagating the SQGAC version.
    public static void main(String[] args)
    {
        max_constraint_test3 o1=new max_constraint_test3();
        
        for(int i=0; i<100000; i++)
        {
            o1.test_random_prob();
            System.out.println("Done test "+i);
        }
    }
    
    void test_random_prob()
    {
        final int numvars=6;
        mid_domain [] vars_max= new mid_domain[numvars];
        mid_domain [] vars_sqgac= new mid_domain[numvars];
        
        qcsp prob_max = new qcsp();
        qcsp prob_sqgac = new qcsp();
        
        int [] coeff=new int[numvars];
        for(int i=0; i<numvars; i++)
        {
            coeff[i]=(int)(Math.random()*20)-10;
            while(coeff[i]==0)
            {
                coeff[i]=(int)(Math.random()*20)-10;
            }
        }
        
        // 5 types of universal to choose from
        // random test.
        
        for(int i=0; i<numvars; i++)
        {
            // existential
            double temp=Math.random();
            int ub, lb;
            if(temp<0.2)
            {   // positive
                ub=12; lb=3;
            }
            else if(temp<0.4)
            {   // includes 0.
                ub=9; lb=0;
            }
            else if(temp<0.6)
            {   // spans 0
                ub=6; lb=-3;
            }
            else if(temp<0.8)
            {   // includes 0 negative
                ub=0; lb=-9;
            }
            else
            {
                ub=-3; lb=-12;
            }
            vars_max[i]=new existential(lb, ub, prob_max, "x"+(i+1));
            vars_sqgac[i]=new existential(lb, ub, prob_sqgac, "x"+(i+1));
        }
        
        int xi = (int) (numvars*Math.random());
        assert xi>=0 && xi<=(numvars-1);
        
        max_constraint c1= new max_constraint(vars_max, xi, prob_max);
        
        predicate_wrapper pred=new max_predicate(xi);
        sqgac c2= new sqgac(vars_sqgac, prob_sqgac, pred);
        
        boolean flag1=prob_max.establish();
        boolean flag2=prob_sqgac.establish();
        
        // same strength
        System.out.println(c1+" after propagation");
        prob_max.printdomains();
        prob_sqgac.printdomains();
        
        assert flag1==flag2;
        assert (!flag1) || same_domains(prob_max, prob_sqgac);
        
        if(flag1) flag1=prob_max.search();
        if(flag1) flag2=prob_sqgac.search();
        assert flag1==flag2;
    }
    
    boolean same_domains(qcsp prob1, qcsp prob2)
    {
        if(prob1.variables.size()!=prob2.variables.size())
            return false;
        
        for(int i=0; i<prob1.variables.size(); i++)
        {
            mid_domain var1 = (mid_domain)prob1.variables.get(i);
            mid_domain var2 = (mid_domain)prob2.variables.get(i);
            
            if(var1.domsize()!=var2.domsize())
                return false;
            
            if(var1.lowerbound()!=var2.lowerbound())
                return false;
            
            if(var1.upperbound()!=var2.upperbound())
                return false;
            
            for(int j=var1.lowerbound(); j<=var1.upperbound(); j++)
            {
                if(var1.is_present(j)!=var2.is_present(j))
                    return false;
            }
        }
        return true;
    }
}

class max_predicate implements predicate_wrapper
{
    max_predicate(int maxindex)
    {
        this.maxindex=maxindex;
    }
    
    final int maxindex;
    
    public boolean predicate(tuple tau)
    {
        int max=(maxindex==0)?tau.vals[1]:tau.vals[0];
        for(int i=0; i<tau.vals.length; i++)
        {
            if(i!=maxindex && tau.vals[i]>max) max=tau.vals[i];
        }
        
        return tau.vals[maxindex]==max;
    }
}
