package queso.core;

import gnu.trove.*;
import java.util.*;
import gnu.math.*;
import java.math.*;

import queso.constraints.constraint;

// it looks like having two versions of every function! one for integers and one for IntNum's
// so that constraints which use either can coexist.

// may move branching into the variable, so that branching can happen on any type of 
// variable polymorphically. e.g. hash variables can branch value by value, and 
// bound variables can branch by domain splitting or whatever.

public abstract class variable
{
    int id;                 // ID number of this variable
    int blocknumber;        // block number of this variable 0.. #blocks-1
    final qcsp problem;     // The qcsp this variable is in.
    
    final boolean quant;    // true for universal
    final String name;      // The name of this variable
    
    boolean functional=false;
    
    /*variable(qcsp problem, boolean quant, String name, boolean dummy)  // adds an extra dummy arg which distinguishes from the other.
    {
        this.quant=quant;
        this.name=name;
        
        this.problem=problem;
    }*/
    
    variable(qcsp problem, boolean quant, String name)  // the normal constructor which registers with the problem.
    {
        this.quant=quant;
        this.name=name;
        
        this.problem=problem;
        
        problem.addvar(this);
    }
    
    public abstract boolean unit();
    public abstract boolean empty();
    
    // a variable should be able to describe itself.
    
    public int id()
    {
        return id;
    }
    
    public String toString()
    {
        return name;
    }
    
    public abstract String domain();
    
    public boolean quant()
    {
        return quant;
    }
    
    public int blocknumber()
    {
        return blocknumber;
    }
    
    public void set_functional()
    {
        functional=true;
    }
    
    public boolean is_functional()
    {
        return functional;
    }
}

/*interface integer_bounds
{
    // everything required by a propagator which revises bounds
    int upperbound();
    int lowerbound();
    
    int positive_smallest();
    int negative_smallest();
    
    boolean exclude_upper(int val, constraint cons);   // val is the new upperbound
    boolean exclude_lower(int val, constraint cons);   // val is the new lowerbound.
    
    // declares that from the old upper/lower bound to val is pure
    boolean pure_upper(int val, constraint cons);
    boolean pure_lower(int val, constraint cons);
    
    // declares that from the old upper/lower bound to val is trivial
    boolean trivial_upper(int val, constraint cons);
    boolean trivial_lower(int val, constraint cons);
    
    boolean is_present(int val);
    
    boolean containsZero();
    
    // perhaps these two can be deleted.
    int domsize();  // for making an array on the propagator side. A constant.
    int offset();   // when addressing an array, use [offset+val]
    
    boolean quant();
    int id();
    
    // replace with upper and lower bound versions.
    void add_wakeup(constraint cons);   // wake up cons on any domain change here.
    
    boolean empty();  // what is this used for?? only in schema.java once.
    boolean unit();
    
    String name();
    
    boolean instantiate(int val);  // it should always succeed
}*/





abstract class hash_var extends variable implements intnum_bounds, mid_domain
{
    int[] hash;             // 0 for present, 1 for checked, 2 for pruned. 
    
    int upperbound;
    int lowerbound;
    
    int offset;
    
    ArrayList<constraint> wakeups_anychange; 
    
    hash_var(int lowerbound, int upperbound, qcsp problem, boolean quant, String name)
    {
        super(problem, quant, name);
        
        assert upperbound-lowerbound>=0;  // allow for constants but not empty variables.
        
        int numvalues=upperbound-lowerbound+1;  // include 0.
        
        hash=new int[numvalues];   
        
        for(int i=0; i<numvalues; i++)
            hash[i]=0;
        
        //wakeups=new ArrayList[numvalues];
        
        /*for(int i=0; i<numvalues; i++)
        {
            wakeups[i]=new ArrayList<constraint>();
        }*/
        wakeups_anychange=new ArrayList<constraint>();
        
        this.lowerbound=lowerbound;
        this.upperbound=upperbound;
        offset=-lowerbound;   // hash[offset+val] is the hash entry for val.
    }
    
    public void add_wakeup(constraint cons)
    {
        //System.out.println("Adding wakeup "+this+":"+cons);
        wakeups_anychange.add(cons);
    }
    
    public int upperbound()
    {
        return upperbound;
    }
    
    public int lowerbound()
    {
        return lowerbound;
    }
    
    public int positive_smallest()
    {
        for(int i=1; i<=upperbound; i++)
            if(is_present(i)) return i;
        System.out.println("["+lowerbound+","+upperbound+"]");
        assert false;
        return -1;
    }
    
    public int negative_smallest()
    {
        for(int i=-1; i>=lowerbound; i--)
            if(is_present(i)) return i;
        assert false;
        return 1;
    }
    
    public boolean containsZero()
    {
        return is_present(0);
    }
    
    public boolean is_present(int val)
    {
        // not really a dumb getter method, but intended to provide some abstraction from
        // the hash array and bounds
        
        assert invariant();
        return val>=lowerbound && val<=upperbound && hash[offset+val]==0;
    }
    
    public boolean is_present(IntNum val)
    {
        int val2=val.intValue();
        return is_present(val2);
    }
    
    public boolean unit()
    {
        return lowerbound==upperbound;
    }
    
    public boolean empty()
    {
        return lowerbound>upperbound;
    }
    
    public int domsize()
    {
        return hash.length;
    }
    
    // low-level events, not seen by the consistency algorithms.
    // Backtracking happens at this level.
    
    boolean remove(int val)
    {
        // actually perform a removal
        hash[offset+val]=2;
        if(val==lowerbound)
        {
            lowerbound++;
            while(lowerbound<=upperbound && hash[offset+lowerbound]==2)  // seek new lower bound
            {
                lowerbound++;
            }
        }
        
        if(val==upperbound)
        {
            upperbound--;
            while(upperbound>=lowerbound && hash[offset+upperbound]==2)  // seek new upper bound
            {
                upperbound--;
            }
        }
        
        problem.backtrack.fd_var.add_backtrack_pair(this.id, val);
        
        assert invariant();
        
        if(lowerbound>upperbound)
            return false;
        else
            return true;
    }
    
    void remove_nofail(int val)
    {
        // if this is the last remaining value, don't remove it.
        // For subsumption semantics
        if(val==upperbound && val==lowerbound)
            return;
        
        // otherwise remove as normal.
        remove(val);
    }
    
    /*boolean remove_upper(int newupperbound)
    {
        assert newupperbound<upperbound;
        problem.backtrack.fd_var.add_backtrack_upperbound(this.id, upperbound);
        upperbound=newupperbound;
        if(lowerbound>upperbound)
            return false;
        else
            return true;
    }
    
    void remove_upper_nofail(int newupperbound)
    {
        if(newupperbound<lowerbound)
            newupperbound=lowerbound;
        // prune it down to one value but no further.
        remove_upper(newupperbound);
    }
    
    boolean remove_lower(int newlowerbound)
    {
        assert newlowerbound<lowerbound;
        problem.backtrack.fd_var.add_backtrack_lowerbound(this.id, lowerbound);
        lowerbound=newlowerbound;
        if(lowerbound>upperbound)
            return false;
        else
            return true;
    }
    
    void remove_lower_nofail(int newlowerbound)
    {
        if(newlowerbound>upperbound)
            newlowerbound=upperbound;
        // prune it down to one value but no further.
        remove_upper(newlowerbound);
    }*/
    
    // backtrack function for the above.
    void unremove(int val)
    {
        // undo a removal
        hash[offset+val]=0;
        if(val<lowerbound)
            lowerbound=val;
        
        if(val>upperbound)
            upperbound=val;
        
        assert invariant();
    }
    
    /*void unremove_upper(int val)
    {
        upperbound=val;
    }
    
    void unremove_lower(int val)
    {
        lowerbound=val;
    }*/
    
    boolean invariant()
    {
        // check that lowerbound and upperbound match the hash array
        
        for(int i=-offset; i<lowerbound; i++)
        {
            if(! (hash[offset+i]==2)) return false;
        }
        
        if(lowerbound<=upperbound)
        {
            if(!(hash[offset+lowerbound]==0)) return false;
            
            if(!(hash[offset+upperbound]==0)) return false;
        }
        
        for(int i=offset+upperbound+1; i<hash.length; i++)
        {
            if(!(hash[i]==2)) return false;
        }
        
        return true;
    }
    
    // intnum_bounds methods in terms of the integer_bounds methods.
    // This is so that these methods do not need to be included in the
    // subclasses.
    
    public boolean exclude_upper(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return exclude_upper(val.intValue(), cons);
    }
    
    public boolean exclude_lower(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return exclude_lower(val.intValue(), cons);
    }
    
    /*public boolean pure_upper(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return pure_upper(val.intValue(), cons);
    }
    
    public boolean pure_lower(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return pure_lower(val.intValue(), cons);
    }*/
    
    /*public boolean trivial_upper(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return trivial_upper(val.intValue(), cons);
    }
    
    public boolean trivial_lower(IntNum val, constraint cons)
    {
        IntNum twobil = new IntNum(2000000000);
        IntNum minustwobil = new IntNum(-2000000000);
        assert val.compare(twobil)<0;
        assert val.compare(minustwobil)>0;
        return trivial_lower(val.intValue(), cons);
    }*/
    
    public boolean instantiate(IntNum val)
    {
        return instantiate(val.intValue());
    }
    
    public IntNum upperbound_big()
    {
        return new IntNum(upperbound);
    }
    
    public IntNum lowerbound_big()
    {
        return new IntNum(lowerbound);
    }
    
    public IntNum positive_smallest_big()
    {
        return new IntNum(positive_smallest());
    }
    
    public IntNum negative_smallest_big()
    {
        return new IntNum(negative_smallest());
    }
    
    public String domain()
    {
        String st="";
        for(int i=-offset; i<(hash.length-offset); i++) if(hash[offset+i]!=2) st+=i+",";
        return st;
    }
    
    public int offset()
    {
        return offset;
    }
}


abstract class bounds_var extends variable implements intnum_bounds
{
    IntNum upperbound;
    IntNum lowerbound;
    
    IntNum initial_domain_size;
    
    ArrayList<constraint> lower_wakeups;    // wakeups=constraints to check when bound pruned. Static during search
    ArrayList<constraint> upper_wakeups;
    
    bounds_var(IntNum lowerbound, IntNum upperbound, qcsp problem, boolean quant, String name)
    {
        super(problem, quant, name);
        
        assert upperbound.compare(lowerbound)>=0;  // allow for constants but not empty.
        
        lower_wakeups=new ArrayList<constraint>();
        upper_wakeups=new ArrayList<constraint>();
        
        this.lowerbound=lowerbound;
        this.upperbound=upperbound;
        this.initial_domain_size=IntNum.add(IntNum.sub(upperbound, lowerbound), IntNum.one());
    }
    
    public void add_wakeup(constraint cons)
    {
        lower_wakeups.add(cons);
        upper_wakeups.add(cons);
    }
    
    public void add_lower_wakeup(constraint cons)
    {
        lower_wakeups.add(cons);
    }
    
    public void add_upper_wakeup(constraint cons)
    {
        upper_wakeups.add(cons);
    }
    
    public IntNum upperbound_big()
    {
        return upperbound;
    }
    
    public int upperbound()
    {
        return upperbound.intValue();
    }
    
    public IntNum lowerbound_big()
    {
        return lowerbound;
    }
    
    public int lowerbound()
    {
        return lowerbound.intValue();
    }
    
    public boolean is_present(IntNum val)
    {
        return val.compare(lowerbound)>=0 && val.compare(upperbound)<=0;
    }
    
    public boolean is_present(int val)
    {
        return lowerbound.compare(val)<=0 && upperbound.compare(val)>=0;
    }
    
    public boolean unit()
    {
        return lowerbound.compare(upperbound)==0;
    }
    
    public boolean empty()
    {
        return lowerbound.compare(upperbound)>0;
    }
    
    public IntNum domsize_big()
    {
        //for consistency with hash_var we should return the original domain size.
        return initial_domain_size;
    }
    
    public int domsize()
    {
        return initial_domain_size.intValue();
    }
    
    public boolean containsZero()
    {
        return is_present(IntNum.zero());
    }
    
    public IntNum negative_smallest_big()
    {
        assert lowerbound.compare(IntNum.zero())<0;
        return IntNum.minusOne();
    }
    
    public IntNum positive_smallest_big()
    {
        assert upperbound.compare(IntNum.zero())>0;
        return IntNum.one();
    }
    
    // low-level events, not seen by the consistency algorithms.
    // Backtracking happens at this level.
    
    boolean remove_upper(IntNum val)
    {
        // actually perform a removal
        
        assert val.compare(upperbound)<0;
        
        problem.backtrack.fd_var.add_backtrack_upperbound(this.id, upperbound);
        
        upperbound=val;
        
        if(lowerbound.compare(upperbound)>0)
            return false;
        else
            return true;
    }
    
    boolean remove_lower(IntNum val)
    {
        // actually perform a removal
        
        assert val.compare(lowerbound)>0;
        
        problem.backtrack.fd_var.add_backtrack_lowerbound(this.id, lowerbound);
        
        lowerbound=val;
        
        if(lowerbound.compare(upperbound)>0)
            return false;
        else
            return true;
    }
    
    // backtrack function for the above.
    void unremove_upper(IntNum val)
    {
        upperbound=val;
    }
    
    void unremove_lower(IntNum val)
    {
        lowerbound=val;
    }
    
    public String domain()
    {
        return "["+lowerbound+","+upperbound+"]";
    }
}

/*
class equals_wrapper extends variable implements mid_domain
{
    // For use in the general disjunction to encode e.g. a==5
    // and allow expressions like a=5 \/ not c=4 ...
    
    // to be used only on ordinary variables, not bounds
    
    // To have the same id and blocknumber as the wrapped variable.
    // Hence when a constraint looks through its array of 
    // variables, it finds the wrapper rather than
    // the actual variable, but the wrapper is
    // otherwise indistinguishable.
    
    // Only to be stored in the variable array of a constraint,
    // not registered with the problem object.
    equals_wrapper(mid_domain v1, int value)
    {
        super(((variable)v1).problem, v1.quant(), v1.name(), true);
        
        this.id=v1.id();
        this.blocknumber=((variable)v1).blocknumber;
        
        this.value=value;
        this.var=v1;
    }
    
    final int value;
    final mid_domain var;
    
    public final boolean exclude(int val, constraint cons)
    {
        assert val==0 || val==1;
        if(val==0)
        {
            for(int i=var.lowerbound(); i<=var.upperbound(); i++)
            {
                if(i!=value && var.is_present(i))
                {
                    if(!var.exclude(i, cons))
                        return false;
                }
            }
            return true;
        }
        else
        {
            // exclude val==1 so just exclude the relevant value
            return var.exclude(value, cons);
        }
    }
    
    public final boolean is_present(int val)
    {
        assert val==0 || val==1;
        if(val==1)
        {
            return var.is_present(value);
        }
        else
        {
            // return true if any other value is present
            return (var.upperbound()>value || var.lowerbound()<value);
        }
    }
    
    public final String domain()
    {
        String st="";
        if(var.is_present(value)) st+="1,";
        if(var.upperbound()>value || var.lowerbound()<value) st+="0";
        return st;  // shuold never be printed.
    }
    
    public final boolean quant()
    {return quant;}
    
    public final boolean pure(int val, constraint cons)
    {
        assert false;
        return true;
    }
    
    public final boolean trivial(int val, constraint cons)
    {
        assert false;
        return true;
    }
    
    public final int upperbound()
    {
        if(is_present(1))
        {
            return 1;
        }
        else
        {
            return 0;
        }
    }
    
    public final int lowerbound()
    {
        if(is_present(0))
        {
            return 0;
        }
        else
        {
            return 1;
        }
    }
    
    public final int domsize()
    {
        return 2;
    }
    
    public final int offset()
    {
        return 0;
    }
    
    public final int id()
    {return id;}
    
    public final void add_wakeup(constraint cons)
    {
        var.add_wakeup(cons);
    }
    
    public final boolean empty()
    {return var.empty();}
    
    public final boolean unit()
    {
        // unit 0 or unit 1
        // either value is not present, or it is and the bounds match
        return !var.is_present(value) ||
            var.upperbound()==var.lowerbound();
    }
    
    public final String name()
    {return name;}
    
    public final boolean instantiate(int val)
    {// this wrapper should not be visible to the search procedure
        assert false;
        return false;
    }
}*/

// rudimentary support for backtracking constraints
class backtrack_constraints extends backtrack  // why extend backtrack??
{
    ArrayList<stateful> conslist;
    
    backtrack_constraints()
    {
        conslist=new ArrayList<stateful>();
    }
    
    public final void add_backtrack_level()
    {
        for(stateful i : conslist)
        {
            i.add_backtrack_level();
        }
    }
    
    public final void backtrack()
    {
        for(stateful i : conslist)
        {
            i.backtrack();
        }
    }
    
    public void backtrack_item()
    {}
}

class fd_backtrack_variables extends backtrack
{
    TIntArrayList removals_vars;      // var-val pairs removed in sequence
    TIntArrayList removals_type;      // The type of change made, 0= val, 1= upper bound intnum, 2= lower bound intnum.
    TIntArrayList removals_vals;      // the value
    
    ArrayList<IntNum> removals_vals_intnum;   //or the old upper bound, or the old lower bound.
    
    qcsp prob;
    
    fd_backtrack_variables(qcsp problem)
    {
        prob=problem;
        removals_vars= new TIntArrayList();
        removals_vals= new TIntArrayList();
        removals_type= new TIntArrayList();
        removals_vals_intnum= new ArrayList<IntNum>();
    }
    
    final void add_backtrack_pair(int var, int val)
    {
        // for backtracking the fd variables
        removals_vars.add(var);
        removals_vals.add(val);
        removals_type.add(0);
        
        backtrack_increment();
    }
    
    final void add_backtrack_upperbound(int var, IntNum val)
    {
        // val is the old value, to be restored on backtracking
        removals_vars.add(var);
        removals_vals_intnum.add(val);
        removals_type.add(1);
        
        backtrack_increment();
    }
    
    final void add_backtrack_lowerbound(int var, IntNum val)
    {
        // val is the old value, to be restored on backtracking
        removals_vars.add(var);
        removals_vals_intnum.add(val);
        removals_type.add(2);
        
        backtrack_increment();
    }
    
    public void backtrack_item()
    {
        int type=removals_type.remove(removals_type.size()-1);
        int var=removals_vars.remove(removals_vars.size()-1);
        
        if(type==0)
        {
            int val=removals_vals.remove(removals_vals.size()-1);
            if(var>=1000000)
            {
                prob.puremonitors.get(var-1000000).unremove(val);
            }
            else
            {
                ((hash_var)prob.variables.get(var)).unremove(val);
            }
        }
        else if(type==1)
        {
            IntNum val=removals_vals_intnum.remove(removals_vals_intnum.size()-1);
            assert var<1000000;  // bug: just makes sure we don't have a puremonitor.
            ((bounds_var)prob.variables.get(var)).unremove_upper(val);  // change
        }
        else if(type==2)
        {
            IntNum val=removals_vals_intnum.remove(removals_vals_intnum.size()-1);
            assert var<1000000;  // bug: just makes sure we don't have a puremonitor.
            ((bounds_var)prob.variables.get(var)).unremove_lower(val);  // change
        }
    }
}

class super_backtrack
{
    fd_backtrack_variables fd_var;
    //rat_backtrack_variables rat_var;
    backtrack_constraints b_cons;
    
    super_backtrack(qcsp problem)
    {
        fd_var= new fd_backtrack_variables(problem);
        //rat_var= new rat_backtrack_variables(problem);
        b_cons=new backtrack_constraints();
    }
    
    void add_backtrack_level()
    {
        fd_var.add_backtrack_level();
        //rat_var.add_backtrack_level();
        b_cons.add_backtrack_level();
    }
    
    void backtrack()
    {
        fd_var.backtrack();
        //rat_var.backtrack();
        b_cons.backtrack();
    }
    
    // blatant hack
    void register_constraint(stateful c)
    {
        b_cons.conslist.add(c);
    }
}
