package savilerow;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2021 Patrick Spracklen, Marc Roig, Ewan Davidson, and Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/

import savilerow.ASTNode;

import java.util.ArrayList;

import savilerow.*;

import java.util.*;
import java.io.*;

public class SMT extends Sat
{
    private static final long CHECK_SAT_LENGTH = ("(check-sat)" + System.lineSeparator()).length();

    protected List<String> variables;
    protected Set<String> integer_aux_variables;

    public SMT(SymbolTable _global_symbols)
    {
        super(_global_symbols);
        try {
            trueVar=getNextVariableNumber();
            addClause(trueVar);
        }
        catch ( IOException e ) {
            CmdFlags.errorExit("Failed to write to SMT output file.");
        }
        integer_aux_variables = new HashSet<>();
    }

    /**
     * For submodel SMT
     */
    public SMT(SMT otherSMT){
        super((Sat)otherSMT);
        variables = otherSMT.variables;
        integer_aux_variables = otherSMT.integer_aux_variables;
    }

    protected void createHeader() throws IOException {
        addInfo("smt-lib-version", "2.6");
        addInfo("source", "| Auto-generated by Savile Row |");
        
        addLogic(CmdFlags.getLogic().toString());
        
        if (!CmdFlags.usingBoolector()) {
            addOption("random-seed", CmdFlags.smtseed);
        }
    }

    public void addInfo(String attribute, String value) throws IOException {
        outstream.write("(set-info :" + attribute + " " + value + ")");
        outstream.newLine();
    }

    public void addLogic(String logic) throws IOException {
        outstream.write("(set-logic " + logic + ")");
        outstream.newLine();
    }

    public void addOption(String name, String value) throws IOException {
        // Integer should be passed with toSMTInt since negative numbers are not accepted by some solvers

        outstream.write("(set-option :" + name + " " + value + ")");
        outstream.newLine();
    }

    public void addVariable(String var, String type) throws IOException {
        outstream.write("(declare-fun " + var + " () " + type + ")");
        outstream.newLine();

        if (variables == null)
            variables = new ArrayList<>();

        variables.add(var);
    }

    //Returns the next available SAT variable number.
    protected long getNextVariableNumber() {
        long var = super.getNextVariableNumber();

        try {
            addVariable("x" + String.valueOf(var), "Bool");
        } catch (IOException e) {
            CmdFlags.errorExit("Failed to open or write to SMT output file.");
        }

        return var;
    }
    
    @Override
    public void generateVariableEncoding1Val(String name, boolean satOutputMapping) throws IOException
    {

        if (global_symbols.isBoolean(name)) {
            super.generateVariableEncoding1Val(name, false);
        }

        if (global_symbols.isInteger(name)) {
            addVariable(name, "Int");

            ASTNode domain=global_symbols.getDomain(name);
            Intpair bnds=domain.getBounds();

            long val1=bnds.lower;
            addSMTClause("(= " + name + " " + toSMTInt(val1) + ")");
        }

        if (global_symbols.isBitVector(name)) {
            addVariable(name, "(_ BitVec " + BitVector.getBits() + ")");

            ASTNode domain=global_symbols.getDomain(name);
            Intpair bnds=domain.getBounds();

            long val1=bnds.lower;
            addSMTClause("(= " + name + " " + toSMTBV(val1) + ")");
        }

    }

    // noConstraints is true if this SR variable is mentioned in no constraints.
    @Override
    public void generateVariableEncoding2Vals(String name, boolean noConstraints, boolean satOutputMapping) throws IOException {

        //If variable isn't mentioned by constraints
        //It's assignment doesn't matter
        //Still need to encode, won't of been marked for encoding
        //by transform collect methods
        if (noConstraints) {
            if (CmdFlags.getUseBV()) { global_symbols.markAsBitVector(name); }
            else if (CmdFlags.getUseInt()) { global_symbols.markAsInteger(name); }
            else { global_symbols.markAsBoolean(name); }
        }

        if (global_symbols.isBoolean(name)) {
            super.generateVariableEncoding2Vals(name, noConstraints, false);
        }

        if (global_symbols.isInteger(name)) {
            addVariable(name, "Int");

            ASTNode domain=global_symbols.getDomain(name);
            Intpair bnds=domain.getBounds();

            long val1=bnds.lower;
            long val2=bnds.upper;

            if (global_symbols.isBoolean(name)) {
                // If the variable is encoded as a boolean, link the integer and boolean encodings by the order encoding
                long satVar1 = getOrderVariable(name, val1);

                addSMTClause("(ite" + toLiteral(satVar1) + " (= " + name + " " + toSMTInt(val1) + ") (= " + name + " " + toSMTInt(val2) + "))");
            }
            else
                addSMTClause("(or (= " + name + " " + toSMTInt(val1) + ") (= " + name + " " + toSMTInt(val2) + "))");
        }

        if (global_symbols.isBitVector(name)) {
            addVariable(name, "(_ BitVec " + BitVector.getBits() + ")");

            ASTNode domain=global_symbols.getDomain(name);
            Intpair bnds=domain.getBounds();

            long val1=bnds.lower;
            long val2=bnds.upper;

            if (global_symbols.isBoolean(name)) {
                // If the variable is encoded as a boolean, link the integer and boolean encodings by the order encoding
                long satVar1 = getOrderVariable(name, val1);

                addSMTClause("(ite" + toLiteral(satVar1) + " (= " + name + " " + toSMTBV(val1) + ") (= " + name + " " + toSMTBV(val2) + "))");
            }
            else
                addSMTClause("(or (= " + name + " " + toSMTBV(val1) + ") (= " + name + " " + toSMTBV(val2) + "))");
        }
    }

    // encodes a variable for any number of possible values
    @Override
    public void generateVariableEncodingNVals(String name, boolean satOutputMapping) throws IOException {

        if (global_symbols.isBoolean(name)) {
            super.generateVariableEncodingNVals(name, false);
        }

        if (global_symbols.isInteger(name)) {
            addVariable(name, "Int");

            ASTNode domain = global_symbols.getDomain(name);
            ArrayList<Intpair> intervalset = domain.getIntervalSet();

            if (false){//!CmdFlags.getUsePunchHoles() && !global_symbols.isAuxiliary(name)) {

                long lowest = intervalset.get(0).lower;
                long highest = intervalset.get(intervalset.size()-1).upper;

                for (int i = 0; i < intervalset.size(); i++) {
                    if (lowest > intervalset.get(i).lower) { lowest = intervalset.get(i).lower; }
                    if (highest < intervalset.get(i).upper) { highest = intervalset.get(i).upper; }
                }

                String clause = "(and (<= " + toSMTInt(lowest) + " " + name + ")" + " (<= " + name + " " + toSMTInt(highest) + "))";
                addSMTClause(clause);
            }
            else {
                // Define the domains
                if (intervalset.size() > 1) {
                    String clause = "(or";
                    for (Intpair interval : intervalset) {

                        if (interval.lower == interval.upper) {
                            clause += " (= " + toSMTInt(interval.lower) + " " + name + ")";
                        }
                        else {
                            clause += " (<= " + toSMTInt(interval.lower) + " " + name + " " + toSMTInt(interval.upper) + ")";
                        }
                    }
                    addSMTClause(clause + ")");
                }
                else {
                    String clause = "";
                    if (intervalset.get(0).lower == intervalset.get(0).upper) {
                        clause += "(= " + toSMTInt(intervalset.get(0).lower) + " " + name + ")";
                    }
                    else {
                        clause += "(<= " + toSMTInt(intervalset.get(0).lower) + " " + name + " " + toSMTInt(intervalset.get(0).upper) + ")";
                    }
                    addSMTClause(clause);
                }
            }

            if (global_symbols.isBoolean(name)) {
                // Link with order encoding
                for(int intervalidx=0; intervalidx<intervalset.size(); intervalidx++) {
                    Intpair bnds=intervalset.get(intervalidx);

                    for (long i=bnds.lower; i<=bnds.upper; i++)
                    {
                        long var = getOrderVariable(name, i);
                        if (var != getTrue())
                            addSMTClause("(ite" + toLiteral(var) + " (<= " + name + " " + toSMTInt(i) + ") (> " + name + " " + toSMTInt(i) + "))");
                    }
                }
            }
        }

        if (global_symbols.isBitVector(name)) {

            addVariable(name, "(_ BitVec " + BitVector.getBits() + ")");

            ASTNode domain = global_symbols.getDomain(name);
            ArrayList<Intpair> intervalset = domain.getIntervalSet();

            String clause = "";

            if (false){//!CmdFlags.getUsePunchHoles() && !global_symbols.isAuxiliary(name)) {

                long lowest = intervalset.get(0).lower;
                long highest = intervalset.get(intervalset.size()-1).upper;

                for (int i = 0; i < intervalset.size(); i++) {
                    if (lowest > intervalset.get(i).lower) { lowest = intervalset.get(i).lower; }
                    if (highest < intervalset.get(i).upper) { highest= intervalset.get(i).upper; }
                }

                clause = "(and (bvsle " + toSMTBV(lowest) + " " + name + ")" + " (bvsle " + name + " " + toSMTBV(highest) + "))";
            }
            else {

                // Define the domains

                int intervalSetSize = intervalset.size();
                if (intervalSetSize > 1) { clause = "(or"; }

                for (Intpair interval : intervalset) {
                    if (intervalSetSize > 1) { clause += " "; }
                    if (interval.lower == interval.upper) {
                        clause += "(= " + toSMTBV(interval.lower) + " " + name + ")";
                    }
                    else {
                        clause += "(and (bvsle " + toSMTBV(interval.lower) + " " + name + ")" + " (bvsle " + name + " " + toSMTBV(interval.upper) + "))";
                    }
                }
                if (intervalSetSize > 1) { clause += ")"; }
            }
            addSMTClause(clause);

            if (global_symbols.isBoolean(name)) {
                // Link with order encoding
                for(int intervalidx=0; intervalidx<intervalset.size(); intervalidx++) {
                    Intpair bnds=intervalset.get(intervalidx);

                    for (long i=bnds.lower; i<=bnds.upper; i++)
                    {
                        long var = getOrderVariable(name, i);
                        if (var != getTrue())
                            addSMTClause("(ite" + toLiteral(var) + " (bvsle " + name + " " + toSMTBV(i) + ") (bvsgt " + name + " " + toSMTBV(i) + "))");
                    }
                }
            }
        }
    }

    // Creates an auxiliary integer variable that will always have the value of mult * variable
    // Used for IDL
    public String createAuxSMTVariableFor(ASTNode var, long mult) throws IOException {

        if (!(var instanceof Identifier)) {
            CmdFlags.errorExit("Attempted to create an auxiliary SMT variable for " + var.toString() + " which is not an identifier");
        }

        if (!CmdFlags.getUseAuxSMT())
            CmdFlags.errorExit("Attempted to create an auxiliary SMT variable when they are not allowed");

        String variable = var.smtEncodeInt(this);

        if (mult == 1) {
            CmdFlags.warning("Trying to create an auxiliary variable for " + variable + " with multiplier 1");

            // Do not create a new variable because it would always have the same value as variable (which makes it useless)
            return variable;
        }

        String new_variable = "_" + mult + "_" + variable;

        boolean relation_case = var.isRelation() && global_symbols.isBoolean(var.toString());
        // vars that are relations (booleans) are special cases because they do not have an integer representation
        if (relation_case)
            new_variable = "_" + mult + "_" + var.toString();

        if (!integer_aux_variables.contains(new_variable)) {
            if (!variables.contains(variable) && !relation_case)
                CmdFlags.errorExit("The auxiliary variable " + new_variable + "could not be created because " + variable + " has not been defined");

            // Add the new variable
            addComment("Adding auxiliary variable " + new_variable + " which will have the value of " + mult + " * " + variable);
            addVariable(new_variable, "Int");

            // For all possible values of variable, new_variable should have the appropriate value (mult * variable)
            ASTNode domain=global_symbols.getDomain(var.toString());
            ArrayList<Intpair> intervalset=domain.getIntervalSet();

            for (Intpair bnds : intervalset) {
                for (long i = bnds.lower; i <= bnds.upper; i++) {
                    String var_val = toSMTInt(i);
                    String new_var_val = toSMTInt(i * mult);

                    addSMTClause("(ite (= " + variable + " " + var_val + ") (= " + new_variable + " " + new_var_val + ") (distinct " + new_variable + " " + new_var_val + "))");
                }
            }

            // Keeping track of the variable so that we do not create it multiple times
            integer_aux_variables.add(new_variable);
        }

        return new_variable;
    }

    protected void unitClauseStart() throws IOException {
        //removed or as probably unnecessary and breaks boolector
        outstream.write("(assert");
    }

    protected void clauseStart() throws IOException {
        //removed or as probably unnecessary and breaks boolector
        outstream.write("(assert (or");
    }

    protected void unitClauseEnd() throws IOException {
        outstream.write(")");
        outstream.newLine();
    }

    protected void clauseEnd() throws IOException {
        outstream.write("))");
        outstream.newLine();
    }

    /*
     * Replace empty clauses with assert False.
     */
    protected void addEmptyClause() throws IOException {

        outstream.write("(assert false)\n");
    }

    public void addClause(long lit1) throws IOException
    {
        if(lit1!=-trueVar) {
            unitClauseStart();
            writeLiteral(lit1);
            unitClauseEnd();
        }
        else { addEmptyClause(); }

        numClauses++;

        if(CmdFlags.getCNFLimit()!=0) {
            if(numClauses>CmdFlags.getCNFLimit()) {
                CmdFlags.println("ERROR: Reached CNF clause limit.");
                throw new IOException();
            }
        }
    }

    public void addClause(long lit1, long lit2) throws IOException
    {
        if (lit1==-trueVar) { addClause(lit2); }
        else if (lit2==-trueVar) { addClause(lit1); }
        else if (lit1==-trueVar && lit2==-trueVar) { addEmptyClause(); }
        else { super.addClause(lit1,lit2); }
    }

    public void addClause(long lit1, long lit2, long lit3) throws IOException
    {
        if (lit1==-trueVar && lit2==-trueVar) { addClause(lit3); }
        else if (lit1==-trueVar && lit3==-trueVar) { addClause(lit2); }
        else if (lit2==-trueVar && lit3==-trueVar) { addClause(lit1); }
        else if (lit1==-trueVar && lit2==-trueVar && lit3==-trueVar) { addEmptyClause(); }
        else { super.addClause(lit1,lit2,lit3); }
    }

    public void addClause(ArrayList<Long> literals) throws IOException
    {
        if (literals.size() == 0) { addEmptyClause(); }
        else if (literals.size() == 1) { addClause(literals.get(0)); }
        else { super.addClause(literals); }
    }

    public void addClauseReified(ArrayList<Long> literals, long auxVar) throws IOException
    {
        if (literals.size()==0) { addClause(-auxVar); }
        else super.addClauseReified(literals, auxVar);
    }

    public static String toLiteral(long lit) {
        if (lit > 0) {
            return " x" + String.valueOf(lit);
        }
        else {
            return " (not x" + String.valueOf(-lit) + ")";
        }
    }

    public static String toSMTInt(long i) {
        if (i < 0)
            return "(- " + Long.toString(-i) + ")";
        else
            return Long.toString(i);
    }

    public static String toSMTBV(long i) {

        return BitVector.toHexString(i);
    }

    protected void writeLiteral(long lit) throws IOException {
        outstream.write(toLiteral(lit));
    }

    public void addComment(String comment) throws IOException
    {
        outstream.write("; ");
        outstream.write(comment);
        outstream.newLine();
    }
    
    private long checkSatLocation=0L;
    
    public void finaliseOutput() throws IOException
    {
        outstream.flush();
        checkSatLocation=fw.getChannel().position();  //  Record the current file position just before (check-sat).
        outstream.write("(check-sat)");
        outstream.newLine();

        addComment("Adding commands to receive the value of the variables");
        // We do not actually need to sort the variables.
        // This is because they already are in the order in which they have been added (which is the correct one)
        // Also, sorting as Strings makes the order of variables with numbers incorrect

        for (String var : variables) {
            outstream.write("(get-value (" + var + "))");
            outstream.newLine();
        }

        // Add the command to get the statistics
        //Not supported by boolector
        if (!CmdFlags.usingBoolector()) {
            outstream.write("(get-info :all-statistics)");
        }

        outstream.newLine();

        outstream.flush();
        fw.getFD().sync();
        outstream.close();
    }

    public static boolean checkParenthesis(String clause) {
        if (clause.length() == 0) {
            CmdFlags.typeError("Trying to add an empty clause");
            return false;
        }
        else if (clause.charAt(0) != '(' || clause.charAt(clause.length() - 1) != ')') {
            CmdFlags.typeError("The following clause is not completely inside parenthesis: " + clause);
            return false;
        }
        else {
            int deep = 0;

            for (int i = 0; i < clause.length(); i++) {
                if (clause.charAt(i) == '(')
                    deep++;
                else if (clause.charAt(i) == ')')
                    deep--;

                if (deep <= 0 && i < clause.length() - 1) {
                    CmdFlags.typeError("The clause is closed before the end (multiple clauses): " + clause);
                    return false;
                }
            }

            if (deep > 0) {
                CmdFlags.typeError("Not all the opened parenthesis are closed for this clause: " + clause);
                return false;
            }

            return true;
        }
    }

    //Add objective function for optimisation problems
    public void addObjective(ASTNode objective) throws IOException {

        assert objective.getChild(0) instanceof Identifier;

        if (objective instanceof Minimising) {
            outstream.write("(minimize " + ((Identifier) objective.getChild(0)).getName() + ")\n");
        }
        else {
            outstream.write("(maximize " + ((Identifier) objective.getChild(0)).getName() + ")\n");
        }
    }

    public void addSMTClause(String clause) throws IOException {
        if (clause==null) { outstream.close(); System.exit(0);}
        assert checkParenthesis(clause);

        outstream.write("(assert ");
        outstream.write(clause);
        outstream.write(")");
        outstream.newLine();

        numClauses++;
        if(CmdFlags.getCNFLimit()!=0) {
            if(numClauses>CmdFlags.getCNFLimit()) {
                CmdFlags.println("ERROR: Reached CNF clause limit.");
                throw new IOException();
            }
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    //
    //   Methods that change the SAT instance after finalise.
    //   These just re-open the file, pass through the add-clause call then 
    //   finalise again.

    public void removeFinalClause() throws IOException {
        assert !CmdFlags.interactiveSolver;
        RandomAccessFile file = new RandomAccessFile(CmdFlags.smtfile, "rws");
        file.setLength(filesizeBackup);
        file.close();
        
        numClauses=numClausesBackup;
        
        fw=new FileOutputStream(CmdFlags.smtfile, true);   ///  true for append.
        outstream = new BufferedWriter(new OutputStreamWriter(fw));
        finaliseOutput();
    }

    public void removeSolvingInstructions() throws IOException {
        assert !CmdFlags.interactiveSolver;
        RandomAccessFile file = new RandomAccessFile(CmdFlags.smtfile, "rw");
        file.setLength(checkSatLocation); // This will remove the check-sat instruction and anything after that
        file.close();
    }
    
    public void addClauseAfterFinalise(ArrayList<Long> clause) throws IOException
    {
        assert !CmdFlags.interactiveSolver;
        removeSolvingInstructions();
        super.addClauseAfterFinalise(clause);
    }
    
    public void addClauseAfterFinalise(long lit1, boolean removable) throws IOException {
        assert !CmdFlags.interactiveSolver;
        removeSolvingInstructions();
        super.addClauseAfterFinalise(lit1, false);
    }
    
    public void addClauseAfterFinalise2(long c1, long c2, boolean removable) throws IOException {
        assert !CmdFlags.interactiveSolver;
        removeSolvingInstructions();
        super.addClauseAfterFinalise2(c1, c2, false);
    }
    
    public void addSMTClauseAfterFinalise(String clause, boolean removable) throws IOException
    {
        assert !CmdFlags.interactiveSolver;
        removeSolvingInstructions();
        fw=new FileOutputStream(CmdFlags.smtfile, true);   ///  true for append.
        numClausesBackup=numClauses;
        filesizeBackup=fw.getChannel().position();
        outstream = new BufferedWriter(new OutputStreamWriter(fw));
        addSMTClause(clause);
        finaliseOutput();
    }
    
    public void addSMTClauseAfterFinalise2(String c1, String c2, boolean removable) throws IOException
    {
        assert !CmdFlags.interactiveSolver;
        removeSolvingInstructions();
        fw=new FileOutputStream(CmdFlags.smtfile, true);   ///  true for append.
        numClausesBackup=numClauses;
        filesizeBackup=fw.getChannel().position();
        outstream = new BufferedWriter(new OutputStreamWriter(fw));
        addSMTClause(c1);
        addSMTClause(c2);
        finaliseOutput();
    }
    
    public long getNumVars() {
        return variables.size();
    }
}
