package savilerow.expression;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2024 Peter Nightingale and Luke Ryan
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/

import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;

import savilerow.CmdFlags;
import savilerow.model.SymbolTable;
import savilerow.treetransformer.TransformDecomposeDisjunctive;

// Unary renewable resource constraint.
// First argument is array of start time variables
// Second argument is durations of each task

public class Disjunctive extends ASTNodeC {
    public static final long serialVersionUID = 1L;
    public Disjunctive(ASTNode start, ASTNode dur) {
        super(start, dur);
    }
    
    public ASTNode copy() {
        return new Disjunctive(getChild(0), getChild(1));
    }
    
    public boolean isRelation() {
        return true;
    }
    
    public boolean typecheck(SymbolTable st) {
        for(int i=0; i<2; i++) {
            if(!getChild(i).typecheck(st)) {
                return false;
            }
        }
        if (getChild(0).getDimension() != 1) {
            CmdFlags.println("ERROR: Expected one-dimensional matrix in first argument of cumulative: " + this);
            return false;
        }
        if (getChild(1).getDimension() != 1) {
            CmdFlags.println("ERROR: Expected one-dimensional matrix in second argument of cumulative: " + this);
            return false;
        }
        
        return true;
    }
    
    public ASTNode simplify() {
        ASTNode starts = getChildConst(0);
        ASTNode durations = getChildConst(1);
        
        if (!(starts instanceof CompoundMatrix)) {
            if (starts instanceof EmptyMatrix) {
                return new BooleanConstant(true);
            }
            else {
                return null; // argument is not yet a matrix literal. 
            }
        }
        
        if(starts.getCategory() == ASTNode.Constant && durations.getCategory() == ASTNode.Constant) {
            // Replace with decomposition to be evaluated. 
            return TransformDecomposeDisjunctive.taskDecomp(starts, durations);
        }
        
        return null;
    }
    
    public ASTNode normalise() {
        // sort by hashcode
        if(!(getChild(0) instanceof CompoundMatrix && getChild(1) instanceof CompoundMatrix)) {
            return this;
        }
        
        ArrayList<ASTNode> ch=new ArrayList<>();
        for(int i=1; i<getChild(0).numChildren(); i++) {
            ch.add(new Container(getChild(0).getChild(i), getChild(1).getChild(i)));
        }
        
        boolean changed = sortByHashcode(ch);
        if (changed) {
            //  Unpack
            ArrayList<ASTNode> x=new ArrayList<>();
            ArrayList<ASTNode> dur=new ArrayList<>();
            
            for(int i=0; i<ch.size(); i++) {
                x.add(ch.get(i).getChild(0));
                dur.add(ch.get(i).getChild(1));
            }
            
            return new Disjunctive(CompoundMatrix.make(x), CompoundMatrix.make(dur));
        } else {
            return this;
        }
    }
    
    public ASTNode normaliseAlpha() {
        // sort alphabetically
        if(!(getChild(0) instanceof CompoundMatrix && getChild(1) instanceof CompoundMatrix)) {
            return null;
        }
        
        ArrayList<ASTNode> ch=new ArrayList<>();
        for(int i=1; i<getChild(0).numChildren(); i++) {
            ch.add(new Container(getChild(0).getChild(i), getChild(1).getChild(i)));
        }
        
        boolean changed = sortByAlpha(ch);
        if (changed) {
            //  Unpack
            ArrayList<ASTNode> x=new ArrayList<>();
            ArrayList<ASTNode> dur=new ArrayList<>();
            
            for(int i=0; i<ch.size(); i++) {
                x.add(ch.get(i).getChild(0));
                dur.add(ch.get(i).getChild(1));
            }
            
            return new Disjunctive(CompoundMatrix.make(x), CompoundMatrix.make(dur));
        } else {
            return null;
        }
    }
    
    @Override 
    public void toFlatzinc(BufferedWriter b, boolean bool_context) throws IOException {
        b.append("constraint fzn_disjunctive(");
        getChild(0).toFlatzinc(b, false);
        b.append(",");
        getChild(1).toFlatzinc(b, false);
        b.append(");");
    }

    @Override
    public void toMinizinc(StringBuilder b, boolean bool_context) {
        b.append("disjunctive(");
        getChild(0).toMinizinc(b, false);
        b.append(",");
        getChild(1).toMinizinc(b, false);
        b.append(")");
    }
    
    public String toString() {
        return generic_to_string("disjunctive");
    }
    
    
    
    ////////////////////////////////////////////////////////////////////////////
    //  JSON output for symmetry detection
    
    //  Todo
    /*
    public void toJSON(StringBuilder bf) {
        toJSONHeader(bf, true);
        // children
        bf.append("\"Children\": [");
        if(getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3) {
            //   Special case for binary != constraint.
            getChild(0).getChild(1).toJSON(bf);
            bf.append(", ");
            getChild(0).getChild(2).toJSON(bf);
        }
        else {
            // Same as toJSON method in ASTNode.
            for (int i = 0; i < numChildren(); i++) {
                bf.append("\n");
                getChild(i).toJSON(bf);
                // not last child
                if (i < numChildren() - 1) {
                    bf.append(",");
                }
            }
        }
        bf.append("]\n}");
    }
    
    public boolean childrenAreSymmetric() {
        return (getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3);
    }
    
    public boolean isChildSymmetric(int childIndex) {
        // If not a binary != ct, then the matrix inside should be regarded as symmetric.
        return !(getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3);
    }

    public boolean canChildBeConvertedToDifference(int childIndex) {
        return isMyOnlyOtherSiblingEqualZero(childIndex);
    }
    */
}
