package savilerow;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2020 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/






import java.util.*;

//  Turn MakeTable type into tableshort constraint.

public class TransformMakeTable extends TreeTransformerBottomUp
{
    public static boolean verbose=false;
    
    TabulationUtils tu;
    
    boolean prop;
    public TransformMakeTable(Model _m, boolean propagate) {
        super(_m);
        tu=new TabulationUtils(m);
        prop=propagate;
    }
    public TransformMakeTable(Model _m) {
        super(_m);
        tu=new TabulationUtils(m);
        prop=false;
    }
    
    protected NodeReplacement processNode(ASTNode curnode)
	{
	    boolean shorttable=(CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4);
	    if(curnode instanceof MakeTable) {
	        if(CmdFlags.make_short_tab==0) {
	            //  Option 0 given on command line. 
	            //  Just throw away the MakeTable function
	            return new NodeReplacement(curnode.getChild(0));
	        }
	        
	        //  Check the cache.
	        TabulationUtils.RetPair ret = tu.tryCacheNormalised(curnode.getChild(0), shorttable);
	        if(ret.nodereplace != null) {
	            return ret.nodereplace;
            }
            
            //  First normalise  before generating table.
            ASTNode a=tu.normalise(curnode.getChild(0));
            ASTNode newTable;
            
	        if(!shorttable) {
	            //  Make conventional table constraint
	            newTable = tu.makeTableLong(a, Long.MAX_VALUE, Long.MAX_VALUE);
	        }
	        else {
	            //  Make short table constraint.
                newTable = tu.makeTableShort(a, Long.MAX_VALUE, Long.MAX_VALUE, Long.MAX_VALUE);
            }
            tu.saveToCacheNormalised(ret.expstring, a, newTable);
            return new NodeReplacement(newTable);
        }
        else if(!prop && (CmdFlags.make_short_tab==3 || CmdFlags.make_short_tab==4) 
            && !(curnode instanceof And) && curnode.getParent()!=null && curnode.getParent().inTopAnd()
            && !(curnode instanceof Tag) && curnode.isRelation()
            && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
            && !(curnode instanceof BooleanConstant)) {
            //   Heuristics for applying the (short) table converter. 
            //   Only applied in final tailoring process.
            //   Top-level constraints only.  Boolean expressions nested within top-level constraints are not tabulated.  
            
            if(heuristic(curnode)) {
                if(verbose) {
                    System.out.println("Trying ct:"+curnode);
                }
                //  Check the cache.
                TabulationUtils.RetPair ret = tu.tryCacheNormalised(curnode, shorttable);
                if(ret.nodereplace != null) {
                    return ret.nodereplace;
                }
                if(tu.tryFailCache(ret.expstring)) {   /// Ideally this check should go above the pcache check.
                    return new NodeReplacement(new Tag(curnode));
                }
                
                ASTNode a = tu.normalise(curnode);
                ASTNode newTable;
                if(CmdFlags.make_short_tab==3) {
                    newTable=tu.makeTableLong(a, 10000, 100000);
                }
                else {
                    newTable=tu.makeTableShort(a, 10000, 100000, 100000);
                }
                
                if(newTable==null) {
                    tu.saveToFailCache(ret.expstring);
                    if(verbose) {
                        System.out.println("Adding to failCache:"+ret.expstring);
                    }
                    return new NodeReplacement(new Tag(curnode));
                }
                
                if(CmdFlags.tabulate_diagnostics) {
                    CmdFlags.println("Tabulated: "+curnode);
                }
                
                // Save in the cache
                tu.saveToCacheNormalised(ret.expstring, a, newTable);
                return new NodeReplacement(newTable);
            }
        }
        else if(!prop && (CmdFlags.make_short_tab==3 || CmdFlags.make_short_tab==4) && CmdFlags.tabulate_num
            && curnode.isNumerical()
            && !(curnode instanceof Mapping)
            && curnode.getDimension()==0
            && !(curnode instanceof NumberConstant)
            && curnode.toFlatten(false)) {
            //  Any non-trivial numerical expression.
            //  Make a fake constraint c: aux=curnode and apply heuristics to c 
            System.out.println("Trying numerical expression:"+curnode);
            ASTNode tmpaux=m.global_symbols.newAuxHelper(curnode);
            ASTNode ct=new Equals(curnode, tmpaux);
            
            // Special case of the strong prop heuristic -- check the constraint 
            // containing curnode.
            
            ASTNode outer_ct=curnode.getParent();
            while(outer_ct!=null && !outer_ct.isRelation()) {
                outer_ct=outer_ct.getParent();
            }
            
            if( (outer_ct.strongProp() && !ct.strongProp() )
                || heuristic(ct) ) {
                
                //  Copy-paste here, take it out...
                if(verbose) {
                    System.out.println("Trying ct:"+ct);
                    
                }
                //  Check the cache.
                TabulationUtils.RetPair ret = tu.tryCacheNormalised(ct, shorttable);
                if(ret.nodereplace != null) {
                    m.global_symbols.deleteSymbol(tmpaux.toString());
                    System.out.println("exit 1");
                    return ret.nodereplace;
                }
                if(tu.tryFailCache(ret.expstring)) {   /// Ideally this check should go above the pcache check.
                    //  Delete the aux variable
                    m.global_symbols.deleteSymbol(tmpaux.toString());
                    System.out.println("exit 2");
                    return new NodeReplacement(new Tag(curnode));
                }
                
                ASTNode a = tu.normalise(ct);
                ASTNode newTable;
                if(CmdFlags.make_short_tab==3) {
                    newTable=tu.makeTableLong(a, 10000, 100000);
                }
                else {
                    newTable=tu.makeTableShort(a, 10000, 100000, 100000);
                }
                
                if(newTable==null) {
                    tu.saveToFailCache(ret.expstring);
                    if(verbose) {
                        System.out.println("Adding to failCache:"+ret.expstring);
                    }
                    //  Delete the aux variable
                    m.global_symbols.deleteSymbol(tmpaux.toString());
                    System.out.println("exit 3");
                    return new NodeReplacement(new Tag(curnode));
                }
                
                if(CmdFlags.tabulate_diagnostics) {
                    CmdFlags.println("Tabulated: "+ct);
                }
                
                // Save in the cache
                //tu.saveToCacheNormalised(ret.expstring, a, newTable, tmpaux);    //  DOESN'T WORK -- GIVES BACK THE TABLE, NOT THE AUX VAR REFERENCE. 
                
                //  different from here. 
                //  Replace curnode with the new auxiliary variable, and add
                //  the new table constraint to the constraint set. 
                CmdFlags.warning("Extracting numerical expression: "+curnode+" before its containing constraint -- wrong order.");
                System.out.println("exit 4: ");
                return new NodeReplacement(tmpaux, null, newTable);
            }
            else {
                m.global_symbols.deleteSymbol(tmpaux.toString());
            }
        }
        return null;
    }
    
    private boolean heuristic(ASTNode curnode) {
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(curnode);
        ArrayList<ASTNode> varlistdups=TabulationUtils.getVariablesDup(curnode);
        
        //  Duplicate variables and within a reasonable size bound.
        //  Should perhaps look at domain size rather than # variables.
        if(varlist.size()<varlistdups.size() && varlist.size()<=10) {
            if(verbose) {
                System.out.println("H1");
            }
            return true;
        }
        
        //  Tree size is much larger than the number of variables.
        //  Either the expression is highly complex or it contains a large 
        //  number of constants. 
        if(curnode.treesize() > 5*varlist.size()) {
            if(verbose) {
                System.out.println("H2");
            }
            return true;
        }
        
        // Constraint does not get GAC, and it overlaps with some other
        // constraint that does get GAC or similar. Uses the strongProp method. 
        if(!curnode.strongProp() && varlist.size()<=10) {
            ASTNode constraints=m.constraints.getChild(0);
            if(constraints instanceof And) {
                for(int i=0; i<constraints.numChildren(); i++) {
                    ASTNode c=constraints.getChild(i);
                    if(c.strongProp()) {
                        for(int varidx=0; varidx<varlist.size(); varidx++) {
                            if(c.contains(varlist.get(varidx))) {
                                if(verbose) {
                                    System.out.println("H3");
                                }
                                return true;
                            }
                        }
                    }
                }
            }
        }
        
        return false;
    }
    
    
}
