package savilerow;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2020 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/

import java.util.* ;
import java.io.* ;

// Contains all tabulation methods
// Has a common instance of TabulationUtils shared by all tabulation methods. 

public class Tabulation 
{
    private static boolean verbose=false;
    
    private Model m;
    
    private TabulationUtils tu;
    
    public Tabulation(Model _m) {
        m=_m;
        tu=new TabulationUtils(m);
    }
    
    public void process(boolean prop) {
        //  First, MakeTable functions.
        processMakeTable(m.constraints);
        
        if(!prop && (CmdFlags.make_short_tab==3 || CmdFlags.make_short_tab==4)) {
            //  First find sets of constraints with the same scope, attempt
            //  tabulation of the conjunction of them.
            identicalScopes(m.constraints);
            
            //  The other three heuristics -- applied to each top-level constraint
            applyHeuristicsBool(m.constraints);
            
            //  Same applied to each numerical expression directly contained in a top-level constraint. 
            if(CmdFlags.tabulate_num) {
                applyHeuristicsNumerical(m.constraints);
            }
            
            //  For strong EDF prblem some new lmimit is needed to stop it 
            //  attempting to tabulate the entire problem.
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Deal with MakeTable functions
    //  Top-down descent of tree. 
    private void processMakeTable(ASTNode curnode) {
        if(curnode instanceof MakeTable) {
            if(CmdFlags.make_short_tab==0) {
	            //  Option 0 given on command line. 
	            //  Remove the MakeTable function
	            ASTNode parent=curnode.getParent();
	            int idx=curnode.getChildNo();
	            curnode.getChild(0).setParent(null);
	            parent.setChild(idx, curnode.getChild(0));
	            
	            //  Recursively process the replacement ASTNode. 
	            processMakeTable(parent.getChild(idx));
	        }
	        else {
	            //  Tabulate with no limits. Always returns a table constraint. 
	            ASTNode newTable=tabulate(curnode.getChild(0), Long.MAX_VALUE, Long.MAX_VALUE, Long.MAX_VALUE);
	            
                curnode.getParent().setChild(curnode.getChildNo(), newTable);
            }
        }
        else {
            //  Recursive descent of the tree
            for(int i=0; i<curnode.numChildren(); i++) {
                processMakeTable(curnode.getChild(i));
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Identical scopes heuristic
    
    private void identicalScopes(ASTNode top) {
        HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=new HashMap<>();
        
        // Populate scopeslist
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode ct=a.getChild(i);
                if(!(ct instanceof Table) && !(ct instanceof TableShort) && !(ct instanceof NegativeTable)
                    && !(ct instanceof Tag)) {
                    assert ct.isRelation();
                    
                    // Get the scope. 
                    ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(ct);
                    ASTNode.sortByAlpha(scope);  //  Sort alphabetically
                    
                    if(! scopeslist.containsKey(scope)) {
                        scopeslist.put(scope, new ArrayList<ASTNode>());
                    }
                    
                    scopeslist.get(scope).add(ct);
                }
            }
        }
        
        //  Find entries in scopeslist where number of constraints > 1
        boolean shorttable=(CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4);
        
        for(Map.Entry<ArrayList<ASTNode>,ArrayList<ASTNode>> p : scopeslist.entrySet()) {
            ArrayList<ASTNode> ctlist=p.getValue();
            if(ctlist.size()>1) {
                ASTNode totabulate=new And(ctlist);
                
                if(verbose) {
                    System.out.println("H4");
                    System.out.println("Trying ct:"+totabulate);
                }
                
                ASTNode newTable=tabulate(totabulate, 10000, 100000, 100000);
                
                if(newTable!=null) {
                    replaceConstraintSet(ctlist, newTable);
                }
            }
        }
    }
    
    private void replaceConstraintSet(ArrayList<ASTNode> ctlist, ASTNode replacement) {
        ASTNode a=ctlist.get(0);
        
        a.getParent().setChild(a.getChildNo(), replacement);
        
        //  Clear the rest of the constraints.
        for(int i=1; i<ctlist.size(); i++) {
            a=ctlist.get(i);
            a.getParent().setChild(a.getChildNo(), new BooleanConstant(true));
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Three single-constraint heuristics
    
    private void applyHeuristicsBool(ASTNode top) {
        //  Simply iterate through the top-level constraints. 
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode curnode=a.getChild(i);
                //  Checks are a bit paranoid, not all are necessary
                if(!(curnode instanceof And) && curnode.isRelation()
                    && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
                    && !(curnode instanceof BooleanConstant)) {
                    //   Heuristics for applying the (short) table converter. 
                    //   Only applied in final tailoring process.
                    //   Top-level constraints only.  Boolean expressions nested within top-level constraints are not tabulated.  
                    
                    if(heuristic(curnode)) {
                        ASTNode newTable=tabulate(curnode, 10000, 100000, 100000);
                        
                        if(newTable!=null) {
                            a.setChild(i, newTable);
                        }
                    }
                }
            }
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Numerical expressions
    
    private void applyHeuristicsNumerical(ASTNode top) {
        //  Iterate through every numerical expression in a top-level constraint
        //  (possibly with a container type in between)
        ArrayList<ASTNode> newCts=new ArrayList<>();
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode outer_ct=a.getChild(i);
                
                //  Find numerical expressions inside outer_ct, looking inside containers if necessary
                //  Treat ch as a stack
                ArrayList<ASTNode> ch=outer_ct.getChildren();
                
                while(ch.size()>0) {
                    ASTNode curnode=ch.remove(ch.size()-1);
                    if(curnode instanceof CompoundMatrix) {
                        ch.addAll(curnode.getChildren(1));  // Put contents of matrices back onto the stack.
                    }
                    else if(curnode.isNumerical() 
                        && !(curnode instanceof Mapping)
                        && curnode.getDimension()==0 
                        && !(curnode instanceof NumberConstant)
                        && curnode.toFlatten(false)) {
                        
                        // Trial auxiliary variable and constraint. 
                        //System.out.println("Trying numerical expression:"+curnode);
                        ASTNode tmpaux=m.global_symbols.newAuxHelper(curnode);
                        ASTNode ct=new Equals(curnode.copy(), tmpaux);
                        
                        // Temporarily replace curnode with tmpaux so that calling strongProp on outer_ct produces the appropriate answer. 
                        ASTNode p=curnode.getParent();
                        p.setChild(curnode.getChildNo(), tmpaux);
                        boolean outer_ct_strong=outer_ct.strongProp();
                        // Restore
                        curnode.setParent(null);
                        p.setChild(curnode.getChildNo(), curnode);
                        
                        //  Special case of strong prop heuristic. 
                        if((outer_ct_strong && !ct.strongProp() ) || heuristic(ct)) {
                            ASTNode newTable=tabulate(ct, 10000, 100000, 100000);
                            
                            if(newTable!=null) {
                                //System.out.println("Worked! Shocker! "+curnode+" replaced with "+tmpaux+" and constraint: "+newTable);
                                
                                newCts.add(newTable);
                                
                                //  Replace the numerical expression with the aux variable. 
                                curnode.getParent().setChild(curnode.getChildNo(), tmpaux);
                            }
                            else {
                                m.global_symbols.deleteSymbol(tmpaux.toString());
                            }
                        }
                        else {
                            m.global_symbols.deleteSymbol(tmpaux.toString());
                        }
                    }
                }
            }
            // Add the new constraints at the top level. 
            top.getChild(0).setParent(null);
            top.setChild(0, new And(top.getChild(0), new And(newCts)));
        }
    }
    
    
    
    /////////////////////////////////////////////////////////////////////////
    // Evaluate the three single-constraint heuristics
    
    private boolean heuristic(ASTNode curnode) {
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(curnode);
        ArrayList<ASTNode> varlistdups=TabulationUtils.getVariablesDup(curnode);
        
        //  Duplicate variables and within a reasonable size bound.
        //  Should perhaps look at domain size rather than # variables.
        if(varlist.size()<varlistdups.size() && varlist.size()<=10) {
            if(verbose) {
                System.out.println("H1");
            }
            return true;
        }
        
        //  Tree size is much larger than the number of variables.
        //  Either the expression is highly complex or it contains a large 
        //  number of constants. 
        if(curnode.treesize() > 5*varlist.size()) {
            if(verbose) {
                System.out.println("H2");
            }
            return true;
        }
        
        // Constraint does not get GAC, and it overlaps with some other
        // constraint that does get GAC or similar. Uses the strongProp method. 
        if(!curnode.strongProp() && varlist.size()<=10) {
            ASTNode constraints=m.constraints.getChild(0);
            if(constraints instanceof And) {
                for(int i=0; i<constraints.numChildren(); i++) {
                    ASTNode c=constraints.getChild(i);
                    if(c.strongProp()) {
                        for(int varidx=0; varidx<varlist.size(); varidx++) {
                            if(c.contains(varlist.get(varidx))) {
                                if(verbose) {
                                    System.out.println("H3");
                                }
                                return true;
                            }
                        }
                    }
                }
            }
        }
        
        return false;
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Actually perform the tabulation of an expression.
    
    private ASTNode tabulate(ASTNode totab, long suplimit, long faillimit, long impliedlimit) {
        boolean shorttable=(CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4);
        //  Check the cache.
        TabulationUtils.RetPair ret = tu.tryCacheNormalised(totab, shorttable);
        if(ret.nodereplace != null) {
            return ret.nodereplace.current_node;
        }
        
        //  First normalise  before generating table.
        ASTNode a=tu.normalise(totab);
        ASTNode newTable;
        
        if(!shorttable) {
            //  Make conventional table constraint
            double coverage=tu.probeLong(a, 10, 100);
            if(verbose) {
                System.out.println("estimated coverage with 1000 node limit:"+coverage);
            }
            
            // hack -- assuming node limit and faillimit are roughly interchangeable
            if( (coverage*faillimit/1000) > 0.5) {
                newTable = tu.makeTableLong(a, suplimit, faillimit);
                if(newTable==null && verbose) {
                    System.out.println("Ouch!");
                    System.out.println("coverage: "+coverage+" constraint:"+a);
                }
            }
            else {
                newTable=null;
            }
        }
        else {
            //  Make short table constraint
            newTable = tu.makeTableShort(a, suplimit, faillimit, impliedlimit);
        }
        
        if(newTable==null) {
            tu.saveToFailCache(ret.expstring);
            if(verbose) {
                System.out.println("Adding to failCache:"+ret.expstring);
            }
        }
        else {
            // Save in the cache
            tu.saveToCacheNormalised(ret.expstring, a, newTable);
            
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println("Tabulated: "+a);
            }
        }
        return newTable;
    }
    
}


