package savilerow;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2021 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/

import java.util.* ;
import java.io.* ;

// Contains all tabulation methods
// Has a common instance of TabulationUtils shared by all tabulation methods. 

public class Tabulation 
{
    public static final String ANSI_RESET  = "\u001B[0m";
    public static final String ANSI_RED    = "\u001B[31m";
    public static final String ANSI_GREEN  = "\u001B[32m";
    
    private static boolean verbose=false;
    
    private Model m;
    
    private TabulationUtils tu;
    
    private long nodelimit;
    
    public Tabulation(Model _m) {
        m=_m;
        tu=new TabulationUtils(m);
        nodelimit=CmdFlags.tabulate_nolimit ? Long.MAX_VALUE : 300000;
    }
    
    public void process(boolean prop) {
        //  First, MakeTable functions.
        processMakeTable(m.constraints);
        
        if(!prop) {
            
            if(CmdFlags.make_short_tab==3 || CmdFlags.make_short_tab==4) {
                //  Tabulation of constraints. 
                //  First find sets of constraints with the same scope, attempt
                //  tabulation of the conjunction of them.
                HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=buildScopesList(m.constraints);
                identicalScopes(m.constraints, scopeslist);
                
                //  The other three heuristics -- applied to each top-level constraint
                applyHeuristicsBool(m.constraints);
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Deal with MakeTable functions
    //  Top-down descent of tree. 
    private void processMakeTable(ASTNode curnode) {
        if(curnode instanceof MakeTable) {
            if(CmdFlags.make_short_tab==0) {
	            //  Option 0 given on command line. 
	            //  Remove the MakeTable function
	            ASTNode parent=curnode.getParent();
	            int idx=curnode.getChildNo();
	            curnode.getChild(0).setParent(null);
	            parent.setChild(idx, curnode.getChild(0));
	            
	            //  Recursively process the replacement ASTNode. 
	            processMakeTable(parent.getChild(idx));
	        }
	        else {
	            //  Tabulate with no limits. Always returns a table constraint. 
                ASTNode newTable=tabulate(curnode.getChild(0), Long.MAX_VALUE, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), "MakeTableFunction");
	            
                curnode.getParent().setChild(curnode.getChildNo(), newTable);
            }
        }
        else {
            //  Recursive descent of the tree
            for(int i=0; i<curnode.numChildren(); i++) {
                processMakeTable(curnode.getChild(i));
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Identical scopes heuristic
    
    private void identicalScopes(ASTNode top, HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist) {
        //  Find entries in scopeslist where number of constraints > 1
        
        for(Map.Entry<ArrayList<ASTNode>,ArrayList<ASTNode>> p : scopeslist.entrySet()) {
            ArrayList<ASTNode> ctlist=p.getValue();
            if(ctlist.size()>1) {
                ASTNode totabulate=new And(ctlist);
                
                if(verbose) {
                    System.out.println("H4");
                    System.out.println("Trying ct:"+totabulate);
                }
                
                ASTNode newTable=tabulate(totabulate, nodelimit, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), "IdenticalScopes");
                
                if(newTable!=null) {
                    replaceConstraintSet(ctlist, newTable);
                    // Update scopeslist
                    ctlist.clear();
                    ctlist.add(newTable);
                }
            }
        }
    }
    
    // Collect scopes of top-level constraints. 
    private HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> buildScopesList(ASTNode top) {
        HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=new HashMap<>();
        
        // Populate scopeslist
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode ct=a.getChild(i);
                if(!(ct instanceof Table) && !(ct instanceof TableShort) && !(ct instanceof NegativeTable)
                    && !(ct instanceof Tag)) {
                    assert ct.isRelation();
                    
                    // Get the scope. 
                    ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(ct);
                    ASTNode.sortByAlpha(scope);  //  Sort alphabetically
                    
                    if(! scopeslist.containsKey(scope)) {
                        scopeslist.put(scope, new ArrayList<ASTNode>());
                    }
                    
                    scopeslist.get(scope).add(ct);
                }
            }
        }
        
        return scopeslist;
    }
    
    private void replaceConstraintSet(ArrayList<ASTNode> ctlist, ASTNode replacement) {
        ASTNode a=ctlist.get(0);
        
        a.getParent().setChild(a.getChildNo(), replacement);
        
        //  Clear the rest of the constraints.
        for(int i=1; i<ctlist.size(); i++) {
            a=ctlist.get(i);
            a.getParent().setChild(a.getChildNo(), new BooleanConstant(true));
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Three single-constraint heuristics
    
    private void applyHeuristicsBool(ASTNode top) {
        //  Simply iterate through the top-level constraints. 
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode curnode=a.getChild(i);
                //  Checks are a bit paranoid, not all are necessary
                if(!(curnode instanceof And) && curnode.isRelation()
                    && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
                    && !(curnode instanceof BooleanConstant)) {
                    //   Heuristics for applying the (short) table converter. 
                    //   Only applied in final tailoring process.
                    //   Top-level constraints only.  Boolean expressions nested within top-level constraints are not tabulated.  
                    String h=heuristic(curnode);
                    if(h!=null) {
                        ASTNode newTable=tabulate(curnode, nodelimit, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), h);
                        
                        if(newTable!=null) {
                            a.setChild(i, newTable);
                        }
                    }
                }
            }
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    // Evaluate the three single-constraint heuristics
    
    private String heuristic(ASTNode curnode) {
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(curnode);
        ArrayList<ASTNode> varlistdups=TabulationUtils.getVariablesDup(curnode);
        
        //  Duplicate variables and within a reasonable size bound.
        //  Should perhaps look at domain size rather than # variables.
        if(varlist.size()<varlistdups.size() && varlist.size()<=10) {
            if(verbose) {
                System.out.println("H1");
            }
            return "DuplicateVariables";
        }
        
        //  Tree size is much larger than the number of variables.
        //  Either the expression is highly complex or it contains a large 
        //  number of constants. 
        if(curnode.treesize() > 5*varlist.size()) {
            if(verbose) {
                System.out.println("H2");
            }
            return "LargeAST";
        }
        
        // Constraint does not get GAC, and it overlaps with some other
        // constraint that does get GAC or similar. Uses the strongProp method. 
        if(!curnode.strongProp() && varlist.size()<=10) {
            ASTNode constraints=m.constraints.getChild(0);
            if(constraints instanceof And) {
                for(int i=0; i<constraints.numChildren(); i++) {
                    ASTNode c=constraints.getChild(i);
                    if(c.strongProp()) {
                        for(int varidx=0; varidx<varlist.size(); varidx++) {
                            if(c.contains(varlist.get(varidx))) {
                                if(verbose) {
                                    System.out.println("H3");
                                }
                                return "WeakPropagation";
                            }
                        }
                    }
                }
            }
        }
        
        return null;
    }
    
    //  Does outer contain inner (not a copy of inner)?
    public boolean contains(ASTNode outer, ASTNode inner) {
        if (outer==inner) {
            return true;
        }
        else {
            for (int i=0; i < outer.numChildren(); i++) {
                if (contains(outer.getChild(i), inner)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Actually perform the tabulation of an expression.
    
    public ASTNode tabulate(ASTNode totab, long nodelimit, boolean shorttable, String heuristic) {
        ASTNode a=tu.normalise(totab);
        
        if(CmdFlags.tabulate_diagnostics) {
            CmdFlags.println(ANSI_RED+heuristic+ANSI_RESET+"   Attempting tabulation: "+a);
        }
        
        //  Check the cache.
        TabulationUtils.RetPair ret = tu.tryCacheNormalised(totab, shorttable);
        if(ret.nodereplace != null) {
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_GREEN+"Tabulated"+ANSI_RESET+" by retrieving from cache.");
            }
            return ret.nodereplace.current_node;
        }
        
        //  First normalise  before generating table.
        //ASTNode a=tu.normalise(totab);
        ASTNode newTable;
        long nodecount=-1;
        
        if(!shorttable) {
            newTable = tu.makeTableLong(a, nodelimit);
            nodecount=tu.nodecount;
        }
        else {
            //  Make short table constraint
            if(nodelimit==Long.MAX_VALUE) {
                //  No limit.
                newTable = tu.makeTableShort(a, nodelimit, nodelimit, nodelimit);
            }
            else {
                newTable = tu.makeTableShort(a, 10000, 100000, 100000);
            }
        }
        
        if(newTable==null) {
            tu.saveToFailCache(ret.expstring);
            if(verbose) {
                System.out.println("Adding to failCache:"+ret.expstring);
            }
        }
        else {
            // Save in the cache
            tu.saveToCacheNormalised(ret.expstring, a, newTable);
            
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_GREEN+"Tabulated"+ANSI_RESET+" in nodes:"+nodecount);
            }
        }
        return newTable;
    }
    
}


