/*
 *===================================================================================================================
 * 																													
 *	Copyright �2013 [see full license agreement, license.txt]
 *	Radu Calinescu, Kenneth Johnson and Yasmin Rafiq, 
 *  {radu.calinescu, kenneth.johnson, yr534}@york.ac.uk> (University of York)
 *===================================================================================================================  
 *
 *	This file is part of the COVE tool suite for the development of self-adaptive service based systems. 
 *					
 *		The COVE tool suite is free software: you can redistribute it and/or modify it 				
 *		under the terms of the GNU Affero General Public License as published by the Free Software Foundation,  		
 *		either version 3 of the License, or (at your option) any later version.  																			
 *																																																																														
 *		You should have received a copy of the GNU Affero General Public License along with IPGen. 
 *		If not, see <http://www.gnu.org/licenses/> 						
 * 																													
 *===================================================================================================================
 */



package ipSupportTool;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import org.apache.log4j.Logger;

public class IPSupport {


	// Array of concrete services managed by the Intelligent Proxy
	public Service[] services; 

	// The service-level agreements for the abstract methods
	public HashMap<String,MethodSLA> slas; 

	// The indices for the abstract methods
	private HashMap<String,Integer> indices; 

	// The index of the last concrete service used for each abstract method
	private HashMap<String,Integer> lastServiceIndices;

	// Setup the log4j loggers 
	static Logger logger = Logger.getLogger("");

	private String prefix; 
	


	public IPSupport(String methodName, String prefix) {
		
		indices = new HashMap<String,Integer>(); // NEW
		indices.put(methodName, new Integer(-1)); // NEW

		// 4. Instantiate the last service index hashmap
		lastServiceIndices = new HashMap<String,Integer>();
		lastServiceIndices.put(methodName, -1);

		this.prefix = prefix; 

	}
	//----------------------------------------------------------------------------------------	
	public HashMap<String, Integer> getIndices() {
		return indices;
	}
	public void setIndices(HashMap<String, Integer> indices) {
		this.indices = indices;
	}

	public HashMap<String, Integer> getLastServiceIndices() {
		return lastServiceIndices;
	}

	public void setLastServiceIndices(HashMap<String, Integer> lastServiceIndices) {
		this.lastServiceIndices = lastServiceIndices;
	}


	public int nconcrete(){
		return services.length; 
	}

	//Note: Given the index of a concrete service, return the number of methods.
	//note that a concrete service may not implement all of the abstract methods. there is no limitations
	//on the number of methods per concrete service.
	public int getNumMethods(){
		return 1; 
	}

	public double[] getCost(String methodName){

		double [] costs = new double[services.length]; 

		for(int i=0; i < services.length; i++){

			costs[i] = services[i].getCost(methodName);
		}

		return costs; 
	}

	public String  getMethodNames(){
		String methodNames = "";
		for(int i=0; i < services.length; i++){
			methodNames = services[i].getMethodName();
		}
		String delims = "[=]+";
		String[] tokens = methodNames.split(delims);
		return tokens[0]; 
	}



	public double [] getpSucc(String methodName){

		double [] succRates = new double[services.length];

		for(int i=0; i < services.length; i++){

			succRates[i] = services[i].getEstimate(methodName);
		}

		return succRates; 
	}


	//----------------------------------------------------------------------------------------


	public String setServiceMethodCost(int serviceIndex, String methodName, double cost){

		// 1. Validate parameters

		// 1. check to see if method exists 
		if (services[serviceIndex].hasMethod(methodName)){

			//1.1 update method parameters 'Cost' and 'pPrior' 
			services[serviceIndex].setMethodCost(methodName, cost);		   		
			return "method Cost updated : " + cost;
		}
		//2. Else return method does not exist 
		else{
			return "method name does not exist"; 
		}
	}


	public String setServiceMethodPriorValue(int serviceIndex, String methodName, double pPrior){

		// 1. Validate parameters

		// 1. check to see if method exists 
		if (services[serviceIndex].hasMethod(methodName)){

			//1.1 update method parameters 'Cost' and 'pPrior' 
			services[serviceIndex].setMethodPriorValue(methodName, pPrior);		   		
			return "method prior updated : " + pPrior;
		}
		//2. Else return method does not exist 
		else{
			return "method name does not exist"; 
		}
	}

	public String setServiceMethodAlphaValue(int serviceIndex, String methodName, double alpha){

		// 1. Validate parameters

		// 1. check to see if method exists 
		if (services[serviceIndex].hasMethod(methodName)){

			//1.1 update method parameters 'Cost' and 'pPrior' 
			services[serviceIndex].setMethodAlphaValue(methodName, alpha);		   		
			return "method alpha updated : " + alpha;
		}
		//2. Else return method does not exist 
		else{
			return "method name does not exist"; 
		}
	}

	public String setServiceMethodSmoothingParameter(int serviceIndex, String methodName, double c0){

		// 1. Validate parameters

		// 1. check to see if method exists 
		if (services[serviceIndex].hasMethod(methodName)){

			//1.1 update method smoothing parameter 
			services[serviceIndex].setMethodSmoothingParameter(methodName, c0);		   		
			return "method smoothing param updated : " + c0;
		}
		//2. Else return method does not exist 
		else{
			return "method name does not exist"; 
		}
	}

	public void setMethodSla(String methodName, MethodSLA sla) {

		//1. Set the SLA for each method
		slas.put(methodName, sla);

		//2. Invalidate the fixed index for the method
		indices.put(methodName, new Integer(-1)); 
	}

	public void setMethodIndex(String methodName, int i) { 
		indices.put(methodName, new Integer(i)); 
	} 

	protected int getServiceIndex(String methodName) throws Exception {

		// 0. If an index is preset, then use it
		int index = this.indices.get(methodName); 
		// 1. Find out required SLA for this abstract method
		MethodSLA sla = slas.get(methodName);

		if (index != -1) { 
			//int serviceIndex = -1;
			List<Service> sList = Arrays.asList(services);
			Iterator itr = sList.iterator(); 
			String logEntry ="";
			while(itr.hasNext()){
				Service ws = (Service)itr.next();
				logEntry = logEntry + ws.getEstimate(methodName) + "                             ";
			}

			long t = System.currentTimeMillis() / 1000;
			logger.warn(prefix + t + "            " + index +  "            " + logEntry);
			return index; 
		}
		else{
			// 1. Identify a concrete service that contains such a method and which satisfies this SLA
			int serviceIndex = -1;
			List<Service> sList = Arrays.asList(services);
			Iterator itr = sList.iterator(); 
			double maxCost = 5000;
			String logEntry ="";

			while(itr.hasNext()){
				Service ws = (Service)itr.next();
				if(ws.satisfiesSLA(methodName, sla) && ws.getCost(methodName) < maxCost){
					maxCost = ws.getCost(methodName);
					serviceIndex = sList.indexOf(ws);
				}
				logEntry = logEntry + ws.getEstimate(methodName) + "                             ";
			}

			// 3. Raise an exception if no service satisfies the SLA
			if (serviceIndex == -1) {
				//throw new Exception("Cannot find concrete service complying with requested SLA");
				logger.error(prefix + "Cannot find concrete service complying with requested SLA");
			}
			long t = System.currentTimeMillis() / 1000;
			logger.warn(prefix + "		" + t + "            " + serviceIndex +  "            " + logEntry);  

			// 4. Return service index
			return serviceIndex;
		}


	}
	

}
