/*
 *===================================================================================================================
 * 																													
 *	Copyright �2013 [see full license agreement, license.txt]
 *	Radu Calinescu, Kenneth Johnson and Yasmin Rafiq, 
 *  {radu.calinescu, kenneth.johnson, yr534}@york.ac.uk> (University of York)
 *===================================================================================================================  
 *
 *	This file is part of the COVE tool suite for the development of self-adaptive service based systems. 
 *					
 *		The COVE tool suite is free software: you can redistribute it and/or modify it 				
 *		under the terms of the GNU Affero General Public License as published by the Free Software Foundation,  		
 *		either version 3 of the License, or (at your option) any later version.  																			
 *																																																																														
 *		You should have received a copy of the GNU Affero General Public License along with IPGen. 
 *		If not, see <http://www.gnu.org/licenses/> 						
 * 																													
 *===================================================================================================================
 */


package ipSupportTool;

import org.apache.log4j.Logger;

public class TransitionProbability {	
	private double c0;            // smoothing parameter
	private double alpha;         // observation ageing coefficient; 
	private long tk;              // current time-stamp in seconds
	private int k;                // number of invocations
	private double fk;            // function 
	private double gk;            // function
	private double p0;            // prior estimate
	private double pk;            // probability estimate after k-th invocation 
	private double beta; 
	private static Logger logger = Logger.getLogger("");
	private long coolingOffTime;private int N; 

	private long prevTimestamp;
	private long[] durations; 
	private int dIndex;
	private boolean dFull;
	private double totalDistance; 
	private double p_max; 
	private double epsilon; 


	public String toString(){
		return this.getEstimate()+"";
	}

	public TransitionProbability(double p0) {
		// 1. Initialise learning parameters
		c0 = 1000;
		alpha = 1.0001; 
		beta = 1.005;
		coolingOffTime = 200;

		// 2. Initialise time and invocation number
		tk =  0; //System.currentTimeMillis() / 1000;
		k = 0;

		N = 200; 
		durations = new long[N]; 
		for(int i=0;i<durations.length;i++){
			durations[i] = 0;
		}
		prevTimestamp = 0; //System.currentTimeMillis() / 1000;
		dIndex = 0;
		dFull = false;
		totalDistance = 0; 
		p_max = 0.05; 
		epsilon = 0.05; 

		// 3. Initialise probabilities
		this.p0 = this.pk = p0;
	}

	public void stoppedMonitoring() {
		k = 0;
	}

	public void successfulInvocation() {
		recordInvocation(1.0);
	}

	public void unsuccessfulInvocation(){
		recordInvocation(0.0);
	}

	private void recordInvocation(double sigmak) {
		// 1. Increment the number of invocations.
		k++; 

		// 2. Obtain the current time
		long t = System.currentTimeMillis() / 1000;
		double overlineT = 0; 

		// 3. Calculate fk, gk
		if (k==1) {
			fk = sigmak;
			gk = 1.0;
		}
		else {

			//sliding window for N observations
			totalDistance -= durations[dIndex];
			durations[dIndex] = t - this.prevTimestamp;
			totalDistance += durations[dIndex];
			if(++dIndex == N){
				dIndex = 0;
				dFull = true;
			}
			this.prevTimestamp = tk;

			if (dFull) { // N or more observations have been made -> time to check alpha
				overlineT = (long) totalDistance / (double) N;

				// calculate new alpha
				double cEpsilon = epsilon * epsilon; 
				double cAlpha = (1 + 4*cEpsilon * p_max) / (1 - 4*cEpsilon * p_max); 
				double minusOverlineT = 1/overlineT; 
				alpha = Math.pow(cAlpha, minusOverlineT); 

				// calculate new c0

				double logAlpha = Math.log10(alpha); 
				c0 = 1 / (2 * logAlpha); 

			}

			double minusObservationAge = tk - t;
			double weight = Math.pow(alpha, minusObservationAge);
			fk = sigmak + fk * weight;
			gk = 1 + gk * weight;
		}

		// Calculate new pk
		pk = (c0 * p0 + k * (fk / gk)) / (c0 + k); 

		// Update tk
		tk = t;

		// Log result
		logger.info("**** recording observation #" + k + " (" + sigmak + "); new estimate is " + pk + ";  at timestamp  " + tk);
		logger.info("[TransitionProbability : Graph"+ tk + "  " + pk  + "  " + alpha  + "  " + c0 + "  " + totalDistance + "   " + overlineT);

	}

	public double getEstimate() {
		// Figure out the current time
		long t = System.currentTimeMillis() / 1000;

		if (t - tk < coolingOffTime) {
			return pk;
		}

		// Calculate age of last observation
		long minusDeltaT = (tk + coolingOffTime) - t;

		// Calculate weights of prior and latest estimate
		double b = Math.pow(beta, minusDeltaT);


		// Return adjusted estimate
		return (1 - b) * p0 + b * pk;

	}

	public long get_tk(){
		return tk;
	}
	
	public double getAlpha(){
		return alpha; 
	}
	
	public double get_c0(){
		return c0; 
	}

	public void setAlpha(double alpha){
		this.alpha = alpha; 
	}

	public void setSmoothingPrameter(double c0){
		this.c0 = c0; 
	}

	public void setPriorValue(double pPrior){
		this.p0 = pPrior; 
	}
}
