/*
 *===================================================================================================================
 * 																													
 *	Copyright �2013 [see full license agreement, license.txt]
 *	Radu Calinescu, Kenneth Johnson and Yasmin Rafiq, 
 *  {radu.calinescu, kenneth.johnson, yr534}@york.ac.uk> (University of York)
 *===================================================================================================================  
 *
 *	This file is part of the COVE tool suite for the development of self-adaptive service based systems. 
 *					
 *		The COVE tool suite is free software: you can redistribute it and/or modify it 				
 *		under the terms of the GNU Affero General Public License as published by the Free Software Foundation,  		
 *		either version 3 of the License, or (at your option) any later version.  																			
 *																																																																														
 *		You should have received a copy of the GNU Affero General Public License along with IPGen. 
 *		If not, see <http://www.gnu.org/licenses/> 						
 * 																													
 *===================================================================================================================
 */

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class PrismModel {
	final String STATE_NAME = "a";//name of the state variable within the module
	private String moduleName;
	public HashMap <String,State> states;//mappings id -> state object
	private HashMap <String,Edge> edges;//mappings   id -> edge object	
	private HashMap <String,Boolean> marked;
	private ArrayList<String> pctlFormulae;


	//------------------------------------------------------------------------------------------
	public ArrayList<String> getPctls(){
		return this.pctlFormulae;

	};
	//------------------------------------------------------------------------------------------
	public PrismModel(){
		this.states = new HashMap<String, State>();
		this.edges = new HashMap<String,Edge>();
		this.marked = new HashMap<String,Boolean>();
		this.moduleName="WorkFlow";
		this.pctlFormulae = new ArrayList<String>();
	}
	//------------------------------------------------------------------------------------------
	public void addpctlFormulae(String formula){
		this.pctlFormulae.add(formula);
	}
	//------------------------------------------------------------------------------------------
	public void addState(State state) {
		this.states.put(state.id, state);
		this.marked.put(state.id, Boolean.FALSE);
	}
	//------------------------------------------------------------------------------------------
	public void addEdge(Edge edge) {
		this.edges.put(edge.id, edge);
	}
	//------------------------------------------------------------------------------------------
	public State findState(String attr){
		for ( Map.Entry<String, State> entry : states.entrySet()){
			State s = entry.getValue();
			if((s.name.contentEquals(attr))||(s.id.contentEquals(attr))||(s.incomingEdges.contentEquals(attr))||
					(s.outgoingEdges.contentEquals(attr))||(s.type.contentEquals(attr)))
				return s;
		}
		return null;
	}
	//------------------------------------------------------------------------------------------
	private ArrayList<Edge> outgoingEdges(State s){
		ArrayList<Edge> egdeArray = new ArrayList<Edge>();			
		for(String id : s.outgoingEdges())			
			egdeArray.add(edges.get(id));

		return egdeArray;
	}
	//------------------------------------------------------------------------------------------
	private String generateWorkFlowModel(State source){
		String module="";
		if(this.marked.get(source.id)) return module;//stop if we've visited this state.

		if(source==findState(Translator.INIT_STATE))	
			module += "\nmodule "+this.moduleName+"\n"+STATE_NAME+
			" : [0.."+(this.states.size()-1)+"] init "
			+(findState(Translator.INIT_STATE)).n+";\n";

		//determine the outgoing edges from the source state 
		ArrayList<Edge> destinations = this.outgoingEdges(source);

		//if there are no transitions from the state then output a self-looping state
		if(destinations.isEmpty()){			
			module+= "["+source.name+"] ("+STATE_NAME+"="+(source.n)+
					") -> 1.0:("+STATE_NAME+"'="+(source.n)+");\n";		
			return module;
		}

		//Define a transition from the source state...
		module+="["+source.name+"] ("+STATE_NAME+"="+(source.n)+") -> ";

		int count=0;
		//..to each destination state, for each edge outgoing from the source
		for(Edge e : destinations){
			State destination = states.get(e.target);
			String lineEnd = (count<destinations.size()-1)?("+"):(";\n");
			module+=e.pTransition+":("+STATE_NAME+"'="+(destination.n)+")"+lineEnd;
			count++;
		}

		//mark this state as visited.
		markState(source);

		//compute transitions for the target state of each outgoing edge from source 
		for(Edge e : destinations)
			module+=generateWorkFlowModel(this.states.get(e.target));

		return module;
	}
	//------------------------------------------------------------------------------------------
	private void markState(State init) {
		this.marked.put(init.id,Boolean.TRUE);	
	}
	//------------------------------------------------------------------------------------------
	private String generateRewards(){
		String output="rewards\n";

		for(String sName : states.keySet()){
			State s = states.get(sName);
			if (s.type.contentEquals(Translator.GENERATE_FAIL_STATE))
				output += "("+STATE_NAME+"="+s.n+") : %"+s.name+"_COST;\n";
		}

		return output+"endrewards\n";
	}
	//------------------------------------------------------------------------------------------
	private String generateLabels(){
		String output = "//pctl formulae\n";
		for(String pctl : this.pctlFormulae)
			output+="//"+pctl+"\n";

		output+="//parameterised probability values\n";		
		for(State s : this.getStates())
			if (s.type.contentEquals(Translator.GENERATE_FAIL_STATE))
				output += "const double "+s.name+"_p = %"+s.name+";\n";

		output+="//labels\n";//label "stop" = a=10;
		for(State s : this.getStates())
			//if (s.type.contentEquals(Translator.GENERATE_FAIL_STATE))
			output += "label \""+s.name+"\" = "+STATE_NAME+"="+s.n+";\n";

		return output;
	}
	//------------------------------------------------------------------------------------------
	private void write(String fileName,String output){
		PrintWriter fileout;
		try {
			fileout = new PrintWriter(fileName, "UTF-8");
			fileout.println(output);				
			fileout.close();
		} catch (FileNotFoundException e) {			
			e.printStackTrace();
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
	}
	//------------------------------------------------------------------------------------------
	//generates a DTMC model of the workflow and outputs it to the text file specified by filename
	public void generateWorkFlowModel(String fileName){		
		write(fileName, generateWorkFlowModel());				
	}
	//------------------------------------------------------------------------------------------
	//generates a DTMC model of the workflow, represented as a prism module file.
	public String generateWorkFlowModel(){				
		//unmark all states
		for (String mark : this.marked.keySet())
			this.marked.put(mark, Boolean.FALSE);

		return "dtmc\n"+generateLabels()+generateRewards()+this.generateWorkFlowModel(this.findState(Translator.INIT_STATE))+"endmodule\n";
	}
	//------------------------------------------------------------------------------------------
	public String generatePctlFormulae(){
		String output="";
		for(String p : this.pctlFormulae)
			output+=(p+"\n");

		return output;
	}
	//------------------------------------------------------------------------------------------
	public String toString(){
		return generateWorkFlowModel();
	}
	//------------------------------------------------------------------------------------------
	public Edge getEdge(String id) {
		return this.edges.get(id);
	}
	//------------------------------------------------------------------------------------------
	public ArrayList<State> getStates() {
		ArrayList<State> stateArray = new ArrayList<State>();
		for (Map.Entry<String, State> entry : states.entrySet()) 
			stateArray.add(entry.getValue());

		return stateArray;
	}
	//------------------------------------------------------------------------------------------
	public void generatePctlFormulae(String fileName) {
		write(fileName, generatePctlFormulae());		
	}
}