/**
 * Represents data structure of a concrete web service
 */

/*******************************************************************************
 * Data structure for parameter
 */
function Parameter(type) {
	this.type = type;
	this.name = "";// return type doesn't require a name
	this.convCode = ""; // Conversion code for parameter or return type
}
/*******************************************************************************
 * Set conversion code for parameter
 * 
 * @param convCode
 */
Parameter.prototype.setConvCode = function(convCode) {
	this.convCode = convCode;
};

/*******************************************************************************
 * check two Parameter objects are equal or not
 * 
 * @param parameter
 *            object of Parameter
 * @returns {Boolean} true if same
 */
Parameter.prototype.isEqual = function(parameter) {
	if (this.type == parameter.type && this.name == parameter.name)
		return true;
	else
		return false;
};
/*******************************************************************************
 * Check only parameters type are same or not, don't care about parameter name
 * 
 * @param parameter
 *            object of Parameter
 * @returns {Boolean} true if same
 */
Parameter.prototype.isSimilar = function(parameter) {
	if (this.type == parameter.type)
		return true;
	else
		return false;
};
/*******************************************************************************
 * Show In Code signature
 */
Parameter.prototype.toCodeSig = function() {
	return this.type + " " + this.name;
};
/*******************************************************************************
 * default to string is in UML format
 */
Parameter.prototype.toString = function() {
	return this.name + ":" + this.type;
};
/*******************************************************************************
 * Print details of object of Operation
 */
Parameter.prototype.print = function() {
	alert(this.toString());
};

/*******************************************************************************
 * Data structure for opeation
 */
function Operation() {
	this.uml;
	this.name;
	this.params = [];// parameters
	this.ret;// return value
}
/*******************************************************************************
 * check two Parameter objects are equal or not
 * 
 * @param operation
 * @returns {Boolean}
 */
Operation.prototype.isEqual = function(operation) {
	// if UML are same then they are same operation
	if ((this.uml == operation.uml)) {
		return true;
	}
	// may uml not set, so if operation name and its parameters are same then it
	// is same operation
	else if (this.name == operation.name
			&& this.params.length == operation.params.length
			&& this.ret.isSimilar(operation.ret)) {
		var isEqual = true;
		for ( var i = 0; i < this.params.lenght; i++) {
			if (!this.params[i].isSimilar(operation.params[i])) {
				isEqual = false;
			}
		}
		if (isEqual)
			return true;
		else
			return false;
	} else
		return false;
};

/*******************************************************************************
 * check two signature is same, it doesn't cares about name of operation and
 * parameters but only types
 * 
 * @param operation
 * @returns {Boolean}
 */
Operation.prototype.isSimilar = function(operation) {
	// if uml are same then they are same operation
	if ((this.uml == operation.uml)) {
		return true;
	}
	// may uml not set, so if operation name and its parameters are same then it
	// is same operation
	else if (this.params.length == operation.params.length
			&& this.ret.isSimilar(operation.ret)) {
		var isEqual = true;
		for ( var i = 0; i < this.params.lenght; i++) {
			if (!this.params[i].isSimilar(operation.params[i])) {
				isEqual = false;
			}
		}
		if (isEqual)
			return true;
		else
			return false;
	} else
		return false;
};
/*******************************************************************************
 * Takes the operation's UML presentation and sets the parameters, with this
 * presentation
 * 
 * @param uml
 */
Operation.prototype.setWithUML = function(uml) {
	// foo(param1:float, param2:float, param3:Object):float
	this.uml = uml;
	var parts = uml.split("(");
	this.name = parts[0];// name
	var paramStr = parts[1].split("):");
	var ret = new Parameter(paramStr[1]);
	this.ret = ret;// return float

	// if there is at least one parameter then take them
	if (paramStr[0]) {
		var params = paramStr[0].split(",");// float param1, float param2,
		// Object
		// param3
		// clear the old parameters, if exist
		this.params.splice(0, params.length);

		for ( var i = 0; i < params.length; i++) {
			var param = params[i].trim().split(":");
			var name = param[0];
			var type = param[1];
			var parameter = new Parameter(type);// type of parameter to
			// constructor
			parameter.name = name;
			this.params.push(parameter);
		}
	}

};
/*******************************************************************************
 * set conversion code to given name and type of parameter
 * 
 * @param type
 *            parameter type
 * @param name
 *            parameter name
 * @param convCode
 *            code conversion for the parameter
 */
Operation.prototype.setConvCode2Param = function(type, name, convCode) {
	var param = new Parameter(type);
	param.name = name;
	// search parameters and find the corresponding parameter then set its
	// conversion code
	for ( var i = 0; i < this.params.length; i++) {
		if (this.params[i].isEqual(param)) {
			this.params[i].setConvCode(convCode);
		}
	}
};
/*******************************************************************************
 * Show in code signature, not in UML for example: float foo(float p1, float p2)
 * 
 * @returns UML signature of operation
 */
Operation.prototype.toCodeSig = function() {
	var uml = this.ret.type + " " + this.name + "(";
	var isFirst = true;
	for ( var i = 0; i < this.params.length; i++) {
		if (isFirst) {
			uml += this.params[i].toCodeSig();
			isFirst = false;
		} else {
			uml += ", " + this.params[i].toCodeSig();
		}
	}
	uml += ")";
	return uml;
};

/*******************************************************************************
 * Default toString presentation is in UML format
 * 
 * @returns UML signature of operation
 */
Operation.prototype.toString = function() {
	var uml = this.name + "(";
	var isFirst = true;
	for ( var i = 0; i < this.params.length; i++) {
		if (isFirst) {
			uml += this.params[i];
			isFirst = false;
		} else {
			uml += ", " + this.params[i];
		}
	}
	uml += "):" + this.ret.type;
	return uml;
};
/*******************************************************************************
 * Print object details of object of Operation
 */
Operation.prototype.print = function() {
	alert(this.toString());
};

/*******************************************************************************
 * Data structure for concrete web services
 */
function ConcreteWS(url, name) {
	this.url = url;
	this.name = name;
	this.operation;// object of Operation
}

/*******************************************************************************
 * Extracts Operation signature from UML string, and sets to operation object
 * 
 * @param uml
 *            uml string
 */
ConcreteWS.prototype.setOperationWithUML = function(uml) {
	// this.operation = operation;
	var operation = new Operation();
	operation.setWithUML(uml);
	this.operation = operation;
};
/*******************************************************************************
 * check two concrete WS are equal or not
 * 
 * @param concreteWS
 *            object of concreteWS
 * @returns {Boolean} true if equal
 */
ConcreteWS.prototype.isEqual = function(concreteWS) {
	if (this.url == concreteWS.url && this.name == concreteWS.name
			&& this.operation.isEqual(concreteWS.operation))
		return true;
	else
		return false;
};
/*******************************************************************************
 * returns concrete web service class
 */
ConcreteWS.prototype.toString = function() {
	return "URL: " + this.url + " Name: " + this.name + " Operation: "
			+ this.operation;
};
/*******************************************************************************
 * alerts concrete web service class
 */
ConcreteWS.prototype.print = function() {
	alert(this.toString());
};

/*******************************************************************************
 * Abstract Operation Data Structure
 */
function AbstractOperation(opUml) {
	this.opUml = opUml;// Abstract operation UML
	this.concreteWSList = [];// List of assigned concrete class
}

/*******************************************************************************
 * Assign new concrete web service
 * 
 * @param concreteWS
 *            the concrete web service
 * @return true if successfully assign false otherwise
 */
AbstractOperation.prototype.assign = function(concreteWS) {
	// add new concrete WS
	var index = this.getConcreteWSIndex(concreteWS);
	if (index == -1) {
		this.concreteWSList.push(concreteWS);
		return true;
	} else {
		// Update the exist concrete WS
		this.concreteWSList[index] = concreteWS;
		return false;
	}
};
/*******************************************************************************
 * Unassign new concrete web service
 * 
 * @param concreteWS
 *            the concrete web service
 * @return true if successfully assign false otherwise
 */
AbstractOperation.prototype.unAssign = function(concreteWS) {
	// get the index of operation to be removed
	var index = this.getConcreteWSIndex(concreteWS);
	if (index != -1) {
		this.concreteWSList.splice(index, 1);
		return true;
	} else {
		return false;
	}
};
/*******************************************************************************
 * Unassign new concrete web service, by its index
 * 
 * @param index
 *            the index of concrete web service in the concreteWSList
 * @return true if successfully assign false otherwise
 */
AbstractOperation.prototype.unAssignByIndex = function(index) {
	if (index != -1) {
		this.concreteWSList.splice(index, 1);
		return true;
	} else {
		return false;
	}
};
/*******************************************************************************
 * returns index of concrete web service by operation name
 * 
 * @param concreteWS
 *            concreteWS name
 * @returns {Number} index
 */
AbstractOperation.prototype.getConcreteWSIndex = function(concreteWS) {
	var index = -1;
	for ( var i = 0; i < this.concreteWSList.length; i++) {
		if (this.concreteWSList[i].isEqual(concreteWS)) {
			index = i;
			break;
		}
	}
	return index;
};
/*******************************************************************************
 * Check any operation of given concrete web service has already been assigned
 * to abstract operation list or not, if it is then returns index of concrete
 * web service in the abstract operation list
 * 
 * @param concreteWS
 *            concreteWS name
 * @returns {Number} index
 */
AbstractOperation.prototype.hasAnyOperationOfConcreteWS = function(concreteWS) {
	var index = -1;
	for ( var i = 0; i < this.concreteWSList.length; i++) {
		if (this.concreteWSList[i].name == concreteWS.name
				&& this.concreteWSList[i].url == concreteWS.url) {
			index = i;
			break;
		}
	}
	return index;
};
/*******************************************************************************
 * Check the given concrete web service already assigned to abstract operation
 * list or not, if it is then returns index of concrete web service in the
 * abstract operation list
 * 
 * @param concreteWS
 *            concreteWS name
 * @returns {Number} index
 */
AbstractOperation.prototype.isAssigned = function(concreteWS) {
	var index = -1;
	for ( var i = 0; i < this.concreteWSList.length; i++) {
		if (this.concreteWSList[i].isEqual(concreteWS)) {
			index = i;
			break;
		}
	}
	return index;
};
/*******************************************************************************
 * check two abstract operations signature are equal
 * 
 * @param opUML
 *            uml signature of operation
 * @returns {Boolean}
 */
AbstractOperation.prototype.isEqual = function(opUML) {
	if (this.opUml == opUML) {
		return true;
	} else
		return false;
};
/*******************************************************************************
 * returns uml of abstract operation and associated concrete operations
 */
AbstractOperation.prototype.toString = function() {
	var text = "Abstract Op UML: " + this.opUml
			+ " Assigned Concrete Operations:[ ";
	for ( var i = 0; i < this.concreteWSList.length; i++) {
		text += concreteWSList[i].toString();
	}
	text += " ]";

	return text;

};
AbstractOperation.prototype.print = function() {
	alert(this.toString());
};

/*******************************************************************************
 * Abstract Web Service Data structure
 */
function IntelligentProxy() {
	this.packageName = "";
	this.proxyName = "";
	this.abstractOpList = [];
}
/*******************************************************************************
 * Add new abstract operation
 * 
 * @param abstractOp
 *            abstract operation
 */
IntelligentProxy.prototype.add = function(abstractOp) {
	// add element
	this.abstractOpList.push(abstractOp);
};
/*******************************************************************************
 * remove abstract operation
 * 
 * @param abstractOp
 *            abstract operation
 */
IntelligentProxy.prototype.remove = function(abstractOp) {
	// get the index of operation to be removed
	var index = this.getAbsOpIndex(abstractOp);
	if (index != -1) {
		this.abstractOpList.splice(index, 1);
	} else {
		alert(abstractOp + " can not be removed");
	}
};
/*******************************************************************************
 * returns abstract operation, by operation name
 * 
 * @param abstractOpUML
 *            abstract operation in UML
 * @returns abstract operation object
 */
IntelligentProxy.prototype.getAbsOp = function(abstractOpUML) {
	var index = -1;
	for ( var i = 0; i < this.abstractOpList.length; i++) {
		if (this.abstractOpList[i].isEqual(abstractOpUML)) {
			index = i;
			break;
		}
	}
	return this.abstractOpList[index];
};
/*******************************************************************************
 * returns index of abstract operation by operation name
 * 
 * @param abstractOpUML
 *            abstract operation
 * @returns {Number} index of abstract operation equals to abstractOpUML
 */
IntelligentProxy.prototype.getAbsOpIndex = function(abstractOpUML) {
	var index = -1;
	for ( var i = 0; i < this.abstractOpList.length; i++) {
		if (this.abstractOpList[i].isEqual(abstractOpUML)) {
			index = i;
		}
	}
	return index;
};
/*******************************************************************************
 * Assign concrete web service to given name of abstract operation
 * 
 * @param abstractOpUML
 *            the uml presentation of abstract operation
 * @param concreteWS
 *            the concrete web service (not its name, it should be a type of web
 *            service)
 * @returns returns true if successful otherwise false
 */
IntelligentProxy.prototype.assign = function(abstractOpUML, concreteWS) {
	// get the index of abstract operation
	var index = this.getAbsOpIndex(abstractOpUML);
	// then assign concrete WS to that abstract operation
	return this.abstractOpList[index].assign(concreteWS);
};
/*******************************************************************************
 * unassign concrete web service to given name of abstract operation
 * 
 * @param abstractOpUML
 *            the uml presentation of abstract operation
 * @param concreteWS
 *            the concrete web service (not its name, it should be a type of web
 *            service)
 * @returns returns true if successful otherwise false
 */
IntelligentProxy.prototype.unAssign = function(abstractOpUML, concreteWS) {
	// get the index of abstract operation
	var index = this.getAbsOpIndex(abstractOpUML);
	// then unassign concrete WS to that abstract operation
	return this.abstractOpList[index].unAssign(concreteWS);
};

IntelligentProxy.prototype.toString = function() {
	var text = "Abstract WS: packageName:" + this.packageName + " proxyName:"
			+ this.proxyName;
	for ( var i = 0; i < this.abstractOpList.length; i++) {
		text += i + 1 + ": " + this.abstractOpList[i].toString() + " ";
	}
	return text;
};
IntelligentProxy.prototype.print = function() {
	alert(this.toString());
};
