/*
 *===================================================================================================================
 * 																													
 *	Copyright �2013 [see full license agreement, license.txt]
 *	Radu Calinescu, Yasmin Rafiq, 
 *  {radu.calinescu, yr534}@york.ac.uk> (University of York)
 *===================================================================================================================  
 *
 *	This file is part of the iPASS intelligent proxy-driven adaptive service-based systems framework for the 
 *  development of self-adaptive computing applications. 
 *					
 *		The iPASS intelligent proxy generator (IPGen) is free software: you can redistribute it and/or modify it 				
 *		under the terms of the GNU Affero General Public License as published by the Free Software Foundation,  		
 *		either version 3 of the License, or (at your option) any later version.  																			
 *																																																																														
 *		You should have received a copy of the GNU Affero General Public License along with IPGen. 
 *		If not, see <http://www.gnu.org/licenses/> 						
 * 																													
 *===================================================================================================================
 */

package proxygen.business;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import proxygen.business.util.Config;
import proxygen.business.util.External;
import proxygen.business.util.Util;
import proxygen.model.json.webservice.AbstractOperation;
import proxygen.model.json.webservice.AbstractWS;
import proxygen.model.json.webservice.ConcreteWS;
import proxygen.model.json.webservice.Operation;
import proxygen.model.json.webservice.Parameter;

public class GenerateProxyFile {

	/**
	 * Created: Jul 13, 2013 6:13:29 PM generates the code for intelligent
	 * proxyName.java file
	 * 
	 * @param sessionID
	 * @param abstractWS
	 * @throws Exception
	 */
	public void generateIntelligentProxy(String sessionID, AbstractWS abstractWS)
			throws Exception {
		// Output file for Java code
		BufferedWriter out = null;
		try {
			Util.debug("Intelligent proxy generating");

			// Output file path of source files
			String outputDir = Config.getTempFolder() + "/" + sessionID;

			// Output directory of .java file
			String proxyDirectory = outputDir + "/src/"
					+ abstractWS.getPackageName();
			Util.createDirectory(proxyDirectory);

			String proxyName = Util.makeUpperFirstLetter(abstractWS
					.getProxyName());
			// Add prefix to proxyName
			String proxyFile = proxyDirectory + "/" + proxyName + ".java";
			// 1. Delete file if it already exists
			External.deletefile(proxyFile);

			// 2. Generate intelligent proxy code

			// Open file in write mode
			FileWriter fstream = new FileWriter(proxyFile, true);
			out = new BufferedWriter(fstream);
			writeHeader(out, outputDir, abstractWS);

			out.write("// The Intelligent Proxy class");
			out.newLine();
			// 4. Write Java class header
			out.write("public class " + proxyName + " extends IPSupport{");
			out.newLine();
			out.newLine();
			out.write("   // The actual stubs (NB: these are application specific!)");
			out.newLine();

			// 5. Generate class fields
			// 5.1. Generate one stub field for each web service
			List<ConcreteWS> uniqueConcreteWSList = new GenerateProxy()
					.getUniqueConcreteWSList(abstractWS);
			generateStubs(out, uniqueConcreteWSList);
			// 5.3 Setup log4j for logging
			out.newLine();
			out.write("   // Setup the log4j logger ");
			out.newLine();
			out.write("   static Logger logger = Logger.getLogger(" + '"' + '"'
					+ ");");
			out.newLine();
			out.newLine();
			// 5.3.2 Create a Model updater
			out.write("   // Create a Model updater");
			out.newLine();
			out.write("   private Model model; ");
			out.newLine();
			out.newLine();
			// 6. Syntheses constructor
			out.newLine();
			out.write("   // Constructor (NB: these are application specific!)");
			out.newLine();
			out.write("   public "
					+ proxyName
					+ "(String methodName, String prefix, Model model) throws Exception {");
			out.newLine();
			out.newLine();
			out.write("    	 super(methodName, prefix);");
			out.newLine();
			out.newLine();
			out.write("         //1. Initialise the Model updater with the initial model provided");
			out.newLine();
			out.write("          this.model = model; ");
			out.newLine();
			out.newLine();
			out.write("       // 2. Initialise stubs for each concrete service");
			out.newLine();
			// initialize Stubs
			initializeStubs(out, uniqueConcreteWSList);

			out.write("       // 3. Create the service array");
			out.newLine();
			out.write("       // 3.1. Instantiate array");
			out.newLine();
			out.write("       services = new Service["
					+ uniqueConcreteWSList.size() + "];");
			out.newLine();
			out.write("       // 3.2. Declare arrays of abstract web method names, costs and a priory success rates");
			out.newLine();
			out.write("       String[] methodNames;");
			out.newLine();
			out.write("       double[] costs;");
			out.newLine();
			out.write("       double[] aPrioriSuccessRates;");
			out.newLine();
			out.newLine();
			out.write("       // 3.3. Instantiate services");
			out.newLine();
			// get abstract operations names associated to each concrete
			// operations.
			Map<String, ArrayList<String>> abstractConcreteMap = abstractsAssociatedConcrete(
					abstractWS, uniqueConcreteWSList);
			associatedAbstractOperations(out, abstractConcreteMap);
			// 4. Instantiate the SLAs for the abstract methods
			writeOutSLAs(out, abstractWS.getAbstractOpList());
			// TODO: remove later
			out.flush();
			// 7. Generate one proxy method for each INTELLIGENT PROXY service
			// method
			generateAbstractMethods(out, abstractWS, uniqueConcreteWSList);
			// 8. Write footer
			out.newLine();
			out.write("}");
			out.flush();
		} catch (Exception e) {
			Util.error("Error: " + e.getMessage());
			throw e;
		} finally {
			// 9. Close Java file in all cases
			out.close();
		}

	}

	/**
	 * 7. Generate one proxy method for each INTELLIGENT PROXY service method
	 * Created: Jul 14, 2013 6:28:05 PM
	 * 
	 * @param out
	 * @param abstractWS
	 * @param uniqueConcreteWSList
	 * @throws IOException
	 */
	private void generateAbstractMethods(BufferedWriter out,
			AbstractWS abstractWS, List<ConcreteWS> uniqueConcreteWSList)
			throws IOException {
		List<AbstractOperation> abstractOpList = abstractWS.getAbstractOpList();
		for (AbstractOperation abstractOperation : abstractOpList) {
			// i is the index of the method code is generated for
			String umlMethodSignature = abstractOperation.getOpUml();
			String[] str = Util.getArgList(umlMethodSignature);
			out.write("   // Abstract web method \"" + str[0] + "\" ");
			out.newLine();
			out.write("   public "
					+ Util.convert2JavaFormat(umlMethodSignature)
					+ " throws Exception {");
			out.newLine();
			out.newLine();
			out.write("        //1. Update the model updater class");
			out.newLine();
			out.write("        model.nextInvocation(\"sendAlarm\");");
			out.newLine();
			out.newLine();
			out.write("       // 2. Remember index of used concrete service");
			out.newLine();
			out.write("      int serviceIndex = getServiceIndex(\"" + str[0]
					+ "\");");
			out.newLine();
			out.newLine();
			out.write("      // 3. Local variable for result");
			out.newLine();
			// if return is void then don't print actual return
			String actualReturn = str[str.length - 1] == "void" ? ""
					: str[str.length - 1] + " actualReturn";
			out.write("           " + actualReturn + ";");
			out.newLine();
			out.newLine();
			out.write("      // 4. Use the identified concrete service");
			out.newLine();
			out.write("      try{");
			out.newLine();
			out.write("         // 4.1. Invoke web method");
			out.newLine();
			out.write("         switch (serviceIndex) {");
			out.newLine();

			List<ConcreteWS> concreteWSList = abstractOperation
					.getConcreteWSList();
			int index = 0;
			for (ConcreteWS concreteWS : concreteWSList) {
				// if it is not last index
				if (index < concreteWSList.size() - 1) {
					out.write("         case " + index + ": {");
				} else {
					out.write("         default: {");
				}
				generateConcreteMethod(out, actualReturn, index, concreteWS);
				index++;
				out.newLine();
				out.write("           }");
			}
			out.newLine();
			out.write("           }");
			out.newLine();
			out.write("           // 4.2. Record success");
			out.newLine();
			out.write("           services[serviceIndex].recordSuccess(\""
					+ str[0] + "\");");
			out.newLine();
			out.newLine();
			out.write("           // 4.3 Return actualReturn");
			out.newLine();
			out.write("           return actualReturn;");
			out.newLine();
			out.newLine();
			out.write("        }");
			out.newLine();
			out.newLine();
			out.write("        // 5. Handle exception");
			out.newLine();
			out.write("        catch (Exception ex){");
			out.newLine();
			out.newLine();
			out.write("           // 5.1 Log an error message");
			out.newLine();
			out.write("           System.out.println(ex.getMessage());");
			out.newLine();
			out.write("           logger.error(" + '"' + "Exception: " + '"'
					+ " + ex.getMessage());");
			out.newLine();
			out.newLine();
			out.write("           // 5.2. Record failure ");
			out.newLine();
			out.write("           services[serviceIndex].recordFailure(\""
					+ str[0] + "\");");
			out.newLine();
			out.newLine();
			out.write("           // 5.3. Throw exception");
			out.newLine();
			out.write("           throw ex;");
			out.newLine();
			out.write("        }");
			out.newLine();
			out.newLine();
			out.write("    }");
			out.newLine();
			out.newLine();
		}

	}

	/**
	 * Generate concrete method invocation as well as code and return conversion
	 * Created: Jul 14, 2013 8:37:37 PM
	 * 
	 * @param out
	 * @param actualReturn
	 * 
	 * @param index
	 * @param concreteWS
	 * @throws IOException
	 */
	private void generateConcreteMethod(BufferedWriter out,
			String actualReturn, int index, ConcreteWS concreteWS)
			throws IOException {
		out.write("             //set conversion of parameters");
		out.newLine();

		Operation operation = concreteWS.getOperation();
		List<Parameter> params = operation.getParams();
		// Generate code conversion
		generateCodeConversion(out, params);

		out.newLine();
		out.write("             //create the request");
		out.newLine();
		// CalculatorStub
		String wsName = Util.makeUpperFirstLetter(concreteWS.getName())
				+ "Stub";
		// Subtract
		String concOpClass = Util.makeUpperFirstLetter(concreteWS
				.getOperation().getName());
		// Generate request
		String request = generateRequest(out, index, params, wsName,
				concOpClass);
		out.write("             //Invoke the service");
		out.newLine();
		generateResponse_Return(out, actualReturn, index, operation, wsName,
				concOpClass, request);
		out.write("             // Done");
		out.newLine();
		out.write("             break;");
		out.newLine();

	}

	/**
	 * Generates response, conversion code for return and return code itself.
	 * Created: Jul 16, 2013 1:40:32 PM
	 * 
	 * @param out
	 * @param actualReturn
	 * @param index
	 * @param operation
	 * @param wsName
	 * @param concOpClass
	 * @param request
	 * @throws IOException
	 */
	private void generateResponse_Return(BufferedWriter out,
			String actualReturn, int index, Operation operation, String wsName,
			String concOpClass, String request) throws IOException {
		int count =index+1;//for starting from 1 not 0
		out.write("             " + wsName + "." + concOpClass
				+ "Response response" + count + " = stub" + count + "."
				+ operation.getName() + "(" + request + ");");
		out.newLine();
		out.newLine();
		out.write("             // Return the result");
		out.newLine();
		String concreteRetType = operation.getRet().getType();
		String tempReturn = concreteRetType == "void" ? "" : concreteRetType
				+ " tempReturn = ";
		// tempReturn+="response" + index + ".getResult();";
		tempReturn += "response" + count + ".get_return();";
		out.write("             " + tempReturn);
		out.newLine();
		// Perform return conversion
		if (actualReturn != "") {
			String ret = "actualReturn = " + operation.getRet().getConvCode();
			// TODO sometimes .get_return()
			out.write("             " + ret);
			out.newLine();
		}
		out.newLine();
	}

	/**
	 * Generate request signature Created: Jul 16, 2013 1:30:56 PM
	 * 
	 * @param out
	 * @param index
	 * @param params
	 * @param wsName
	 * @param concOpClass
	 * @return
	 * @throws IOException
	 */
	private String generateRequest(BufferedWriter out, int index,
			List<Parameter> params, String wsName, String concOpClass)
			throws IOException {
		String request = "";
		// if method has some parameter then request will be created...
		 if (params.size() > 0) {
		// CalculatorStub.Subtract
		String requestClass = wsName + "." + concOpClass;
		// request0;
		int count = index+1;
		request = "request" + count;//to get rid of starting from 0
		// CalculatorStub.Subtract request1 = new
		// CalculatorStub.Subtract();
		String requestSignature = requestClass + " " + request + " = new "
				+ requestClass + "();";
		out.write("             " + requestSignature);
		out.newLine();
		for (Parameter param : params) {
			// request0.setFirstNumber(9);
			String paramName = getParameterLastPart(param.getName());
			String invokeParams = request + ".set"
					+ Util.makeUpperFirstLetter(paramName);
			invokeParams += "(" + paramName + ");";
			out.write("             " + invokeParams);
			out.newLine();
		}
		 }
		out.newLine();
		return request;
	}

	/**
	 * Generates code conversion for parameters Created: Jul 16, 2013 1:24:17 PM
	 * 
	 * @param out
	 * @param params
	 * @throws IOException
	 */
	private void generateCodeConversion(BufferedWriter out,
			List<Parameter> params) throws IOException {
		for (Parameter param : params) {
			// removes _ and brings just pure name of parameter
			String paramName = getParameterLastPart(param.getName());
			String convCode = param.getType() + " " + paramName + "="
					+ param.getConvCode();
			out.write("             " + convCode);
			out.newLine();
		}

	}

	/**
	 * Removes _ and brings just pure name of parameter Created: Jul 15, 2013
	 * 12:52:23 PM
	 * 
	 * @param name
	 * @return
	 */
	private String getParameterLastPart(String name) {
		// name format add_somthing_name
		String names[] = name.split("_");
		// get last part will be only 'name'
		String result = names[names.length - 1];
		return result;
	}

	/**
	 * Instantiate the SLAs for the abstract methods, indices for the abstract
	 * methods and the last service index hashmap Created: Jul 14, 2013 6:07:09
	 * PM
	 * 
	 * @param out
	 * @param abstractOperations
	 * @throws IOException
	 */
	private void writeOutSLAs(BufferedWriter out,
			List<AbstractOperation> abstractOperations) throws IOException {
		out.write("       // 4. Instantiate the SLAs for the abstract methods, indices for the abstract methods and the last service index hashmap");
		out.newLine();
		out.write("       slas = new HashMap<String,MethodSLA>();");
		out.newLine();
		out.flush();
		for (AbstractOperation abstractOperation : abstractOperations) {
			String[] str = Util.getArgList(abstractOperation.getOpUml());
			out.write("       slas.put(\""
					+ str[0]
					+ "\", new MethodSLA(0.0, Double.MAX_VALUE)); // anything is acceptable until the client tells otherwise");
			out.newLine();
		}
		out.write("	}");
		out.newLine();
		out.newLine();
	}

	/**
	 * Write out abstract operations names associated to each concrete
	 * operations.
	 * 
	 * Created: Jul 14, 2013 9:56:41 AM
	 * 
	 * @param abstractWS
	 * @param uniqueConcreteWSList
	 * @return
	 * @throws IOException
	 */
	private Map<String, ArrayList<String>> abstractsAssociatedConcrete(
			AbstractWS abstractWS, List<ConcreteWS> uniqueConcreteWSList)
			throws IOException {
		Map<String, ArrayList<String>> abstractConcreteMap = new LinkedHashMap<String, ArrayList<String>>();
		// Visit each concrete service
		for (ConcreteWS concreteWS : uniqueConcreteWSList) {
			// check which abstract operations associated with concreteWS
			for (AbstractOperation abstOp : abstractWS.getAbstractOpList()) {
				for (ConcreteWS concreteWS2 : abstOp.getConcreteWSList()) {
					if (concreteWS.equals(concreteWS2)) {
						ArrayList<String> tempAbstList;
						String concreteName = concreteWS.getName();
						String abstractName = Util
								.getArgList(abstOp.getOpUml())[0];
						if (abstractConcreteMap.containsKey(concreteName)) {
							tempAbstList = abstractConcreteMap
									.get(concreteName);
							if (!tempAbstList.contains(abstractName)) {
								tempAbstList.add(abstractName);
								abstractConcreteMap.put(concreteName,
										tempAbstList);
							}
						} else {
							tempAbstList = new ArrayList<String>();
							tempAbstList.add(abstractName);
							abstractConcreteMap.put(concreteName, tempAbstList);
						}
					}
				}
			}
		}
		return abstractConcreteMap;

	}

	/**
	 * Write out abstract operations names associated to each concrete
	 * operations.
	 * 
	 * Created: Jul 14, 2013 9:56:41 AM
	 * 
	 * @param out
	 * @param abstractWS
	 * @param uniqueConcreteWSList
	 * @throws IOException
	 */
	private void associatedAbstractOperations(BufferedWriter out,
			Map<String, ArrayList<String>> abstractConcreteMap)
			throws IOException {
		int count = 0;
		// Visit each concrete service
		for (String key : abstractConcreteMap.keySet()) {
			String a = "", b = "", c = "";
			ArrayList<String> tempAbstList = abstractConcreteMap.get(key);
			for (String string : tempAbstList) {
				if (a.length() > 0) { // this is not the first entry, so
					// add
					// a comma
					a = a + ", ";
					b = b + ", ";
					c = c + ", ";
				}

				a = a + "\"" + string + "\"";
				b = b + "0.45";
				c = c + "0.958";
			}
			// Now we are ready to generate the code
			int v = count + 1;
			out.write("       // 3.3." + v + ". Instantiate service " + v);
			out.newLine();

			out.write("       methodNames = new String[] { " + a + " };");
			out.newLine();
			out.write("       costs = new double[] { " + b + " };");
			out.newLine();
			out.write("       aPrioriSuccessRates = new double[] { " + c
					+ " };");
			out.newLine();
			out.write("       services["
					+ count
					+ "] = new Service(methodNames, costs, aPrioriSuccessRates);");
			out.newLine();
			out.newLine();
			count++;
		}

	}

	/**
	 * Created: Jul 13, 2013 8:42:26 PM
	 * 
	 * @param out
	 * @param abstractWS
	 * @throws IOException
	 */
	private void initializeStubs(BufferedWriter out,
			List<ConcreteWS> uniqueConcreteWSList) throws IOException {
		if (uniqueConcreteWSList.size() > 0) {
			int i = 1;
			for (ConcreteWS concreteWS : uniqueConcreteWSList) {
				out.write("       // 2." + i
						+ " Initialise stub for web service number " + i);
				out.newLine();
				out.write("       stub" + i);
				out.write(" = new ");
				out.write(Util.makeUpperFirstLetter(concreteWS.getName())
						+ "Stub();");
				out.newLine();
				out.write("       EndpointReference targetEpr" + i
						+ " = new EndpointReference(" + '"'
						+ concreteWS.getUrl() + '"' + ");");
				out.newLine();
				out.write("       stub" + i
						+ "._getServiceClient().setTargetEPR(targetEpr" + i
						+ ");");
				out.newLine();
				out.newLine();
				i++;
			}
			// TODO: remove later
			out.flush();
		}
	}

	/**
	 * Created: Jul 13, 2013 8:25:34 PM
	 * 
	 * @param uniqueConcreteWSList
	 * @throws IOException
	 */
	private void generateStubs(BufferedWriter out,
			List<ConcreteWS> uniqueConcreteWSList) throws IOException {
		if (uniqueConcreteWSList.size() > 0) {
			int i = 1;
			for (ConcreteWS concreteWS : uniqueConcreteWSList) {
				out.write("   private ");
				out.write(Util.makeUpperFirstLetter(concreteWS.getName()));
				out.write("Stub stub" + i + ";");
				i++;
				out.newLine();
			}
			// TODO: remove later
			out.flush();
		}
	}

	/**
	 * Created: Jul 13, 2013 6:29:33 PM Write header of java class
	 * 
	 * @param out
	 *            output java file
	 * @param sessionID
	 *            userID
	 * @param abstractWS
	 *            Abstract Web Service
	 * @throws Exception 
	 */
	private void writeHeader(BufferedWriter out, String outputDir,
			AbstractWS abstractWS) throws Exception {
		try {
			Util.debug("//Write Header");
			out.write("package " + abstractWS.getPackageName() + ";");
			out.newLine();
			out.newLine();
			out.write("import org.apache.axis2.addressing.EndpointReference;");
			out.newLine();
			out.write("import org.apache.log4j.*;");
			out.newLine();
			out.write("import java.util.*;");
			out.newLine();
			out.write("import ipSupportTool.*;");
			out.newLine();
			out.newLine();
			out.write("// The namespaces for the concrete services (NB: these are application specific!)");
			out.newLine();
			// TODO: remove later
			out.flush();
			try {
				String srcFolder = outputDir + "/src";
				File aFile = new File(srcFolder);
				ArrayList<String> pathName = listDirectories(aFile);
				if (pathName.size() > 0) {
					for (int i = 0; i < pathName.size(); i++) {
						out.write("import "
								+ pathName.get(i).replace("src.", "") + ".*;");
						out.newLine();
					}
				}
				// TODO: remove later
				out.flush();
			} catch (Exception e) {
				Util.error(e.getMessage());
				throw e;
			}
			out.newLine();
			out.newLine();
			out.flush();
		} catch (Exception e) {
			Util.error(e.getMessage());
			throw e;
		}

	}

	/***
	 * method to read the file structure and return the jar file identifier for
	 * import into java program
	 * 
	 * 
	 * @param srcFolder
	 *            src folder
	 * @return
	 */
	private ArrayList<String> listDirectories(File srcFolder) {
		ArrayList<String> x = new ArrayList<String>();
		String thisDirectory = srcFolder.getName();
		File[] listOfFiles = srcFolder.listFiles();

		// At least there should be a file
		if (listOfFiles != null) {

			boolean hasSubdirectories = false;

			for (int i = 0; i < listOfFiles.length; i++) {
				if (listOfFiles[i].isDirectory()) {
					hasSubdirectories = true;
					ArrayList<String> y = listDirectories(listOfFiles[i]);
					for (int j = 0; j < y.size(); j++) {
						x.add(thisDirectory + "." + y.get(j));
					}
				}
			}

			if (!hasSubdirectories) {
				x.add(thisDirectory);
			}
		}

		return x;
	}

	public static void main(String[] args) {
		System.out.println("Starts");
		GenerateProxyFile buildProxy = new GenerateProxyFile();
		String srcFolder = "C:/Users/mehmetmin/workspace/.metadata/.plugins/org.eclipse.wst.server.core/tmp1/wtpwebapps/WebApp-03/temp/B19D93D059C08F3CB967E51D38BDCE58/src//";
		File aFile = new File(srcFolder);
		ArrayList<String> pathName = buildProxy.listDirectories(aFile);
		if (pathName.size() > 0) {
			for (int i = 0; i < pathName.size(); i++) {
				System.out.println("import "
						+ pathName.get(i).replace("src.", "") + ".*;");
			}
		}
	}
}
