//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.IO;
using System.Xml;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Collections.Generic;

/// <summary>
///   The "manageability adaptor" web service for the policy engine.
/// </summary>
[WebService(Namespace = "http://www.yupe.com/")]
[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
public class PolicyEngine : ManagedResource<policyEngine>
{
    // Return itself.
    protected override object[] GetRawResources()
    {
        return new object[] { this };
    }

    // Return the required property value for the policy engine.
    protected override object GetResourceProperty(object rawResource, string property)
    {
        // Obtain the policy processor.
        PolicyEngineState state = Tools.State;

        // Return the requested property value.
        switch (property)
        {
            case "period":
                return state.Period;

            case "system":
                return state.SystemModel;

            case "policySet":
                return state.PolicySet;

            case "resourceUrls":
                // List of resource URLs prefixed by the resource type.
                List<string> annotatedUrls = new List<string>();

                // List of all resource URLs supplied to the policy engine.
                List<string> urls = new List<string>(state.ResourceUrls);

                // The parsed resoruce URLs from the policy engien processor.
                Dictionary<string, List<string>> parsedUrls = Tools.Processor.ParsedUrls;

                // Add the parsed URLs to the annotated URL list.
                if (parsedUrls != null)
                {
                    foreach (string resourceType in parsedUrls.Keys)
                    {
                        foreach (string url in parsedUrls[resourceType])
                        {
                            annotatedUrls.Add("[" + resourceType + "] " + url);
                            urls.Remove(url);
                        }
                    }
                }

                // Add all remainder URLs.
                if (state.ResourceUrls != null)
                {
                    foreach (string url in state.ResourceUrls)
                    {
                        if (urls.Contains(url))
                        {
                            annotatedUrls.Add("[???] " + url);
                        }
                    }
                }

                // Return the annotated URLs.
                return annotatedUrls.ToArray();

            default:
                throw new Exception("unknown property (" + property + ")");
        }
    }

    // Set the required property or properties of the policy engine.
    protected override ManagedResourceOpResult SetResourceProperties(policyEngine resource)
    {
        // Initialise result object.
        ManagedResourceOpResult result = new ManagedResourceOpResult();

        // Obtain the policy engine state.
        PolicyEngineState state = Tools.State;

        // Update the period if required.
        if (resource.period.HasValue)
        {
            state.Period = resource.period.Value;
        }

        // Update the model if required:
        // - remove old policies
        // - remove automatically generated manageability adaptor proxies and related code 
        //   associated with the old model
        // - dynamically build proxies for accessing the supported managed resources from the
        //   new model
        //
        if (resource.system != null)
        {
            // An exception may be thrown if the access rights for the policy engine web service
            // are not set up correctly and the dynamically generated code cannot be writen to the
            // appropriate locations.
            try
            {
                // Validate the system model.
                Tools.ValidateModel(resource.system);

                // Find the service path.
                string servicePath = System.Configuration.ConfigurationManager.AppSettings["servicePath"];

                // Remove all old proxies if present.
                if (state.SystemModel != null)
                {
                    foreach (resource resourceType in state.SystemModel.resource)
                    {
                        string proxyDir = servicePath + "\\App_WebReferences\\" + Tools.NormaliseTypeName(resourceType.ID) + "Proxy";
                        Directory.Delete(proxyDir, true);
                    }
                }

                // Remove old model-specific tools.
                string modelSpecificTools = servicePath + "\\App_Code\\ModelSpecificTools.cs";
                if (File.Exists(modelSpecificTools))
                {
                    File.Delete(modelSpecificTools);
                }

                // Process each resource type in the model and generate manageability adaptor proxies.
                foreach (resource resourceType in resource.system.resource)
                {
                    string proxyDir = servicePath + "\\App_WebReferences\\" + Tools.NormaliseTypeName(resourceType.ID) + "Proxy";
                    Directory.CreateDirectory(proxyDir);
                    File.Copy(servicePath + "\\App_Data\\ManagedResourceProxyTemplate\\ManageabilityAdaptor.disco", proxyDir + "\\ManageabilityAdaptor.disco");
                    File.Copy(servicePath + "\\App_Data\\ManagedResourceProxyTemplate\\ManageabilityAdaptor.discomap", proxyDir + "\\ManageabilityAdaptor.discomap");
                    using (StreamReader wsdlTemplate = new StreamReader(servicePath + "\\App_Data\\ManagedResourceProxyTemplate\\ManageabilityAdaptor.wsdl"))
                    {
                        using (StreamWriter wsdlFile = new StreamWriter(proxyDir + "\\ManageabilityAdaptor.wsdl"))
                        {
                            string wsdlLine;
                            while ((wsdlLine = wsdlTemplate.ReadLine()) != null)
                            {
                                if (wsdlLine.Trim().Equals("YUPE_schema"))
                                {
                                    AddSchema(resourceType, wsdlFile);
                                }
                                else
                                {
                                    wsdlFile.WriteLine(wsdlLine.Replace("YUPE_resource", Tools.NormaliseTypeName(resourceType.ID)).Replace("YUPE_Resource", Tools.NormaliseTypeName(resourceType.ID).Substring(0, 1).ToUpper() + Tools.NormaliseTypeName(resourceType.ID).Substring(1)));
                                }
                            }
                        }
                    }
                }

                // Generate the model-specific tools.
                using (StreamWriter toolsFile = new StreamWriter(modelSpecificTools))
                {
                    Tools.AddCopyrightNotice(toolsFile);

                    toolsFile.WriteLine("//------------------------------------------------------------------------------");
                    toolsFile.WriteLine("// <auto-generated>");
                    toolsFile.WriteLine("//     This code is auto-generated by the policy engine when the system model");
                    toolsFile.WriteLine("//     changes.");
                    toolsFile.WriteLine("//");
                    toolsFile.WriteLine("//     Changes to this file may cause incorrect behavior and will be lost if");
                    toolsFile.WriteLine("//     the code is regenerated.");
                    toolsFile.WriteLine("// </auto-generated>");
                    toolsFile.WriteLine("//------------------------------------------------------------------------------");
                    toolsFile.WriteLine("");

                    toolsFile.WriteLine("public class ModelSpecificTools {");
                    toolsFile.WriteLine("  static public System.Type GetResourceType(string resourceId) {");
                    foreach (resource resourceType in resource.system.resource)
                    {
                        toolsFile.WriteLine("    if (resourceId.Equals(\"" + resourceType.ID + "\")) {");
                        toolsFile.WriteLine("      " + resourceType.ID + "Proxy." + resourceType.ID + " r = new " + resourceType.ID + "Proxy." + resourceType.ID + "();");
                        toolsFile.WriteLine("      return r.GetType();");
                        toolsFile.WriteLine("    }");
                    }
                    toolsFile.WriteLine("    return null;");
                    toolsFile.WriteLine("  }");
                    toolsFile.WriteLine("  static public object[] GetResourceArray(string resourceId, System.Collections.Generic.List<object> resourceList) {");
                    foreach (resource resourceType in resource.system.resource)
                    {
                        toolsFile.WriteLine("    if (resourceId.Equals(\"" + resourceType.ID + "\")) {");
                        toolsFile.WriteLine("      System.Collections.Generic.List<" + resourceType.ID + "Proxy." + resourceType.ID + "> typedList = new System.Collections.Generic.List<" + resourceType.ID + "Proxy." + resourceType.ID + ">();");
                        toolsFile.WriteLine("      foreach (object o in resourceList)");
                        toolsFile.WriteLine("        typedList.Add((" + resourceType.ID + "Proxy." + resourceType.ID + ")o);");
                        toolsFile.WriteLine("      return typedList.ToArray();");
                        toolsFile.WriteLine("    }");
                    }
                    toolsFile.WriteLine("    return null;");
                    toolsFile.WriteLine("  }");
                    toolsFile.WriteLine("  static public object GetServiceProxy(string resourceId) {");
                    foreach (resource resourceType in resource.system.resource)
                    {
                        toolsFile.WriteLine("    if (resourceId.Equals(\"" + resourceType.ID + "\"))");
                        toolsFile.WriteLine("      return new " + resourceType.ID + "Proxy.ManageabilityAdaptor();");
                    }
                    toolsFile.WriteLine("    return null;");
                    toolsFile.WriteLine("  }");
                    toolsFile.WriteLine("}");
                }

                // Update the system model.
                state.SystemModel = resource.system;
            }

            catch (Exception e)
            {
                result.Code = ManagedResourceOpResultCode.Failure;
                result.Message = e.Message;
            }
        }

        // Update the resource URLs if required.
        if (result.Code == ManagedResourceOpResultCode.Success && resource.resourceUrls != null)
        {
            state.ResourceUrls = resource.resourceUrls;
        }

        // Update the policy set if required.
        if (result.Code == ManagedResourceOpResultCode.Success && resource.policySet != null)
        {
            // Try to set (and parse) the new set of policies.
            try
            {
                // Update the policy engine state.
                state.PolicySet = resource.policySet;

                // Handle resource definition policies.
                Dictionary<string, List<object>> highLevelResources = Tools.HighLevelResources;
                if (highLevelResources != null)
                {
                    highLevelResources.Clear();
                }

                // Generate the policy-specific tools.
                string servicePath = System.Configuration.ConfigurationManager.AppSettings["servicePath"];
                string policySpecificTools = servicePath + "\\App_Code\\PolicySpecificTools.cs";
                using (StreamWriter toolsFile = new StreamWriter(policySpecificTools))
                {
                    Tools.AddCopyrightNotice(toolsFile);
                    toolsFile.WriteLine("//------------------------------------------------------------------------------");
                    toolsFile.WriteLine("// <auto-generated>");
                    toolsFile.WriteLine("//     This code is auto-generated by the policy engine when the \"resource");
                    toolsFile.WriteLine("//     definition\" policies are set.");
                    toolsFile.WriteLine("//");
                    toolsFile.WriteLine("//     Changes to this file may cause incorrect behavior and will be lost if");
                    toolsFile.WriteLine("//     the code is regenerated.");
                    toolsFile.WriteLine("// </auto-generated>");
                    toolsFile.WriteLine("//------------------------------------------------------------------------------");
                    toolsFile.WriteLine("");

                    toolsFile.WriteLine("public class PolicySpecificTools {");
                    toolsFile.WriteLine("  static public object GetResource(string resourceId) {");

                    foreach (autonomicComputingPolicy policy in state.PolicySet)
                    {
                        if (policy.action.StartsWith("RESDEF"))
                        {
                            HandleResourceDefinition(highLevelResources, policy, toolsFile);
                        }
                    }
                    toolsFile.WriteLine("    return null;");
                    toolsFile.WriteLine("  }");
                    toolsFile.WriteLine("}");
                } 
            }

            // Handle exceptions.
            catch (Exception e)
            {
                result.Code = ManagedResourceOpResultCode.Failure;
                result.Message = e.Message;
            }
        }

        // Save the new state if necessary.
        Tools.State.Save();

        // Done.
        return result;
    }

    private void AddSchema(resource resourceType, StreamWriter wsdlFile)
    {
        XmlNamespaceManager namespaceManager = new XmlNamespaceManager(new NameTable());
        namespaceManager.AddNamespace("xs", "http://www.w3.org/2001/XMLSchema");

        wsdlFile.WriteLine("       <s:complexType name=\"" + Tools.NormaliseTypeName(resourceType.ID) + "\">");
        wsdlFile.WriteLine("         <s:sequence>");
        foreach (property p in resourceType.property)
        {
            string propertyTypeName = null;
            foreach (XmlNode dataTypeItem in (XmlNode[])p.propertyDataType)
            {
                XmlNode elementDef;
                if ((elementDef = dataTypeItem.SelectSingleNode("@type", namespaceManager)) != null)
                {
                    propertyTypeName = elementDef.InnerText;
                    break;
                }
            }
            wsdlFile.WriteLine("           <s:element name=\"" + p.ID + "\" nillable=\"true\" type=\"s2:" + propertyTypeName + "\" />");
            if (p.modifiability == modifiability.derived)
            {
                wsdlFile.WriteLine("           <s:element name=\"" + p.ID + "Definition\" nillable=\"true\" type=\"s:string\" />");
            }
        }
        wsdlFile.WriteLine("         </s:sequence>");
        wsdlFile.WriteLine("       </s:complexType>");
        foreach (property p in resourceType.property)
        {
            foreach (XmlNode dataTypeItem in (XmlNode[])p.propertyDataType)
            {
                if (!dataTypeItem.OuterXml.StartsWith("<xs:element"))
                {
                    // wsdlFile.WriteLine(dataTypeItem.OuterXml);

                    string tmp = dataTypeItem.OuterXml;
                    string s = "";
                    int idx;
                    while ((idx = tmp.IndexOf("type=\"")) > 0)
                    {
                        s += tmp.Substring(0, idx + 6);
                        tmp = tmp.Substring(idx + 6);
                        if (!tmp.StartsWith("xs:"))
                        {
                            s += "s2:";
                        }
                    }
                    s += tmp;
                    wsdlFile.WriteLine(s);

                }
            }
        }
    }

    private void HandleResourceDefinition(
        System.Collections.Generic.Dictionary<string, System.Collections.Generic.List<object>> highLevelResources,
        autonomicComputingPolicy policy, StreamWriter toolsFile)
    {
        string servicePath = System.Configuration.ConfigurationManager.AppSettings["servicePath"];

        // Identify the parameters for the new resource type.
        string[] actionParams = Tools.ParseTokens(policy.action.Substring(7, policy.action.Length - 8), new char[] { ',' });
        string resourceId = actionParams[0];

        // Create the dictionary entry for this resource.
        highLevelResources.Add(resourceId, new System.Collections.Generic.List<object>());

        // Generate the .asmx file.
        using (StreamWriter asmxFile = new StreamWriter(servicePath + "\\" + resourceId + "ManageabilityAdaptor.asmx"))
        {
            asmxFile.WriteLine("<%@ WebService Language=\"C#\" CodeBehind=\"~/App_Code/" + resourceId + "ManageabilityAdaptor.cs\" Class=\"" + resourceId + "ManageabilityAdaptor\" %>");
        }

        // Generate the .cs file.
        using (StreamWriter csFile = new StreamWriter(servicePath + "\\App_Code\\" + resourceId + "ManageabilityAdaptor.cs"))
        {
            csFile.WriteLine("using System;");
            csFile.WriteLine("using System.Web;");
            csFile.WriteLine("using System.Collections;");
            csFile.WriteLine("using System.Web.Services;");
            csFile.WriteLine("using System.Web.Services.Protocols;");
            csFile.WriteLine("");

            csFile.WriteLine("[WebService(Namespace = \"http://www.yupe.com/\")]");
            csFile.WriteLine("[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]");
            csFile.WriteLine("public class " + resourceId + "ManageabilityAdaptor : ManagedResource<" + resourceId + ">");
            csFile.WriteLine("{");
            csFile.WriteLine("  protected override object[] GetRawResources()");
            csFile.WriteLine("  {");
            csFile.WriteLine("    System.Collections.Generic.Dictionary<string, System.Collections.Generic.List<object>> highLevelResources = ");
            csFile.WriteLine("      Tools.HighLevelResources;");
            csFile.WriteLine("    return highLevelResources[\"" + resourceId + "\"].ToArray();");
            csFile.WriteLine("  }");
            csFile.WriteLine("");

            csFile.WriteLine("  protected override object GetResourceProperty(object rawResource, string property)");
            csFile.WriteLine("  {");
            csFile.WriteLine("    " + resourceId + " r = (" + resourceId + ")rawResource;");
            csFile.WriteLine("    switch(property)");
            csFile.WriteLine("    {");

            for (int i = 1; i < actionParams.Length; i++)
            {
                string[] propertyDetails = Tools.ParseTokens(actionParams[i].Substring(1, actionParams[i].Length - 2), new char[] { ',' });

                csFile.WriteLine("      case \"" + propertyDetails[0] + "\":");
                csFile.WriteLine("        return r." + propertyDetails[0] + ";");
            }

            csFile.WriteLine("     default:");
            csFile.WriteLine("       throw new Exception(\"unknown property (\" + property + \")\");");
            csFile.WriteLine("    }");
            csFile.WriteLine("  }");
            csFile.WriteLine("");

            csFile.WriteLine("  protected override ManagedResourceOpResult SetResourceProperties(" + resourceId + " resource)");
            csFile.WriteLine("  {");
            csFile.WriteLine("    return new ManagedResourceOpResult();");
            csFile.WriteLine("  }");
            csFile.WriteLine("}");
        }

        // Generate the data type file.
        using (StreamWriter typeFile = new StreamWriter(servicePath + "\\App_Code\\" + resourceId + ".cs"))
        {
            Tools.AddCopyrightNotice(typeFile);

            typeFile.WriteLine("using System.Xml.Serialization;");
            typeFile.WriteLine("");

            typeFile.WriteLine("[System.CodeDom.Compiler.GeneratedCodeAttribute(\"xsd\", \"2.0.50727.42\")]");
            typeFile.WriteLine("[System.SerializableAttribute()]");
            typeFile.WriteLine("[System.Diagnostics.DebuggerStepThroughAttribute()]");
            typeFile.WriteLine("[System.ComponentModel.DesignerCategoryAttribute(\"code\")]");
            typeFile.WriteLine("[System.Xml.Serialization.XmlTypeAttribute(Namespace=\"http://www.rcc.com/system\")]");
            typeFile.WriteLine("[System.Xml.Serialization.XmlRootAttribute(Namespace=\"http://www.rcc.com/system\", IsNullable=false)]");
            typeFile.WriteLine("public partial class " + resourceId + " {");

            for (int i = 1; i < actionParams.Length; i++)
            {
                string[] propertyDetails = Tools.ParseTokens(actionParams[i].Substring(1, actionParams[i].Length - 2), new char[] { ',' });

                string propertyType = (Char.IsDigit(propertyDetails[1][0]) || propertyDetails[1].StartsWith("SUM")) ? "System.Nullable<double>" : "string";

                typeFile.WriteLine("  private " + propertyType + " " + propertyDetails[0] + "Field;");
                typeFile.WriteLine("");

                typeFile.WriteLine("  [System.Xml.Serialization.XmlElementAttribute(IsNullable=true)]");
                typeFile.WriteLine("  public " + propertyType + " " + propertyDetails[0] + " {");
                typeFile.WriteLine("    get {");
                typeFile.WriteLine("      return this." + propertyDetails[0] + "Field;");
                typeFile.WriteLine("    }");
                typeFile.WriteLine("    set {");
                typeFile.WriteLine("      this." + propertyDetails[0] + "Field = value;");
                typeFile.WriteLine("    }");
                typeFile.WriteLine("  }");
                if (i < actionParams.Length - 1)
                {
                    typeFile.WriteLine("");
                }
            }

            typeFile.WriteLine("}");
        }

        toolsFile.WriteLine("    if (resourceId.Equals(\"" + resourceId + "\")) {");
        toolsFile.WriteLine("      " + resourceId + " r = new " + resourceId + "();");
        toolsFile.WriteLine("      return r;");
        toolsFile.WriteLine("    }");
    }
}
