//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Data;
using System.Configuration;
using System.Text;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using System.Collections;
using System.Collections.Generic;

using log4net;
using log4net.Config;

/// <summary>
///   
/// The policy engine "processor" handles the policies specified by the system administrator.
/// 
/// </summary>
public class PolicyEngineProcessor
{
    // Logger.
    private static readonly ILog log = LogManager.GetLogger(typeof(PolicyEngineProcessor));

    // Parsed policies.
    private Policy[] policies;
    private Dictionary<string, List<string>> monitoredProperties;

    // Parsed resource URLs.
    private Dictionary<string, List<string>> parsedUrls;

    // Getter method for the parsed URLs dictionary.
    public Dictionary<string, List<string>> ParsedUrls
    {
        get
        {
            return parsedUrls;
        }
    }

    // Constructor.
    public PolicyEngineProcessor()
    {
        // Get initial system state.
        PolicyEngineState state = Tools.State;

        // Parse policies into the internal format.
        ParsePolicies(state.PolicySet);

        // Parse URLs into the internal format.
        ParseURLs(state.ResourceUrls);
    }
    
    // Method that parses policy set into the internal format.
    public void ParsePolicies(autonomicComputingPolicy[] unparsed)
    {
        // Add log entry.
        log.Debug("Parsing autonomic computing policies");

        // Initialise buffer area for parsing the new policies.
        Dictionary<string, List<string>> newMonitoredProperties = new Dictionary<string, List<string>>();
        Policy[] newPolicies;

        // Parse the new policies in the buffer area - an exception is generated if the new policy set is invalid.
        if (unparsed == null)
        {
            newPolicies = null;
        }
        else
        {
            newPolicies = new Policy[unparsed.Length];
            for (int i = 0; i < unparsed.Length; i++)
            {
                newPolicies[i] = Policy.Parse(unparsed[i], newMonitoredProperties, i + 1);
            }
        }
         
        // Use critical section to change the policies.
        if (System.Threading.Monitor.TryEnter(this, 20000))
        {
            // Change the policies.
            try
            {
                this.monitoredProperties = newMonitoredProperties;
                this.policies = newPolicies;
            }
            finally
            {
                System.Threading.Monitor.Exit(this);
            }
        }

        // Cannot acquire lock to change policies.
        else
        {
            log.Warn("Cannot acquire lock to update policies");
            throw new Exception("Autonomic manager is busy, please retry");
        }
    }

    // Method that "parses" the resource URLs into the internal format,
    // i.e., a dictionary mapping types of resources to the URLs of the
    // manageability adaptors that expose instances of these resource types.
    //
    // Note that this method is called whenever a new set of URLs is specified;
    // however, it is easy to change the policy engine to call it before every policy
    // evaluation, as would be necessary if contacting the manageability adaptors
    // were unreliable.
    //
    public void ParseURLs(string[] unparsed)
    {
        // Add log entry.
        log.Debug("Parsing resource URLs");

        // Create empty dictionary of resource types to URLs.
        this.parsedUrls = new Dictionary<string, List<string>>();

        // Check the URLs and record the resource types it each exposes.
        GenericResourceProxy.ManageabilityAdaptor svc = new GenericResourceProxy.ManageabilityAdaptor();
        for (int i = 0; i < unparsed.Length; i++)
        {
            try
            {
                // Find our the type of resource supported by the manageability adaptor.
                svc.Url = unparsed[i];
                string resourceType = svc.SupportedResource();

                // Add to dictionary (NB: This included resources whose types are not defined in the model).
                if (!this.parsedUrls.ContainsKey(resourceType))
                {
                    this.parsedUrls[resourceType] = new List<string>();
                }
                this.parsedUrls[resourceType].Add(unparsed[i]);
            }
            catch
            {
                // Ignore invalid URLs.
                ; 
            }
        }
    }

    // Policy evaluation.
    public void PolicyEngineStep(object dummy)
    {
        // Get top-level and info loggers.
        ILog log = LogManager.GetLogger("PolicyEvaluator");
        ILog infoLog = LogManager.GetLogger("PolicyEvaluator.Info");

        // Update audit-trail log.
        log.Info("POLICY EVALUATION STEP");

        // Policy evaluation critical section.
        if (System.Threading.Monitor.TryEnter(this))
        {
            infoLog.Info("Starting policy evaluation");
            try
            {
                EvaluatePolicies();
            }
            catch (Exception e)
            {
                log.Error("Error in policy evaluation: " + e.Message);
            }
            finally
            {
                System.Threading.Monitor.Exit(this);
            }
            infoLog.Info("Finished policy evaluation");
        }

        // Log skipped policy evaluation.
        else
        {
            // Add log entry.
            infoLog.Info("Skipping policy evaluation step as another step is underway");
        }

        // Update audit-trail log.
        log.Info("FINISHED POLICY EVALUATION STEP");
    }

    // Policy evaluation.
    private void EvaluatePolicies()
    {
        // Get info logger.
        ILog infoLog = LogManager.GetLogger("PolicyEvaluator.Info");

        // Obtain the resource instances associated with the current policy set.
        infoLog.Info("Obtaining resource instances associated with the current policy set from manageability adaptors");
        Dictionary<ResourceTypeUrlPair, IList> resourceLists = GetResourceInstances();
        infoLog.Info("Finished obtaining resource instances associated with the current policy set");

        // Return if the resource lists are empty.
        if (resourceLists.Keys.Count == 0)
        {
            infoLog.Info("The manageability adaptors expose no system resources.");
            return;
        }

        // The set of all resources.
        Dictionary<string, List<object>> allResources = Tools.BuildResourceDictionary(resourceLists);

        // Get policy, policy scope and policy scope detail loggers.
        ILog policyLog = LogManager.GetLogger("PolicyEvaluator.Policy");
        ILog policyScopeLog = LogManager.GetLogger("PolicyEvaluator.Policy.Scope");
        ILog policyScopeDetailsLog = LogManager.GetLogger("PolicyEvaluator.Policy.Scope.Details");
        ILog policyTriggerLog = LogManager.GetLogger("PolicyEvaluator.Policy.Trigger");
        ILog policyActionLog = LogManager.GetLogger("PolicyEvaluator.Policy.Action");

        // Handle each policy in the policy set.
        int policyIdx = 1;
        infoLog.Info("Handling policies");
        foreach (Policy policy in this.policies)
        {
            // Update policy log.
            policyLog.Info("Handling policy " + policyIdx);

            // Update policy scope log.
            policyScopeLog.Info("Building policy scope");

            // The scope for the current policy.
            Dictionary<ResourceTypeUrlPair, IList> policyScope = new Dictionary<ResourceTypeUrlPair, IList>();

            // Visit all resource instances to identify the resources in the policy scope.
            foreach (ResourceTypeUrlPair resTypeUrlPair in resourceLists.Keys)
            {
                // Update audit-trail log.
                policyScopeLog.Info("Identifying '" + resTypeUrlPair.ResourceType + 
                    "' resources in the policy scope, exposed by manageability adaptor at URL '" + resTypeUrlPair.Url + "'");

                // Examine all resource instances from the selected manageability adaptor.
                foreach (object res in resourceLists[resTypeUrlPair])
                {
                    if (res != null && policy.Scope.evalBool(res, allResources))
                    {
                        if (!policyScope.ContainsKey(resTypeUrlPair))
                        {
                            policyScope[resTypeUrlPair] =
                                (IList)Tools.CreateGenericObject("System.Collections.Generic.List", ModelSpecificTools.GetResourceType(resTypeUrlPair.ResourceType));
                        }
                        policyScope[resTypeUrlPair].Add(res);

                        // Update the audit-trail log.
                        if (policyScopeDetailsLog.IsInfoEnabled)
                        {
                            StringBuilder msg = new StringBuilder("Policy scope contains " + resTypeUrlPair.ResourceType + " with");
                            foreach (string prop in this.monitoredProperties[resTypeUrlPair.ResourceType])
                            {
                                msg.Append(" " + prop + "=" + res.GetType().GetProperty(prop).GetGetMethod().Invoke(res, new object[] { }).ToString());
                            }
                            policyScopeDetailsLog.Info(msg);
                        }
                    }
                }

                // Update audit-trail log.
                if (policyScopeLog.IsInfoEnabled)
                {
                    int resCount = (!policyScope.ContainsKey(resTypeUrlPair)) ? 0 : policyScope[resTypeUrlPair].Count;

                    policyScopeLog.Info("Finished examining '" + resTypeUrlPair.ResourceType + 
                        "' resources exposed by manageability adaptor at URL '" + resTypeUrlPair.Url + 
                        "' (found " + resCount + " resource instance(s) in scope)");
                }
            }

            // Update audit-trail log.
            policyScopeLog.Info("Finished building policy scope");

            // Skip remainder steps for this policy if the scope is empty.
            if (policyScope.Keys.Count == 0)
            {
                policyLog.Info("Finished handling policy " + policyIdx + " early because its scope is empty");
                policyIdx++;
                continue;
            }

            // The set of all resources in scope.
            Dictionary<string, List<object>> allResourcesInScope = Tools.BuildResourceDictionary(policyScope);

            // Evaluate the policy trigger.
            bool triggerValue = policy.Trigger.evalBool(null, allResourcesInScope);

            // Update the audit-trail log.
            policyTriggerLog.Info("The policy trigger is " + triggerValue.ToString());

            // Implement the policy for all the resources in the policy scope if the trigger is true.
            if (triggerValue)
            {
                // Update the audit-trail log.
                policyActionLog.Info("Starting implementation of policy action");

                // Handle action policy action.
                if (policy.Action is ActionPA)
                {
                    ActionPA action = (ActionPA)policy.Action;
                    foreach (ResourceTypeUrlPair resTypeUrlPair in policyScope.Keys)
                    {
                        // Update the audit-trail log.
                        policyActionLog.Info("Considering read-write property changes for '" + resTypeUrlPair.ResourceType +
                            "' resources exposed by manageability adaptor at URL '" + resTypeUrlPair.Url + "'");

                        // The list of resource instances whose read-write fields will be modified.
                        List<object> modifiedResources = new List<object>();

                        // Check the trigger/condition for all resources in the policy scope.
                        foreach (object res in policyScope[resTypeUrlPair])
                        {
                            // Update the audit-trail log.
                            if (policyActionLog.IsInfoEnabled)
                            {
                                StringBuilder msg = new StringBuilder("Applying policy action to " + resTypeUrlPair.ResourceType + " with");
                                foreach (string prop in this.monitoredProperties[resTypeUrlPair.ResourceType])
                                {
                                    msg.Append(" " + prop + "=" + res.GetType().GetProperty(prop).GetGetMethod().Invoke(res, new object[] { }).ToString());
                                }
                                policyActionLog.Info(msg);
                            }

                            // Implement policy action.
                            action.Implement(res, allResourcesInScope);

                            // Add resource to the list of modified resources.
                            modifiedResources.Add(res);
                        }

                        // Update resources if necessary.
                        if (modifiedResources.Count > 0)
                        {
                            // Prepare service proxy.
                            Object svc = ModelSpecificTools.GetServiceProxy(resTypeUrlPair.ResourceType);
                            System.Reflection.PropertyInfo urlProperty = svc.GetType().GetProperty("Url");
                            urlProperty.GetSetMethod().Invoke(svc, new object[] { resTypeUrlPair.Url });

                            // Execute the set method.
                            try
                            {
                                System.Reflection.MethodInfo setMethod = svc.GetType().GetMethod("SetResources");
                                object result = setMethod.Invoke(svc,
                                    new object[] { ModelSpecificTools.GetResourceArray(resTypeUrlPair.ResourceType, modifiedResources) });
                            }
                            catch (Exception e)
                            {
                                log.Warn(e.Message);
                            }
                        }

                        // Update the audit-trail log.
                        policyActionLog.Info("Finished action implementation for '" + resTypeUrlPair.ResourceType +
                            "' resources exposed by manageability adaptor at URL '" + resTypeUrlPair.Url + "'");
                    }
                }

                // Handle utility policy.
                else if (policy.Action is UtilityPA)
                {
                    // Update the audit-trail log.
                    policyActionLog.Info("Implementing " + (((UtilityPA)policy.Action).IsGlobal() ? "global" : "local") + " utility-function policy");

                    // Implementing utility-function policy.
                    UtilityPA action = (UtilityPA)policy.Action;
                    List<object> allModifiedResources = action.Implement(allResourcesInScope);

                    // Update configuration of all modified resources.
                    if (allModifiedResources != null)
                    {
                        // Extract modified resource type.
                        string modifiedResourceType = allModifiedResources[0].GetType().Name;

                        // Prepare service proxy.
                        Object svc = ModelSpecificTools.GetServiceProxy(modifiedResourceType);
                        System.Reflection.PropertyInfo urlProperty = svc.GetType().GetProperty("Url");

                        // Examine the manageability adaptors of resources of this type.
                        foreach (ResourceTypeUrlPair resTypeUrlPair in policyScope.Keys)
                        {
                            if (resTypeUrlPair.ResourceType.Equals(modifiedResourceType))
                            {
                                // Build list of modified resources exposed by this manageability adaptor.
                                List<object> modifiedResources = new List<object>();

                                foreach (object modifiedResource in allModifiedResources)
                                {
                                    foreach (object resource in policyScope[resTypeUrlPair])
                                    {
                                        if (modifiedResource == resource)
                                        {
                                            modifiedResources.Add(modifiedResource);
                                            break;
                                        }
                                    }
                                }

                                // Implement changes if any.
                                if (modifiedResources.Count > 0)
                                {
                                    urlProperty.GetSetMethod().Invoke(svc, new object[] { resTypeUrlPair.Url });

                                    // Execute the set method.
                                    try
                                    {
                                        System.Reflection.MethodInfo setMethod = svc.GetType().GetMethod("SetResources");
                                        object result = setMethod.Invoke(svc,
                                            new object[] { ModelSpecificTools.GetResourceArray(resTypeUrlPair.ResourceType, modifiedResources) });
                                    }
                                    catch (Exception e)
                                    {
                                        log.Warn(e.Message);
                                    }
                                }
                            }
                        }
                    }

                    // Update the audit-trail log.
                    policyActionLog.Info("Finished implementing utility function policy");
                }

                // Update the audit-trail log.
                policyActionLog.Info("Finished implementation of policy action");
            }

            // Update the audit-trail log.
            else
            {
                policyActionLog.Info("Policy action not required because trigger is false");
            }

            // Update the audit-trail log.
            policyLog.Info("Finished handling policy " + policyIdx);
            policyIdx++;
        }

        // Update log.
        infoLog.Info("Finished handling policies");
    }

    // Get all instances of resources involved in the current autonomic computing policies,
    // together with their relevant properties. 
    private Dictionary<ResourceTypeUrlPair, IList> GetResourceInstances()
    {
        // Initialise log.
        ILog log = LogManager.GetLogger("PolicyEvaluator.Resources");
        ILog detailsLog = LogManager.GetLogger("PolicyEvaluator.Resources.Details");

        // Initialise the dictionary that will map relevant (resourceType, manageabilityAdaptorUrl)
        // pair to its list of associated resource instances.
        Dictionary<ResourceTypeUrlPair, IList> resourceLists = new Dictionary<ResourceTypeUrlPair, IList>();

        // Examine all resource types from the monitoredProperties dictionary.
        foreach (string resourceType in this.monitoredProperties.Keys)
        {
            // Update log.
            if (log.IsInfoEnabled)
            {
                StringBuilder msg = new StringBuilder("Obtaining resources of type '" + resourceType + "', with relevant property set = {");
                bool firstItem = true;
                foreach (string prop in this.monitoredProperties[resourceType])
                {
                    if (!firstItem)
                    {
                        msg.Append(", " + prop);
                    }
                    else
                    {
                        firstItem = false;
                        msg.Append(prop);
                    }
                }
                msg.Append('}');
                log.Info(msg);
            }

            // Instantiate manageability adaptor proxy using reflection.
            Object svc = ModelSpecificTools.GetServiceProxy(resourceType);
            System.Reflection.PropertyInfo urlProperty = svc.GetType().GetProperty("Url");
            System.Reflection.MethodInfo method = svc.GetType().GetMethod("GetResources");

            // Examine all manageability adaptors that expose this type of resource.
            if (parsedUrls.ContainsKey(resourceType))
            {
                foreach (string url in parsedUrls[resourceType])
                {
                    try
                    {
                        // Update log.
                        log.Info("Querying manageability adaptor at URL '" + url + "'");

                        // Set the proxy URL.
                        urlProperty.GetSetMethod().Invoke(svc, new object[] { url });

                        // Create (resource type, url) pair.
                        ResourceTypeUrlPair resTypeUrlPair = new ResourceTypeUrlPair(resourceType, url);

                        // Add entry in the resource instance dictionary
                        resourceLists[resTypeUrlPair] =
                            (IList)Tools.CreateGenericObject("System.Collections.Generic.List", ModelSpecificTools.GetResourceType(resourceType));

                        // Call the manageability adaptor web method that returns the exposed list of resource instances.
                        object result = method.Invoke(svc, new object[] { monitoredProperties[resourceType].ToArray() });

                        // Use reflection to extract the actual resource instance list.
                        System.Reflection.PropertyInfo resourceArrayProperty = result.GetType().GetProperty("ResourceList");
                        object[] resourceList = (object[])resourceArrayProperty.GetGetMethod().Invoke(result, null);

                        // Add the obtained resource instances to the dictionary.
                        foreach (object res in resourceList)
                        {
                            resourceLists[resTypeUrlPair].Add(res);

                            // Update log if enabled.
                            if (detailsLog.IsInfoEnabled)
                            {
                                StringBuilder msg = new StringBuilder(resourceType + " with");
                                foreach (string prop in this.monitoredProperties[resourceType])
                                {
                                    msg.Append(" " + prop + "=" + res.GetType().GetProperty(prop).GetGetMethod().Invoke(res, new object[] { }).ToString());
                                }
                                detailsLog.Info(msg);
                            }
                        }

                        // Update log.
                        log.Info("Finished querying manageability adaptor at URL '" + url + "' (found " + resourceList.Length + " resource instance(s))");
                    }

                    // Ignore exceptions since temporarily inaccessible manageability adaptors are expected.
                    catch
                    {
                        ;
                    }
                }
            }

            else
            {
                // Update log.
                log.Info("No manageability adaptor for resources of type '" + resourceType + "'");
            }

            // Update log.
            log.Info("Finished obtaining resources of type '" + resourceType + "'");
        }

        // Return the resource instances.
        return resourceLists;
    }
}
