//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Configuration;

namespace DiskDriveSimulator
{
    public partial class DiskDriverSimulatorForm : Form
    {
        struct Plot
        {
            public Color c;
            public int[] y;
            public int head;          // Index of the oldest value in the graph.
            public int tail;          // Index of the next value to write: head == tail => empty graph.
        }

        class Graph
        {
            string xAxisKey;
            string xAxisUnit;
            string yAxisKey;
            string yAxisUnit;
            int minX;
            int maxX;
            int minY;
            int maxY;
            Plot[] plots;

            public Graph(Color[] colours, string xAxisKey, string xAxisUnit, string yAxisKey, string yAxisUnit,
                int minX, int maxX, int minY, int maxY, int nSamples)
            {
                this.xAxisKey = xAxisKey;
                this.xAxisUnit = xAxisUnit;
                this.yAxisKey = yAxisKey;
                this.yAxisUnit = yAxisUnit;
                this.minX = minX;
                this.maxX = maxX;
                this.minY = minY;
                this.maxY = maxY;
                plots = new Plot[colours.Length];
                for (int i = 0; i < colours.Length; i++)
                {
                    plots[i].c = colours[i];
                    plots[i].y = new int[nSamples];
                    plots[i].head = plots[i].tail = 0;
                }
            }

            /* public Color Color { get { return this.c; } }

            public int[] Y { get { return this.y; } }

            public int Head { get { return this.head; } set { this.head = value; } }

            public int Tail { get { return this.tail; } set { this.tail = value; } } */

            public Plot[] Plots { get {return this.plots; } }

            public string XAxisKey { get { return this.xAxisKey; } }

            public string XAxisUnit { get { return this.xAxisUnit; } }

            public string YAxisKey { get { return this.yAxisKey; } }

            public string YAxisUnit { get { return this.yAxisUnit; } }

            public int MinX { get { return this.minX; } }

            public int MaxX { get { return this.maxX; } }

            public int MinY { get { return this.minY; } }

            public int MaxY { get { return this.maxY; } }
        }

        // Simulation parameters.
        int timeoutMs;
        int activationN;
        int utilityFunctionIndex;

        Graph[] graphs;
        string key;

        AppServerSimulator.ManageabilityAdaptor.diskDriveManageabilityAdaptor configService;

        public DiskDriverSimulatorForm()
        {
            this.DoubleBuffered = true;
            InitializeComponent();
            comboBox1.Text = "1000";
            comboBox2.Text = "20";
            comboBox3.Text = this.displayedUtilityFunction;
            radioButton4.Checked = true;
            graphs = null;
            configService = new AppServerSimulator.ManageabilityAdaptor.diskDriveManageabilityAdaptor();
        }

        private void AppServerSimulatorForm_Paint(object sender, PaintEventArgs e)
        {
            // References to object we will use
            Graphics graphicsObject = e.Graphics;

            // Clear the window.
            // graphicsObject.Clear(Color.FloralWhite);

            // Check if there is work to do.
            if (graphs == null)
            {
                return;
            }

            // Define pens.
            Pen axisPen = new Pen(Color.Black, 2);
            Pen simplePen = new Pen(Color.Black, 1);

            // Define brushes.
            Brush axisBrush = Brushes.Black;
            Brush keyBrush = Brushes.Black;

            // Define fonts.
            Font axisFont = new Font(FontFamily.GenericSansSerif, 8);
            Font keyFont = new Font(FontFamily.GenericSerif, 18);

            // Keep track of y offset.
            int yOffset = 10;

            // Define left margin.
            int leftMargin = 30;

            // Calculate graph height.
            int graphHeight = (ClientSize.Height - 40) / graphs.Length;
            graphicsObject.DrawString(key, keyFont, keyBrush, 25, ClientSize.Height - 25);

            foreach (Graph graph in graphs)
            {
                // Draw x axis.
                int xAxisPosition = yOffset + graphHeight - 10;
                graphicsObject.DrawLine(axisPen, leftMargin, xAxisPosition, ClientSize.Width - 10, xAxisPosition);
                graphicsObject.DrawString(graph.XAxisKey + " [" + graph.XAxisUnit + "]", axisFont, axisBrush, ClientSize.Width - 50, xAxisPosition);
                if (graph.MinX != graph.MinY)
                {
                    graphicsObject.DrawString(graph.MinX.ToString(), axisFont, axisBrush, 25, xAxisPosition + 5);
                }
                graphicsObject.DrawLine(simplePen, /*ClientSize.Width - 50*/leftMargin + graph.Plots[0].y.Length, xAxisPosition - 5, /*ClientSize.Width - 50*/leftMargin + graph.Plots[0].y.Length, xAxisPosition + 5);
                graphicsObject.DrawString(graph.MaxX.ToString(), axisFont, axisBrush, leftMargin + graph.Plots[0].y.Length, xAxisPosition - 20);

                // Draw y axis.
                graphicsObject.DrawLine(axisPen, leftMargin, yOffset + 10, leftMargin, xAxisPosition);
                graphicsObject.DrawString(graph.YAxisKey + (graph.YAxisUnit.Equals("") ? "" : " [" + graph.YAxisUnit + "]"), axisFont, axisBrush, leftMargin, yOffset);
                graphicsObject.DrawLine(simplePen, leftMargin - 5, yOffset + 20, leftMargin + 5, yOffset + 20);
                graphicsObject.DrawString(graph.MinY.ToString(), axisFont, axisBrush, 15, xAxisPosition - 5);
                graphicsObject.DrawString(graph.MaxY.ToString(), axisFont, axisBrush, 0, yOffset + 15);

                // Draw the graph.
                foreach (Plot p in graph.Plots)
                {
                    // Create the graph pen.
                    Pen graphPen = new Pen(p.c, 1);

                    int nextI = 0;
                    int x = leftMargin;
                    for (int i = p.head; i != p.tail; i = nextI)
                    {
                        nextI = (i + 1) % p.y.Length;
                        if (nextI != p.tail)
                        {
                            if (p.y[i] >= graph.MinY && p.y[i] <= graph.MaxY &&
                                p.y[nextI] >= graph.MinY && p.y[nextI] <= graph.MaxY)
                            {
                                graphicsObject.DrawLine(graphPen,
                                    x, ScaleValue(p.y[i], graph.MinY, graph.MaxY, yOffset + 20, xAxisPosition),
                                    ++x, ScaleValue(p.y[nextI], graph.MinY, graph.MaxY, yOffset + 20, xAxisPosition));
                            }
                            else
                            {
                                x++;
                            }
                        }
                    }
                }

                // Update y offset.
                yOffset += graphHeight;
            }
        }

        private int ScaleValue(int y, int minY, int maxY, int yMinOffset, int yMaxOffset)
        {
            return yMaxOffset - ((y - minY) * (yMaxOffset - yMinOffset)) / (maxY - minY);
        }

        struct ClientRequest
        {
            public Int64 time; // simulation time when the client request is added to the request queue, in microseconds
            public int cpu;  // amount of CPU time required to handle client request, in microseconds

            public ClientRequest(Int64 time, int cpu) { this.time = time; this.cpu = cpu; }
        }

        // Memoryless, exponential distribution random number generator.
        private double exponentialRnd(Random rnd, double rate)
        {
            double ranf = (double)rnd.Next(1000000) / 1000000.0;
            return (- Math.Log(1.0 - ranf) / rate);
        }

        // Gaussian random number generator (uses the polar form of the Box-Muller Transformation).
        // Box, G.E.P, M.E. Muller 1958; A note on the generation of random normal deviates, Annals Math. Stat, V. 29, pp. 610-611.
        private int gaussianRnd(Random rnd, int mean, int stdDev)
        {
            double ranf, x1, x2, w, y1;

	        do {
                ranf = (double)rnd.Next(1000000) / 1000000.0;
		        x1 = 2.0 * ranf - 1.0;
                ranf = (double)rnd.Next(1000000) / 1000000.0;
		        x2 = 2.0 * ranf - 1.0;
		        w = x1 * x1 + x2 * x2;
	        } while ( w >= 1.0 );

	        w = Math.Sqrt( (-2.0 * Math.Log(w) ) / w );
	        y1 = x1 * w;

            int result = (int)(mean + y1 * stdDev);
            if (result < 0) 
                result = mean;

	        return result;
        }

        private void AppServerSimulatorForm_Load(object sender, EventArgs e)
        {

        }

        private void label2_Click(object sender, EventArgs e)
        {

        }

        private void button3_Click(object sender, EventArgs e)
        {

        }

        enum State {
            Sleep,
            Busy,
            Idle
        }

        // Experiment duration (milliseconds).
        Int64 T = 3600000;

        // Service rate.
        double serviceRate = 1.0/8;

        // Request queue size.
        int requestQueueSize = 20;

        // Power usage figures.
        Dictionary<State, double> powerUse;
        Dictionary<string, double> transitionEnergy;

        //
        // Start simulation.
        //
        private void button1_Click_1(object sender, EventArgs e)
        {
            // Hide initial controls.
            groupBox1.Visible = false;
            groupBox2.Visible = false;
            button1.Visible = false;
            Refresh();

            // Read simulation parameters.
            timeoutMs = Int32.Parse(comboBox1.Text);
            activationN = Int32.Parse(comboBox2.Text);
            utilityFunctionIndex = 0; // only one, hardcoded option is supported

            // Create the graphs.
            graphs = new Graph[(radioButton3.Checked || radioButton4.Checked) ? 5 : 4];
            if (!radioButton4.Checked)
            {
                graphs[0] = new Graph(new Color[] { Color.Black }, "time", "s", "Request average inter-arrival time", "ms", 0, 3600, 0, 2000, 600);
                graphs[1] = new Graph(new Color[] { Color.Blue }, "time", "s", "Average request queue length", "", 0, 3600, 0, 20, 600);
                graphs[2] = new Graph(new Color[] { Color.Green }, "time", "s", "Average power", "mW", 0, 3600, 0, 2500, 600);
                graphs[3] = new Graph(new Color[] { Color.Red }, "time", "s", "Utility", "", 0, 3600, 0, 200, 600);
            }
            else
            {
                Color[] colours = new Color[] { Color.Red, Color.Green, Color.Blue };

                graphs[0] = new Graph(new Color[] { Color.Black, Color.Black, Color.Black }, "time", "s", "Request average inter-arrival time", "ms", 0, 3600, 0, 2000, 600);
                graphs[1] = new Graph(colours, "time", "s", "Average request queue length", "", 0, 3600, 0, 20, 600);
                graphs[2] = new Graph(colours, "time", "s", "Average power", "mW", 0, 3600, 0, 2500, 600);
                graphs[3] = new Graph(colours, "time", "s", "Utility", "", 0, 3600, 0, 200, 600);
            }
            if (radioButton3.Checked || radioButton4.Checked)
            {
                graphs[4] = new Graph(new Color[] { Color.Blue }, "time", "s", "switch-to-sleep-probability * 100", "", 0, 3600, 0, 75, 600);
            }

            // Power usage.
            powerUse = new Dictionary<State, double>();
            powerUse[State.Busy] = 2.15;
            powerUse[State.Idle] = 0.95;
            powerUse[State.Sleep] = 0.13;
            transitionEnergy = new Dictionary<string, double>();
            transitionEnergy["SleepToIdle"] = 7000;
            transitionEnergy["IdleToSleep"] = 67;

            // Simulate.
            if (radioButton1.Checked)
            {
                key = "TimeOut policy";
                Simulate(Policy.TimeOut);
            }
            else if (radioButton2.Checked)
            {
                key = "Activation policy";
                Simulate(Policy.Activation);
            }
            else if (radioButton3.Checked)
            {
                key = "Adaptive policy";
                Simulate(Policy.Adaptive);
            }
            else // radioButton4.Checked
            {
                key = "All policies: RED=TimeOut; GREEN=Activation; BLUE=Adaptive";
                Simulate(Policy.TimeOut);
                Simulate(Policy.Activation);
                Simulate(Policy.Adaptive);
            }
        }

        enum Policy {
            TimeOut,
            Activation,
            Adaptive
        }

        enum SimulatedEvent
        {
            None,
            RequestArrival,
            RequestService,
            RequestRateChange,
            DriverStateChange,
            GraphUpdate,
            IdleTimeout,
            AverageCalculation
        };

        // Discrete-event simulation with four event queues:
        // - request arrival;
        // - request service;
        // - changes to request inter-arrival rate;
        // - changes to disk drive state.
        private void Simulate (Policy policy) {
            
            // Read the inter-arrival time intervals.
            List<int> timestamps = new List<int>();
            List<int> interArrivals = new List<int>();

            using (System.IO.StreamReader sr = new System.IO.StreamReader(ConfigurationSettings.AppSettings["ScenarioFile"]))
            {
                while (!sr.EndOfStream)
                {
                    string[] vals = sr.ReadLine().Split(' ');
                    timestamps.Add(Int32.Parse(vals[0]));
                    interArrivals.Add(Int32.Parse(vals[1]));
                }
            }

            // Next event timestamps;
            System.Collections.Generic.Dictionary<SimulatedEvent, Int64> events = 
                new Dictionary<SimulatedEvent, long>();

            // Random number generators for the simulation.
            Random seedRnd = new Random(32572);
            Random requestRnd = new Random(seedRnd.Next());
            Random serviceRnd = new Random(seedRnd.Next());
            Random adaptiveRnd = new Random(seedRnd.Next());
            Random toSleepRnd = new Random(seedRnd.Next());
            Random toIdleRnd = new Random(seedRnd.Next());

            // Simulation time (milliseconds).
            Int64 t = 0;
            
            // Request inter-arrival rate, and timestamp of the next change.
            double requestRate = 1.0 / interArrivals[0];
            timestamps.RemoveAt(0);
            interArrivals.RemoveAt(0);
            if (timestamps.Count > 0)
            {
                events[SimulatedEvent.RequestRateChange] = 1000 * timestamps[0];
            }

            // Timestamp of next request joining the queue (milliseconds).
            events[SimulatedEvent.RequestArrival] = (Int64) exponentialRnd(requestRnd, requestRate);

            // Request queue, initially empty.
            System.Collections.Generic.Queue<Int64> requestQueue = new Queue<long>();

            // Disk drive state, initially idle.
            State state = State.Idle;

            // Switch to sleep probability, for the adaptive policy.
            double switchToSleepProbability = 0.6;

            // Plan graph updates (every 6s).
            events[SimulatedEvent.GraphUpdate] = 6000;

            // Plan average calculation (every 120s).
            events[SimulatedEvent.AverageCalculation] = 120000;

            // Values for the other graphs.
            double queueLenAvg = -1.0;
            Int64 queueLenAux = 0;
            int powerAvg = -1;
            double energy = 0;

            // Simulation loop.
            while (t < T)
            {
                // Handle Windows form events.
                Application.DoEvents();

                // Remember event type.
                SimulatedEvent e = GetNext(events);

                // Queue length calculation.
                queueLenAux += requestQueue.Count * (events[e] - t);

                // Energy calculation.
                energy += powerUse[state] * (events[e] - t);

                // Progress simulation time.
                t = events[e];

                // Handle the event appropriately.
                switch (e) {

                    // Handle request arrival.
                    case SimulatedEvent.RequestArrival:

                        // Add new request to queue.
                        if (requestQueue.Count < requestQueueSize)
                        {
                            requestQueue.Enqueue(t);
                        }

                        // Get timestamp of next request.
                        events[e] = t + (Int64)exponentialRnd(requestRnd, requestRate);

                        // New queue size may trigger state change.
                        // Changes from idle (but not going to sleep) to busy are instantaneous.
                        if (state == State.Idle && !events.ContainsKey(SimulatedEvent.DriverStateChange)) 
                        {
                            state = State.Busy;

                            // Remove any pending timeout.
                            if (policy == Policy.TimeOut && events.ContainsKey(SimulatedEvent.DriverStateChange))
                            {
                                events.Remove(SimulatedEvent.DriverStateChange);
                            }
                        }

                        // A change from sleep to idle may be triggered.
                        else if (state == State.Sleep &&
                            (policy == Policy.TimeOut ||
                            (policy == Policy.Activation && requestQueue.Count == activationN) ||
                            (policy == Policy.Adaptive && requestQueue.Count >= requestQueueSize/2))) {
                            events[SimulatedEvent.DriverStateChange] = 
                                t + (Int64)exponentialRnd(toIdleRnd, 1.0/1600);
                        }
                   
                        break;

                    // Handle request service.
                    case SimulatedEvent.RequestService:
                        // Remove first request from the queue.
                        requestQueue.Dequeue();

                        // Remove event.
                        events.Remove(e);

                        // State changes instantaneously to idle if the queue is empty.
                        if (requestQueue.Count == 0)
                        {
                            state = State.Idle;

                            // This may further trigger a change towards sleep...
                            if (policy == Policy.TimeOut) {
                                events[SimulatedEvent.IdleTimeout] = t + timeoutMs;
                            }

                            // ...or to sleep.
                            else if (policy == Policy.Activation ||
                                (policy == Policy.Adaptive && ((adaptiveRnd.Next() % 10000) < 10000 * switchToSleepProbability)))
                            {
                                events[SimulatedEvent.DriverStateChange] = t + (Int64)exponentialRnd(toSleepRnd, 1.0/670);
                            }
                        }

                        break;

                    // Handle TimeOut policy transition to sleep.
                    case SimulatedEvent.IdleTimeout:
                        // Going to sleep.
                        events[SimulatedEvent.DriverStateChange] = t + (Int64)exponentialRnd(toSleepRnd, 1.0 / 670);

                        // Remove event.
                        events.Remove(e);

                        break;

                    case SimulatedEvent.DriverStateChange:
                        // Change the disk drive state.
                        state = (state == State.Idle) ? State.Sleep : ((requestQueue.Count == 0) ? State.Idle : State.Busy);

                        // Count the transition energy.
                        energy += (state == State.Sleep) ? transitionEnergy["IdleToSleep"] : transitionEnergy["SleepToIdle"];

                        // Remove event.
                        events.Remove(e);

                        break;

                    case SimulatedEvent.RequestRateChange:
                        // Update the request inter-arrival rate.
                        requestRate = 1.0 / interArrivals[0];
                        if (policy == Policy.Adaptive)
                        {
                            try
                            {
                                switchToSleepProbability = configService.GetConfig((int)(1.0 / requestRate));
                            }
                            catch { }
                        }

                        // Move to the next element in the list.
                        timestamps.RemoveAt(0);
                        interArrivals.RemoveAt(0);
                        if (timestamps.Count > 0)
                        {
                            events[SimulatedEvent.RequestRateChange] = 1000 * timestamps[0];
                        }
                        else
                        {
                            events.Remove(SimulatedEvent.RequestRateChange);
                        }

                        // New request.
                        events[SimulatedEvent.RequestArrival] = t + (Int64)exponentialRnd(requestRnd, requestRate);

                        break;

                    case SimulatedEvent.GraphUpdate:
                        events[e] += 6000;

                        int index = radioButton4.Checked ? (int)policy : 0;
                        int tail = graphs[0].Plots[index].tail;
                        int head = graphs[0].Plots[index].head;
                        int size = graphs[0].Plots[index].y.Length;

                        graphs[0].Plots[index].y[tail] = (int)(1.0 / requestRate);
                        graphs[1].Plots[index].y[tail] = (int) queueLenAvg;
                        graphs[2].Plots[index].y[tail] = powerAvg;
                        graphs[3].Plots[index].y[tail] = Utility(queueLenAvg, powerAvg);
                        if (policy == Policy.Adaptive)
                        {
                            graphs[4].Plots[0].y[tail] = (int)(100.0 * switchToSleepProbability);
                        }

                        tail = (tail + 1) % size;
                        graphs[0].Plots[index].tail = tail;
                        graphs[1].Plots[index].tail = tail;
                        graphs[2].Plots[index].tail = tail;
                        graphs[3].Plots[index].tail = tail;
                        if (policy == Policy.Adaptive)
                        {
                            graphs[4].Plots[0].tail = tail;
                        }

                        if (head == tail)
                        {
                            head = (head + 1) % size;
                            graphs[0].Plots[index].head = head;
                            graphs[1].Plots[index].head = head;
                            graphs[2].Plots[index].head = head;
                            graphs[3].Plots[index].head = head;
                            if (policy == Policy.Adaptive)
                            {
                                graphs[4].Plots[0].head = head;
                            }
                        }
                        if ((t % 60000) == 0)
                        {
                            Refresh();
                        }

                        if (policy == Policy.Adaptive)
                        {
                            try
                            {
                                switchToSleepProbability = configService.GetConfig((int)(1.0 / requestRate));
                            }
                            catch
                            {
                            }
                            System.Threading.Thread.Sleep(200);
                        }
                        else
                        {
                            System.Threading.Thread.Sleep(5);
                        }

                        break;

                    case SimulatedEvent.AverageCalculation:
                        queueLenAvg = queueLenAux / 120000.0;
                        queueLenAux = 0;
                        powerAvg = (int)(energy / 120);
                        energy = 0;
                        events[e] += 120000;
                        break;
                }

                // Schedule the next service if the disk driver is busy.
                if (state == State.Busy) 
                {
                    events[SimulatedEvent.RequestService] = t + (Int64)exponentialRnd(serviceRnd, serviceRate);
                }
            }
        }

        //
        // CALCULATION OF UTILITY FUNCTION: CHANGE IN LINE WITH POLICY SUPPLIED TO THE AUTONOMIC MANAGER
        // (NB: only values between 0 and 200 will be displayed, so please scale to this range.)
        //
        private string displayedUtilityFunction = "100*min(1,max(0,9-queueLength))+100*min(1,max(0,1.2-power))";
        private int Utility(double queueLenAvg, int powerAvg)
        {
            switch (utilityFunctionIndex)
            {

                case 0: 
                    double utility = -1.0;
                    if (queueLenAvg != -1 && powerAvg != -1)
                    {
                        double powerInWatt = ((double)powerAvg)/1000.0;

                        utility = 
                            100.0 * System.Math.Min(1.0, System.Math.Max(0.0, (9.0 - queueLenAvg) / 1.0)) +
                            100.0 * System.Math.Min(1.0, System.Math.Max(0.0, (1.2 - powerInWatt) / 1.0));
                    }
                    return (int)utility;

                default:
                    return -1;
            }
        }

        // Return the type of the next event.
        private SimulatedEvent GetNext(System.Collections.Generic.Dictionary<SimulatedEvent, Int64> events)
        {
            Int64 next = T + 1;
            SimulatedEvent sNext = SimulatedEvent.None;

            foreach (SimulatedEvent s in events.Keys)
            {
                if (events[s] < next)
                {
                    next = events[s];
                    sNext = s;
                }
            }

            return sNext;
        }
    
    }
}