//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Data;
using System.Collections;
using System.Collections.Generic;
using System.Configuration;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using System.Web.Services;
using System.Web.Services.Protocols;

[System.SerializableAttribute()]
[System.Xml.Serialization.XmlTypeAttribute(Namespace = "http://www.yupe.com/schemas/service")]
public enum ManagedResourceOpResultCode
{
    /// Operation completed successfully.
    Success,

    /// Operation failed.
    Failure,
}

public class ManagedResourceOpResult 
{
    private ManagedResourceOpResultCode codeField;

    private string messageField;

    /// <remarks/>
    public ManagedResourceOpResultCode Code
    {
        get
        {
            return this.codeField;
        }
        set
        {
            this.codeField = value;
        }
    }

    /// <remarks/>
    public string Message
    {
        get
        {
            return this.messageField;
        }
        set
        {
            this.messageField = value;
        }
    }

    public ManagedResourceOpResult() { this.Code = ManagedResourceOpResultCode.Success; this.Message = "Done"; }
}

public class ManagedResourceRetrieveAllOpResult<T> : ManagedResourceOpResult
{
    private T[] resourceListField;

    /// <remarks/>
    public T[] ResourceList
    {
        get
        {
            return this.resourceListField;
        }
        set
        {
            this.resourceListField = value;
        }
    }

    public ManagedResourceRetrieveAllOpResult() : base() { this.resourceListField = null; }
}

/// <summary>
/// Summary description for ManagedResource
/// </summary>
[WebService(Namespace = "http://www.yupe.com/")]
[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
abstract public class ManagedResource<T> : System.Web.Services.WebService where T : new()
{
    // Empty constructor.
	public ManagedResource() {}

    // Returns the type of resource supported by the manageability adaptor.
    [WebMethod]
    public string SupportedResource()
    {
        return typeof(T).ToString();
    }

    // Resource-independent method that returns the set of all resources.
    [WebMethod]
    public ManagedResourceRetrieveAllOpResult<T> GetResources(List<string> propertyList)
    {
        // Declare the result.
        ManagedResourceRetrieveAllOpResult<T> result = new ManagedResourceRetrieveAllOpResult<T>();

        // Ensure that exceptions are being dealt with.
        try
        {
            // Get the raw resources.
            object[] rawResources = GetRawResources();

            // Build managed resource array.
            result.ResourceList = new T[rawResources.Length];

            // Populate the resource array.
            for (int i = 0; i < rawResources.Length; i++)
            {
                // Create individual resource.
                result.ResourceList[i] = new T();

                // Use reflection to fill in the required fields.
                foreach (string property in propertyList)
                {
                    System.Reflection.PropertyInfo constraintProperty = typeof(T).GetProperty(property);
                    if (constraintProperty != null)
                    {
                        constraintProperty.GetSetMethod().Invoke(result.ResourceList[i], new object[] { GetResourceProperty(rawResources[i], property) });

                        if ((constraintProperty = typeof(T).GetProperty(property + "Specified")) != null)
                        {
                            constraintProperty.GetSetMethod().Invoke(result.ResourceList[i], new object[] { true });
                        }
                    }
                }

            }
        }

        // Handle exceptions.
        catch (Exception e)
        {
            result.Code = ManagedResourceOpResultCode.Failure;
            result.Message = e.Message;
        }

        // Done.
        return result;
    }

    // Generic method for setting the properties of a set of resources.
    [WebMethod]
    public ManagedResourceOpResult SetResources(T[] resources)
    {
        // Create the return object.
        ManagedResourceOpResult result = new ManagedResourceOpResult();

        // Set the properties of each resource.
        foreach (T resource in resources)
        {
            // Set resource properties.
            ManagedResourceOpResult localResult = SetResourceProperties(resource);

            // Check the result.
            if (localResult.Code != ManagedResourceOpResultCode.Success)
            {
                if (result.Code == ManagedResourceOpResultCode.Success)
                {
                    result.Code = ManagedResourceOpResultCode.Failure;
                    result.Message = localResult.Message;
                }
                else
                {
                    result.Message += "\n" + localResult.Message;
                }
            }
        }

        // Done.
        return result;
    }

    // Resource-specific method returning the set of all resources.
    abstract protected object[] GetRawResources();

    // Resource-specific method returning the value of a resource property.
    abstract protected object GetResourceProperty(object rawResource, string property);

    // Resource-specific method that sets the writeable properties of a resource.
    abstract protected ManagedResourceOpResult SetResourceProperties(T resource);
}
