using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.IO;
using System.Windows.Forms;

namespace AppServerSimulator
{
    public partial class AppServerSimulatorForm : Form
    {
        class Graph
        {
            Color c;
            string xAxisKey;
            string xAxisUnit;
            string yAxisKey;
            string yAxisUnit;
            int minX;
            int maxX;
            int minY;
            int maxY;
            int[] y;
            int head;          // Index of the oldest value in the graph.
            int tail;          // Index of the next value to write: head == tail => empty graph.

            public Graph(Color c, string xAxisKey, string xAxisUnit, string yAxisKey, string yAxisUnit,
                int minX, int maxX, int minY, int maxY, int nSamples)
            {
                this.c = c;
                this.xAxisKey = xAxisKey;
                this.xAxisUnit = xAxisUnit;
                this.yAxisKey = yAxisKey;
                this.yAxisUnit = yAxisUnit;
                this.minX = minX;
                this.maxX = maxX;
                this.minY = minY;
                this.maxY = maxY;
                y = new int[nSamples];
                head = tail = 0;
            }

            public Color Color { get { return this.c; } }

            public int[] Y { get { return this.y; } }

            public int Head { get { return this.head; } set { this.head = value; } }

            public int Tail { get { return this.tail; } set { this.tail = value; } }

            public string XAxisKey { get { return this.xAxisKey; } }

            public string XAxisUnit { get { return this.xAxisUnit; } }

            public string YAxisKey { get { return this.yAxisKey; } }

            public string YAxisUnit { get { return this.yAxisUnit; } }

            public int MinX { get { return this.minX; } }

            public int MaxX { get { return this.maxX; } }

            public int MinY { get { return this.minY; } }

            public int MaxY { get { return this.maxY; } }
        }

        const int Timeout = 5; // client requests timeout after 5 seconds.

        Graph[] graphs;

        ConfigurationService.serviceManageabilityAdaptor configService;

        public AppServerSimulatorForm()
        {
            InitializeComponent();

            graphs = null;
            configService = new ConfigurationService.serviceManageabilityAdaptor();
        }

        private void AppServerSimulatorForm_Paint(object sender, PaintEventArgs e)
        {
            // References to object we will use
            Graphics graphicsObject = e.Graphics;

            // Clear the window.
            // graphicsObject.Clear(Color.FloralWhite);

            // Check if there is work to do.
            if (graphs == null)
            {
                return;
            }

            // Define pens.
            Pen axisPen = new Pen(Color.Black, 2);
            Pen simplePen = new Pen(Color.Black, 1);

            // Define brushes.
            Brush axisBrush = Brushes.Black;

            // Define fonts.
            Font axisFont = new Font(FontFamily.GenericSansSerif, 8);

            // Keep track of y offset.
            int yOffset = 10;

            // Define left margin.
            int leftMargin = 30;

            // Calculate graph height.
            int graphHeight = (ClientSize.Height - 20) / graphs.Length;

            foreach (Graph graph in graphs)
            {
                // Create the graph pen.
                Pen graphPen = new Pen(graph.Color, 1);

                // Draw x axis.
                int xAxisPosition = yOffset + graphHeight - 10;
                graphicsObject.DrawLine(axisPen, leftMargin, xAxisPosition, ClientSize.Width - 10, xAxisPosition);
                graphicsObject.DrawString(graph.XAxisKey + " [" + graph.XAxisUnit + "]", axisFont, axisBrush, ClientSize.Width - 50, xAxisPosition);
                if (graph.MinX != graph.MinY)
                {
                    graphicsObject.DrawString(graph.MinX.ToString(), axisFont, axisBrush, 25, xAxisPosition + 5);
                }
                graphicsObject.DrawLine(simplePen, /*ClientSize.Width - 50*/leftMargin + graph.Y.Length, xAxisPosition - 5, /*ClientSize.Width - 50*/leftMargin + graph.Y.Length, xAxisPosition + 5);
                graphicsObject.DrawString(graph.MaxX.ToString(), axisFont, axisBrush, leftMargin + graph.Y.Length, xAxisPosition - 20);

                // Draw y axis.
                graphicsObject.DrawLine(axisPen, leftMargin, yOffset + 10, leftMargin, xAxisPosition);
                graphicsObject.DrawString(graph.YAxisKey + (graph.YAxisUnit.Equals("") ? "" : " [" + graph.YAxisUnit + "]"), axisFont, axisBrush, leftMargin, yOffset);
                graphicsObject.DrawLine(simplePen, leftMargin - 5, yOffset + 20, leftMargin + 5, yOffset + 20);
                graphicsObject.DrawString(graph.MinY.ToString(), axisFont, axisBrush, 15, xAxisPosition - 5);
                graphicsObject.DrawString(graph.MaxY.ToString(), axisFont, axisBrush, 0, yOffset + 15);

                // Draw the graph.
                int nextI = 0;
                int x = leftMargin;
                for (int i = graph.Head; i != graph.Tail; i = nextI)
                {
                    nextI = (i + 1) % graph.Y.Length;
                    if (nextI != graph.Tail)
                    {
                        graphicsObject.DrawLine(graphPen, 
                            x, ScaleValue(graph.Y[i], graph.MinY, graph.MaxY, yOffset + 20, xAxisPosition),
                            ++x, ScaleValue(graph.Y[nextI], graph.MinY, graph.MaxY, yOffset + 20, xAxisPosition));
                    }
                }

                // Update y offset.
                yOffset += graphHeight;
            }
        }

        private int ScaleValue(int y, int minY, int maxY, int yMinOffset, int yMaxOffset)
        {
            return yMaxOffset - ((y - minY) * (yMaxOffset - yMinOffset)) / (maxY - minY);
        }

        struct ClientRequest
        {
            public Int64 time; // simulation time when the client request is added to the request queue, in microseconds
            public int cpu;  // amount of CPU time required to handle client request, in microseconds

            public ClientRequest(Int64 time, int cpu) { this.time = time; this.cpu = cpu; }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            // Declare the services.
            string[] serviceNames = { "premium", "standard" };

            // Declare the mean CPU handling times and standard deviation for the request types, in microseconds (gaussian distribution is used).
            int[] requestCpuMean = { 550 }; // 1500, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400 }; // 5% of the requests require approx 1.5ms; 95% of the requests require approx 0.4ms.
            int[] requestCpuStdDev = { 100 }; // 100, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10 };

            // Create the graphs.
            graphs = new Graph[4 * serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                graphs[4 * i] = new Graph(Color.Black, "time", "s", "Request mean inter-arrival time - " + serviceNames[i] + " service", "us", 0, 600, 0, 5000, 600);
                graphs[4 * i + 1] = new Graph(Color.Blue, "time", "s", "CPU allocation - " + serviceNames[i] + " service", "%", 0, 600, 0, 100, 600);
                graphs[4 * i + 2] = new Graph(Color.Green, "time", "s", "Average response time - " + serviceNames[i] + " service", "ms", 0, 600, 0, 5000, 600);
                graphs[4 * i + 3] = new Graph(Color.Red, "time", "s", "Timed-out requests - " + serviceNames[i] + " service", "", 0, 600, 0, 1000, 600);
            }

            // Refresh the display.
            button1.Visible = false;
            button2.Visible = false;
            Refresh();

            // Declare the client request queues.
            System.Collections.Generic.Queue<ClientRequest>[] requestQueues = new Queue<ClientRequest>[serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                requestQueues[i] = new Queue<ClientRequest>();
            }

            // Declare random number generators.
            Random seedRnd = new Random(32572);
            Random[] requestRnd = new Random[serviceNames.Length];
            Random[] requestTypeRnd = new Random[serviceNames.Length];
            Random[] requestHandlerRnd = new Random[serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                requestRnd[i] = new Random(seedRnd.Next());
                requestTypeRnd[i] = new Random(seedRnd.Next());
                requestHandlerRnd[i] = new Random(seedRnd.Next());
            }

            // Declare simulation time, in microseconds.
            Int64[] simulationTime = new Int64[serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                simulationTime[i] = 0;
            }

            // Declare request rate.
            int[] requestRate = new int[serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                requestRate[i] = 5000;
            }

            // Declare CPU allocation.
            int[] cpuAllocation = new int[serviceNames.Length];
            for (int i = 0; i < serviceNames.Length; i++)
            {
                cpuAllocation[i] = configService.GetCpuAllocation(serviceNames[i]);
            }

            // Declare timed-out request counters.
            int[] timedOutRequests = new int[serviceNames.Length];

            // Declare simulation step.
            Int64 simulationStep = 1;

            // Prepare rates.
            List<int[]> reqMeanRateList = experimentRequestRates2();

            while (simulationStep < reqMeanRateList[0].Length)
            {
                // Handle events so as not to block.
                Application.DoEvents();

                // Simulate each of the services.
                for (int i = 0; i < serviceNames.Length; i++)
                {
                    // Add client requests to the queue for the current step.
                    while (simulationTime[i] < simulationStep * 1000000)
                    {
                        simulationTime[i] += exponentialRnd(requestRnd[i], requestRate[i]);

                        int reqType = requestTypeRnd[i].Next() % requestCpuMean.Length;
                        int cpu = gaussianRnd(requestHandlerRnd[i], requestCpuMean[reqType], requestCpuStdDev[reqType]);
                        requestQueues[i].Enqueue(new ClientRequest(simulationTime[i], cpu));
                    }

                    // Simulate processing.
                    Int64 t = (simulationStep - 1) * 1000000;
                    long responseTime = 0;
                    int reqCounter = 0;
                    timedOutRequests[i] = 0;
                    while (t < simulationStep * 1000000 && requestQueues[i].Count > 0 && requestQueues[i].Peek().time < simulationStep * 1000000)
                    {
                        // Dequeue the request.
                        ClientRequest req = requestQueues[i].Dequeue();

                        // Check that the request did not time out.
                        if (simulationStep > Timeout && req.time < (simulationStep - Timeout) * 1000000)
                        {
                            timedOutRequests[i]++;
                        }

                        // Handle the request.
                        else
                        {
                            int wallProcessingTime = (req.cpu * 100) / cpuAllocation[i];
                            t = (t > req.time) ? (t + wallProcessingTime) : (req.time + wallProcessingTime);
                            reqCounter++;
                            responseTime += (long)t - req.time;
                        }
                    }

                    // Calculate average response time.
                    if (reqCounter > 0)
                    {
                        responseTime /= (reqCounter * 1000);
                    }
                    else
                    {
                        responseTime = 0;
                    }

                    // Update graphs: cpuUtilisation * cpuAllocation * requestRate * timed-outRequests
                    int tail = graphs[4 * i].Tail;
                    graphs[4 * i].Y[tail] = requestRate[i];
                    graphs[4 * i + 1].Y[tail] = cpuAllocation[i];
                    graphs[4 * i + 2].Y[tail] = (int)responseTime;
                    graphs[4 * i + 3].Y[tail] = timedOutRequests[i];

                    // Move tail.
                    int nextTail = (tail + 1) % 600;
                    graphs[4 * i].Tail = graphs[4 * i + 1].Tail = graphs[4 * i + 2].Tail = graphs[4 * i + 3].Tail = nextTail;

                    // If necessary, also move head.
                    int head = graphs[4 * i].Head;
                    if (nextTail == head)
                    {
                        int nextHead = (head + 1) % 600;
                        graphs[4 * i].Head = graphs[4 * i + 1].Head = graphs[4 * i + 2].Head = graphs[4 * i + 3].Head = nextHead;
                    }

                    // Export the response time and the mean inter-arrival time.
                    configService.UpdateState(serviceNames[i], (int)responseTime, requestRate[i]);

                    // Update the request rate and cpu allocation.
                    requestRate[i] = reqMeanRateList[i][simulationStep];
                    cpuAllocation[i] = configService.GetCpuAllocation(serviceNames[i]);
                }

                // Re-paint the display.
                if ((simulationStep % 5) == 0)
                    Refresh();

                // Let other application have a share of the CPU.
                System.Threading.Thread.Sleep(500);

                // Increment simulation time.
                simulationStep++;
            }

            // Re-paint for the last time.
            Refresh();
        }

        // Memoryless, exponential distribution random number generator.
        private int exponentialRnd(Random rnd, int rate)
        {
            double ranf = (double)rnd.Next(1000000) / 1000000.0;

            int result = (int)(- (double)rate * Math.Log(1.0 - ranf));

            return result;
        }

        // Gaussian random number generator (uses the polar form of the Box-Muller Transformation).
        // Box, G.E.P, M.E. Muller 1958; A note on the generation of random normal deviates, Annals Math. Stat, V. 29, pp. 610-611.
        private int gaussianRnd(Random rnd, int mean, int stdDev)
        {
            double ranf, x1, x2, w, y1;

	        do {
                ranf = (double)rnd.Next(1000000) / 1000000.0;
		        x1 = 2.0 * ranf - 1.0;
                ranf = (double)rnd.Next(1000000) / 1000000.0;
		        x2 = 2.0 * ranf - 1.0;
		        w = x1 * x1 + x2 * x2;
	        } while ( w >= 1.0 );

	        w = Math.Sqrt( (-2.0 * Math.Log(w) ) / w );
	        y1 = x1 * w;

            int result = (int)(mean + y1 * stdDev);
            if (result < 0) 
                result = mean;

	        return result;
        }

        // Build request rates for the two services - the main experiment.
        private List<int[]> experimentRequestRates() {
            List<int[]> reqMeanRateList = new List<int[]>();
            reqMeanRateList.Add(new int[600]);
            reqMeanRateList.Add(new int[600]);

            int i;
            for (i = 0; i < 10; i++)
                reqMeanRateList[0][i] = reqMeanRateList[1][i] = 5000;

            for (; i < 100; i++)
            {
                reqMeanRateList[0][i] = 5000;
                reqMeanRateList[1][i] = reqMeanRateList[1][i - 1] - 42;
            }

            for (; i < 170; i++)
            {
                reqMeanRateList[0][i] = reqMeanRateList[0][i - 1] - 57;
                reqMeanRateList[1][i] = 900;
            }

            for (; i < 170; i++)
            {
                reqMeanRateList[0][i] = 1000;
                reqMeanRateList[1][i] = 900;
            }

            for (; i < 230; i++)
            {
                reqMeanRateList[0][i] = reqMeanRateList[0][i - 1] + 40;
                reqMeanRateList[1][i] = 900;
            }

            for (; i < 300; i++)
            {
                reqMeanRateList[0][i] = reqMeanRateList[0][i - 1] - 5;
                reqMeanRateList[1][i] = 900;
            }

            for (; i < 400; i++)
            {
                reqMeanRateList[0][i] = reqMeanRateList[0][i - 1] - 10;
                reqMeanRateList[1][i] = reqMeanRateList[1][i - 1] + 3;
            }

            for (; i < 600; i++)
            {
                reqMeanRateList[0][i] = reqMeanRateList[1][i] = 2000;
            }

            /*
            using (StreamWriter sw = new StreamWriter(@"C:\tmp\tmp.txt"))
            {
                for (int i = 0; i < reqMeanRateList[0].Length; i++)
                {
                    sw.Write(reqMeanRateList[0][i] + ", ");
                    if ((i + 1) % 10 == 0)
                    {
                        sw.WriteLine("");
                    }
                }

                sw.WriteLine("");
                sw.WriteLine("");
                sw.WriteLine("");

                for (int i = 0; i < reqMeanRateList[1].Length; i++)
                {
                    sw.Write(reqMeanRateList[1][i] + ", ");
                    if ((i + 1) % 10 == 0)
                    {
                        sw.WriteLine("");
                    }
                }
            }
            */
            return reqMeanRateList;
        }

        private List<int[]> experimentRequestRates2()
        {
            List<int[]> reqMeanRateList = new List<int[]>();
            
            reqMeanRateList.Add(new int[] { 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 
                5000, 5000, 5000, 5000, 5000, 4985, 4965, 4940, 4915, 4890,
                4865, 4835, 4800, 4760, 4710, 4658, 4601, 4544, 4487, 4430, 
                4373, 4316, 4259, 4202, 4145, 4088, 4031, 3974, 3917, 3860, 
                3803, 3746, 3689, 3632, 3575, 3518, 3461, 3404, 3347, 3290, 
                3233, 3176, 3119, 3062, 3005, 2948, 2891, 2834, 2777, 2720, 
                2663, 2606, 2549, 2492, 2435, 2378, 2321, 2264, 2207, 2150, 
                2093, 2036, 1979, 1922, 1865, 1808, 1751, 1694, 1637, 1580, 
                1523, 1466, 1409, 1320, 1200, 1135, 1080, 1030, 1010, 1010, 
                1010, 1020, 1040, 1070, 1120, 1190, 1275, 1330, 1370, 1410, 
                1450, 1490, 1530, 1570, 1610, 1650, 1690, 1730, 1770, 1810, 
                1850, 1890, 1930, 1970, 2010, 2050, 2090, 2130, 2170, 2210, 
                2250, 2290, 2330, 2370, 2410, 2450, 2490, 2530, 2570, 2610, 
                2650, 2690, 2730, 2770, 2810, 2850, 2890, 2930, 2970, 3010, 
                3050, 3090, 3130, 3170, 3210, 3250, 3290, 3320, 3345, 3365, 
                3375, 3380, 3385, 3385, 3385, 3380, 3375, 3370, 3365, 3360, 
                3355, 3350, 3345, 3340, 3335, 3330, 3325, 3320, 3315, 3310, 
                3305, 3300, 3295, 3290, 3285, 3280, 3275, 3270, 3265, 3260, 
                3255, 3250, 3245, 3240, 3235, 3230, 3225, 3220, 3215, 3210, 
                3205, 3200, 3195, 3190, 3185, 3180, 3175, 3170, 3165, 3160, 
                3155, 3150, 3145, 3140, 3135, 3130, 3125, 3120, 3115, 3110, 
                3105, 3100, 3095, 3090, 3085, 3080, 3075, 3070, 3065, 3060, 
                3050, 3040, 3030, 3020, 3010, 3000, 2990, 2980, 2970, 2960, 
                2950, 2940, 2930, 2920, 2910, 2900, 2890, 2880, 2870, 2860, 
                2850, 2840, 2830, 2820, 2810, 2800, 2790, 2780, 2770, 2760, 
                2750, 2740, 2730, 2720, 2710, 2700, 2690, 2680, 2670, 2660, 
                2650, 2640, 2630, 2620, 2610, 2600, 2590, 2580, 2570, 2560, 
                2550, 2540, 2530, 2520, 2510, 2500, 2490, 2480, 2470, 2460, 
                2450, 2440, 2430, 2420, 2410, 2400, 2390, 2380, 2370, 2360, 
                2350, 2340, 2330, 2320, 2310, 2300, 2290, 2280, 2270, 2260, 
                2250, 2240, 2230, 2220, 2210, 2200, 2190, 2180, 2170, 2160, 
                2150, 2140, 2130, 2120, 2110, 2100, 2090, 2080, 2070, 2060, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000
            });

            reqMeanRateList.Add(new int[] { 
                5000, 5000, 5000, 5000, 4990, 4980, 4965, 4950, 4935, 4915, 
                4895, 4875, 4855, 4835, 4790, 4748, 4706, 4664, 4622, 4580, 
                4538, 4496, 4454, 4412, 4370, 4328, 4286, 4244, 4202, 4160, 
                4118, 4076, 4034, 3992, 3950, 3908, 3866, 3824, 3782, 3740, 
                3698, 3656, 3614, 3572, 3530, 3488, 3446, 3404, 3362, 3320, 
                3278, 3236, 3194, 3152, 3110, 3068, 3026, 2984, 2942, 2900, 
                2858, 2816, 2774, 2732, 2690, 2648, 2606, 2564, 2522, 2480, 
                2438, 2396, 2354, 2312, 2270, 2228, 2186, 2144, 2102, 2060, 
                2018, 1976, 1934, 1892, 1850, 1808, 1766, 1724, 1682, 1640, 
                1598, 1556, 1514, 1472, 1430, 1388, 1346, 1304, 1262, 1220, 
                1185, 1152, 1120, 1095, 1075, 1056, 1038, 1020, 1005, 990, 
                977, 965, 954, 934, 925, 918, 912, 907, 903, 901, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                900, 900, 900, 900, 900, 900, 900, 900, 900, 900, 
                903, 906, 909, 912, 915, 918, 921, 924, 927, 930, 
                933, 936, 939, 942, 945, 948, 951, 954, 957, 960, 
                963, 966, 969, 972, 975, 978, 981, 984, 987, 990, 
                993, 996, 999, 1002, 1005, 1008, 1011, 1014, 1017, 1020, 
                1023, 1026, 1029, 1032, 1035, 1038, 1041, 1044, 1047, 1050, 
                1053, 1056, 1059, 1062, 1065, 1068, 1071, 1074, 1077, 1080, 
                1083, 1086, 1089, 1092, 1095, 1098, 1101, 1104, 1107, 1110, 
                1113, 1116, 1119, 1122, 1125, 1128, 1131, 1134, 1137, 1140, 
                1143, 1146, 1149, 1152, 1155, 1158, 1161, 1164, 1167, 1170, 
                1173, 1176, 1179, 1182, 1185, 1188, 1191, 1194, 1197, 1200, 
                1210, 1220, 1235, 1255, 1300, 1360, 1430, 1510, 1600, 1700, 
                1780, 1860, 1910, 1940, 1960, 1970, 1980, 1985, 1990, 1995, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 
                2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000, 2000            
            });

            return reqMeanRateList;
        }

        private void AppServerSimulatorForm_Load(object sender, EventArgs e)
        {

        }

        private void button2_Click(object sender, EventArgs e)
        {
             /*
             using (StreamWriter sw = new StreamWriter(@"C:\tmp\tmpmodel.txt")) {
                for (int requestRate = 500; requestRate <= 5000; requestRate += 100) 
                {
                    for (int cpuAllocation = 5; cpuAllocation <= 100; cpuAllocation += 5)
                    {
                        // Declare the mean CPU handling times and standard deviation for the request types, in microseconds (gaussian distribution is used).
                        int[] requestCpuMean = { 600 }; // 1500, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400, 400 }; // 5% of the requests require approx 1.5ms; 95% of the requests require approx 0.4ms.
                        int[] requestCpuStdDev = { 100 }; // 100, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10 };

                        // Declare the client request queue.
                        System.Collections.Generic.Queue<ClientRequest> requestQueue = new Queue<ClientRequest>();

                        // Declare random number generators.
                        Random seedRnd = new Random(32572);
                        Random requestRnd = new Random(seedRnd.Next());
                        Random requestTypeRnd = new Random(seedRnd.Next());
                        Random requestHandlerRnd = new Random(seedRnd.Next());

                        // Declare simulation time, in microseconds.
                        Int64 simulationTime = 0;

                        // Declare simulation step.
                        Int64 simulationStep = 1;
                        long responseTime = 0;

                        while (simulationStep < 20)
                        {
                            // Add client requests to the queue for the current step.
                            while (simulationTime < simulationStep * 1000000)
                            {
                                simulationTime += exponentialRnd(requestRnd, requestRate);

                                int reqType = requestTypeRnd.Next() % requestCpuMean.Length;
                                int cpu = gaussianRnd(requestHandlerRnd, requestCpuMean[reqType], requestCpuStdDev[reqType]);
                                requestQueue.Enqueue(new ClientRequest(simulationTime, cpu));
                            }

                            // Simulate processing.
                            Int64 t = (simulationStep - 1) * 1000000;
                            responseTime = 0;
                            int reqCounter = 0;
                            while (t < simulationStep * 1000000 && requestQueue.Count > 0 && requestQueue.Peek().time < simulationStep * 1000000)
                            {
                                // Dequeue the request.
                                ClientRequest req = requestQueue.Dequeue();

                                // Check that the request did not time out.
                                if (simulationStep > Timeout && req.time < (simulationStep - Timeout) * 1000000)
                                {
                                    ;
                                }

                                // Handle the request.
                                else
                                {
                                    int wallProcessingTime = (req.cpu * 100) / cpuAllocation;
                                    t = (t > req.time) ? (t + wallProcessingTime) : (req.time + wallProcessingTime);
                                    reqCounter++;
                                    responseTime += (long)t - req.time;
                                }
                            }

                            // Calculate average response time.
                            if (reqCounter > 0)
                            {
                                responseTime /= (reqCounter * 1000);
                            }
                            else
                            {
                                responseTime = 0;
                            }

                            // Increment simulation time.
                            simulationStep++;
                        }

                        sw.WriteLine(requestRate + ", " + cpuAllocation + ", " + responseTime);
                    }
                }
            }
            */
        }
    }
}