//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Diagnostics;

[WebService(Namespace = "http://www.yupe.com/")]
[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
public class ProcessManageabilityAdaptor : ManagedResource<process>
{
    // Supply array of "process" instances.
    protected override object[] GetRawResources()
    {
        return Process.GetProcesses();
    }

    // Provide access to the "process" properties.
    protected override object GetResourceProperty(object resource, string property)
    {
        Process p = (Process)resource;

        switch (property)
        {
            case "pid": 
                return p.Id;

            case "name":
                return p.ProcessName;

            case "serverName":
                return System.Environment.MachineName;

            case "cpuTime":
                double val = 0.0;
                try
                {
                    val = p.TotalProcessorTime.TotalMinutes;
                }
                catch
                {
                }
                return val;

            case "stop":
                return false; // The process is not stopped.

            default:
                throw new Exception("unknown property (" + property + ")");
        }
    }

    // Set the read-write resource properties.
    protected override ManagedResourceOpResult SetResourceProperties(process resource)
    {
        // Check the only read-write parameter of the process resource instance.
        if (resource.stop.HasValue)
        {
            // Check that the process belongs to this server.
            if (resource.serverName.Equals(System.Environment.MachineName))
            {
                // If the "stop" property was set to 'true', stop the process.
                if (resource.stop.Value == true)
                {
                    Process.GetProcessById(resource.pid.Value).Kill();
                }
            }
        }

        // Return 'success'.
        return new ManagedResourceOpResult();
    }
}
