//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

using log4net;


// Value types for expressions in policies.
//
public enum ExpressionValueType { STRING, BOOL, NUMERICAL, NONSCALAR };

// The base abstract class for all expressions in a policy.
//
public abstract class Expr
{
    // The type of the resource in this expressions.
    public string exprType;

    // Function to return the value type of this expression.
    abstract public ExpressionValueType ValueType();

    // Function to evaluate this expression.
    abstract public Object eval(Object o, Dictionary<string, List<object>> resources);

    // A boolean evaluation function for the case in which one is not available.
    public virtual bool evalBool(Object o, Dictionary<string, List<object>> resources)
    {
        if (this is BooleanExpr)
        {
            return ((BooleanExpr)this).evalBool(o, resources);
        }
        else
        {
            Object result = this.eval(o, resources);

            if (!(result is bool))
            {
                throw new Exception("Type mismatch in boolean expression");
            }
            return (bool)result;
        }
    }
}

// The base abstract class for all boolean expressions in a policy.
//
public abstract class BooleanExpr : Expr
{
    // Boolean expressions don't need their own generic evaluator.
    public override Object eval(Object o, Dictionary<string, List<object>> resources)
    {
        return evalBool(o, resources);
    }

    // The value type of any subclass is Boolean.
    public override ExpressionValueType ValueType()
    {
        return ExpressionValueType.BOOL;
    }

    // Boolean expressions must provide a boolean evaluator.
    abstract public override bool evalBool(Object o, Dictionary<string, List<object>> resources);
}

// The base abstract class for all arithmetic expressions is a policy.
//
public abstract class ArithmeticExpr : Expr
{
    // Just a base class to make arithmetic expressions stand out.

    // The value type of all subclasses is numerical.
    public override ExpressionValueType ValueType()
    {
        return ExpressionValueType.NUMERICAL;
    }
}

// This class represents the parsed scope of a policy. 
// Policy scopes have the form
//
//   scope-term; ...; scope-term
//
public class ScopeExpr : BooleanExpr
{
    // The list of terms in the scope expression.
    private List<SetComprehensionExpr> scopeTerms;

    // Basic constructor.
    public ScopeExpr(List<SetComprehensionExpr> scopeTerms)
    {
        this.scopeTerms = scopeTerms;
        this.exprType = "";
    }

    // Scope parser.
    static public ScopeExpr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // List for storing the terms of the policy scope.
        List<SetComprehensionExpr> terms = new List<SetComprehensionExpr>();

        // Flag that stays true while more scope terms need to be parsed.
        bool moreTerms = true;

        // Parse the scope terms.
        while (moreTerms)
        {
            // The scope term must be non-empty.
            if (tokens.Count == 0)
            {
                throw new Exception("Empty term in policy scope");
            }

            // Parse the next term.
            terms.Add(SetComprehensionExpr.Parse(tokens, monitoredProperties));

            // Check whether the scope contains more terms.
            if ((moreTerms = tokens.Count > 0 && tokens[0].Equals(";")) == true)
            {
                tokens.RemoveAt(0);
            }
        }

        // All the tokens must have been consumed.
        if (tokens.Count > 0)
        {
            throw new Exception("Garbage at the end of policy scope");
        }

        // Instantiate and return scope object.
        return new ScopeExpr(terms);
    }

    // Evaluator.
    public override bool evalBool(Object o, Dictionary<string, List<object>> resources)
    {
        foreach (SetComprehensionExpr term in this.scopeTerms)
        {
            if (term.evalBool(o, resources))
            {
                return true;
            }
        }
        return false;

        // Identify the type of resource.
        // string resourceID = (string) o.GetType().Name;
    }
}

// This class represents a term in the scope of a policy.
// Policy scope terms have the form
//
//   resource-type | bool-expr
//
// where resource-type or bool-expr (but not both) may be missing.
//
public class SetComprehensionExpr : BooleanExpr
{
    // The type of resource for the scope term.
    private string resourceID;

    // The filter used to select resource instances to which the policy should be applied.
    private Expr filter;

    // Basic constructor.
    public SetComprehensionExpr(string resourceID, Expr filter)
    {
        this.exprType = this.resourceID = resourceID;
        this.filter = filter;
    }

    // Getter method.
    public string ResourceType
    {
        get
        {
            return this.resourceID;
        }
    }

    // Scope term parser. 
    static public SetComprehensionExpr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // The ID for the resource type filtered by this policy scope term.
        string resID = null;

        // Identify the resource type to which the policy scope term refers.
        // Handle the case when the resource type is specified explicitely.
        if (tokens.Count == 1 || (tokens.Count >= 2 && (tokens[1].Equals("|") || SetComprehensionExpr.EndOfComprehensionToken(tokens[1]))))
        {
            // The first token specifies the resource type.
            resID = tokens[0];

            // Check that this is a known resource type.
            if (!Tools.State.ValidateResourceType(resID))
            {
                throw new Exception("Invalid resource type " + resID + " in policy");
            }

            // Remove the resource type.
            tokens.RemoveAt(0);

            // Check if an expression is provided.
            if (tokens.Count > 0 && tokens[0].Equals("|"))
            {
                tokens.RemoveAt(0);
            }

            // Or add Boolean constant 'true' otherwise.
            else
            {
                tokens.Insert(0, "true");
            }
        }

        // Parse the body of the term.
        Expr e = BoolExpr.Parse(tokens, monitoredProperties);

        // Initialise resource ID if required.
        if (resID == null)
        {
            // Check that the expression has a type.
            if (e.exprType.Equals(""))
            {
                throw new Exception("Boolean expression in set comprehension must have a type");
            }

            // Inherit the resource type of the expression.
            resID = e.exprType;
        }

        // Or check that the type of the boolean expression is as expected.
        else if (resID != e.exprType && !(e is BoolConstant))
        {
            throw new Exception("Invalid set comprehension with resource type " + resID +
                " and boolean expression for resource of type '" + e.exprType + "'");
        }

        // Check that e is a boolean expression, e.g., not a number.
        if (!(e is BooleanExpr))
        {
            throw new Exception("Invalid expression type in set comprehension (must be Boolean)");
        }

        // Return new ScopeTerm.
        return new SetComprehensionExpr(resID, e);
    }

    // Method that checks for the end of the set comprehension expression.
    static private bool EndOfComprehensionToken(string token)
    {
        return token.Equals(";") || token.Equals(",") || token.Equals(")");
    }

    // Filter a dictionary of resources, returning a dictionary.
    public Dictionary<string, List<object>> Filter(Dictionary<string, List<object>> resources)
    {
        Dictionary<string, List<object>> filteredResources = new Dictionary<string, List<object>>();
        filteredResources[this.resourceID] = new List<object>();

        if (resources.ContainsKey(this.resourceID))
        {
            foreach (object res in resources[this.resourceID])
            {
                if (this.evalBool(res, resources))
                {
                    filteredResources[this.resourceID].Add(res);
                }
            }
        }

        return filteredResources;
    }

    // Filter a dictionary of resources, returning a resource list.
    public List<object> Filter2(Dictionary<string, List<object>> resources)
    {
        List<object> filteredResources = new List<object>();

        if (resources.ContainsKey(this.resourceID))
        {
            foreach (object res in resources[this.resourceID])
            {
                if (this.evalBool(res, resources))
                {
                    filteredResources.Add(res);
                }
            }
        }

        return filteredResources;
    }

    // Boolean evaluator.
    public override bool evalBool(object o, Dictionary<string, List<object>> resources)
    {
        // Verify the type of resource.
        string resourceID = o.GetType().Name;
        if (!resourceID.Equals(this.resourceID))
        {
            return false;
        }

        // Apply the filter.
        return filter.evalBool(o, resources);
    }
}

// This class represents a boolean expression (i.e., a disjunction
// of boolean terms) and has the form
//
//   bool-term | bool-term | ... | bool-term
//
public class BoolExpr : BooleanExpr
{
    // List of boolean terms in this expression.
    private List<Expr> boolTerms;

    // Basic, private constructor.
    private BoolExpr(List<Expr> boolTerms)
    {
        this.boolTerms = boolTerms;

        // Calculate type of Boolean expression.
        this.exprType = "";
        foreach (Expr term in boolTerms)
        {
            // Check if the term has a different type.
            if (!term.exprType.Equals(this.exprType))
            {
                // Initialise expression type.
                if (this.exprType.Equals(""))
                {
                    this.exprType = term.exprType;
                }

                // Or throw exception if the type is already initialised to a different value.
                else
                {
                    throw new Exception("Invalid combination of resource types in Boolean expression ('" + this.exprType + "' and '" + term.exprType + "')");
                }
            }

            // Check the type of the term.
            if (term.ValueType() != ExpressionValueType.BOOL)
            {
                throw new Exception("Boolean expression contains term that is not of Boolean type in policy");
            }
        }
    }

    // Parser that handles a list of tokens. The list may contain
    // elements after the boolean expression to parse.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // List to store the terms of this conjunction.
        List<Expr> terms = new List<Expr>();

        // Flag that stays true while more scope terms need to be parsed.
        bool moreTerms = true;

        // Parse the terms.
        while (moreTerms)
        {
            // The scope term must be non-empty.
            if (tokens.Count == 0)
            {
                throw new Exception("Empty term in policy expression");
            }

            // Parse the next term.
            terms.Add(BoolTerm.Parse(tokens, monitoredProperties));

            // Check whether the scope contains more terms.
            if ((moreTerms = tokens.Count > 0 && tokens[0].Equals("|")) == true)
            {
                tokens.RemoveAt(0);
            }
        }

        // If only one term, return it; otherwise build and return an expression.
        return (terms.Count == 1) ? terms[0] : new BoolExpr(terms);
    }

    public override bool evalBool(Object o, Dictionary<string, List<object>> resources)
    {
        foreach (Expr e in this.boolTerms)
        {
            if (e.evalBool(o, resources))
            {
                return true;
            }
        }
        return false;
    }
}

// This class represents a boolean term (i.e., a conjunction
// of boolean factors) and has the form
//
//   bool-factor & bool-factor & ... & bool-factor
//
public class BoolTerm : BooleanExpr
{
    // List of boolean terms in this expression.
    private List<Expr> boolFactors;

    // Basic, private constructor.
    private BoolTerm(List<Expr> boolFactors)
    {
        this.boolFactors = boolFactors;

        // Calculate type of Boolean factor.
        this.exprType = "";
        foreach (Expr factor in boolFactors)
        {
            // Check if the term has a different type.
            if (!factor.exprType.Equals(this.exprType))
            {
                // Initialise expression type.
                if (this.exprType.Equals(""))
                {
                    this.exprType = factor.exprType;
                }

                // Or throw exception if the type is already initialised to a different value.
                else
                {
                    throw new Exception("Invalid combination of resource types in Boolean expression ('" + this.exprType + "' and '" + factor.exprType + "')");
                }
            }

            // Check the type of the factor.
            if (factor.ValueType() != ExpressionValueType.BOOL)
            {
                throw new Exception("Boolean factor contains elements that are not of Boolean type in policy");
            }
        }
    }

    // Parser that handles a list of tokens. The list may contain
    // elements after the boolean expression to parse.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // List to store the terms of this conjunction.
        List<Expr> factors = new List<Expr>();

        // Flag that stays true while more scope terms need to be parsed.
        bool moreFactors = true;

        // Parse the terms.
        while (moreFactors)
        {
            // The scope term must be non-empty.
            if (tokens.Count == 0)
            {
                throw new Exception("Empty term in policy expression");
            }

            // Parse the next term.
            factors.Add(BoolFactor.Parse(tokens, monitoredProperties));

            // Check whether the scope contains more terms.
            if ((moreFactors = tokens.Count > 0 && tokens[0].Equals("&")) == true)
            {
                tokens.RemoveAt(0);
            }
        }

        // If only one factor, return it; otherwise build and return a term.
        return (factors.Count == 1) ? factors[0] : new BoolTerm(factors);
    }

    public override bool evalBool(Object o, Dictionary<string, List<object>> resources)
    {
        foreach (Expr e in this.boolFactors)
        {
            if (!e.evalBool(o, resources))
            {
                return false;
            }
        }
        return true;
    }
}

// This class represents a boolean factor that can have one
// of the forms:
//
//   bool-constant (i.e., 'true or 'false')
//   resource-property
//   resource-property-selector
//   (bool-expr)
//   relational-expr
//
// and which can optionally be prefixed by '!' (i.e., logical negation).
//
public class BoolFactor : BooleanExpr
{
    // Flag set if the factor is negated.
    private bool negate;

    // The actual factor, before negation.
    private Expr factor;

    // Basic constructor.
    private BoolFactor(bool negate, Expr factor)
    {
        this.negate = negate;
        this.factor = factor;
        this.exprType = factor.exprType;

        // Check the type of the inner expression.
        if (factor.ValueType() != ExpressionValueType.BOOL)
        {
            throw new Exception("Boolean operator '!' applied to factor that is not of Boolean type in policy");
        }
    }

    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Flag recording whether this factor is negated.
        bool neg;

        // Actual factor.
        Expr e = null;

        // Check if the factor is negated.
        if ((neg = (tokens.Count > 0 && tokens[0].Equals("!"))) == true)
        {
            tokens.RemoveAt(0);
        }

        // Check that the token list is not empty.
        if (tokens.Count == 0)
        {
            throw new Exception("Invalid boolean expression in policy");
        }

        // Check if the factor is a boolean constant.
        if (tokens[0].ToLower().Equals("true") || tokens[0].ToLower().Equals("false"))
        {
            e = new BoolConstant(tokens[0].ToLower().Equals("true"));
            tokens.RemoveAt(0);
        }

        // Check if the factor is a resource property.
        else if (Tools.IsResourceID(tokens[0]))
        {
            // Parse the resource property.
            e = PropertyExpr.Parse(tokens, monitoredProperties, true);
        }

        // Check if the factor is a selected-resource property.
        else if (PropertySelectorExpr.IsSelectorFunction(tokens[0]))
        {
            // Parse the selector.
            e = PropertySelectorExpr.Parse(tokens, monitoredProperties);
        }

        // Check if the factor starts with a paranthesised expression.
        else if (tokens[0].Equals("("))
        {
            // Keep a copy of the token list.
            List<string> tokensBak = new List<string>(tokens);

            // Remove left paranthesis.
            tokens.RemoveAt(0);

            // Try to parse the expression as a boolean.
            try
            {
                e = BoolExpr.Parse(tokens, monitoredProperties);

                // Make sure there is a matching right paranthesis.
                if (tokens.Count == 0 || !tokens[0].Equals(")"))
                {
                    throw new Exception();
                }
                tokens.RemoveAt(0);
            }

            // Expressions that are not boolean are part of a relational 
            // expression, and are parsed as such.
            catch
            {
                // Nulify the parsed expression if any.
                e = null;

                // Restore the token list.
                tokens.Clear();
                tokens.AddRange(tokensBak);
            }
        }

        // Otherwise, the factor is a relational expression.
        if (e == null)
        {
            // Relational expressions cannot start with '!'.
            if (neg)
            {
                throw new Exception("Misplaced '!' in policy expression");
            }

            // Parse the relational expression.
            e = RelExpr.Parse(tokens, monitoredProperties);
        }

        // Record negation if applicable.
        else
        {
            // Add negation if applicable.
            if (neg)
            {
                e = new BoolFactor(neg, e);
            }

            // Check for relational operator.
            if (RelExpr.IsRelOperator(tokens))
            {
                e = RelExpr.Parse(e, tokens, monitoredProperties);
            }
        }

        // Return the expression.
        return e;
    }

    public override bool evalBool(object o, Dictionary<string, List<object>> resources)
    {
        bool result = this.factor.evalBool(o, resources);
        return this.negate ? !result : result;
    }
}

// Boolean constant: true or false.
//
public class BoolConstant : BooleanExpr
{
    // The value of the constant boolean expression.
    private bool val;

    // Basic constructor.
    public BoolConstant(bool val)
    {
        this.val = val;
        this.exprType = "";
    }

    // The evaluator returns the value of the constant.
    public override bool evalBool(object o, Dictionary<string, List<object>> resources)
    {
        return val;
    }
}

// A resource property, of the form
//
//   resourceType.propertyName
//
public class PropertyExpr : Expr
{
    // The resource type.
    private string resourceType;

    // The property name.
    private string propertyName;

    // The value type of the property.
    private ExpressionValueType propertyValueType;

    // Basic constructor.
    private PropertyExpr(string resourceType, string propertyName, bool isMonitored, Dictionary<string, List<string>> monitoredProperties)
    {
        // Validate the resource property.
        if (!Tools.IsBuiltInVariableProperty(resourceType, propertyName) && !Tools.State.ValidateResourceProperty(resourceType, propertyName))
        {
            throw new Exception("Invalid resource property '" + resourceType + "." + propertyName + "' in policy");
        }

        // Initialise fields.
        this.resourceType = resourceType;
        this.propertyName = propertyName;
        this.exprType = Tools.IsBuiltInVariable(resourceType) ? "" : resourceType;

        // Keep track of monitored resource properties.
        if (!Tools.IsBuiltInVariable(resourceType))
        {
            // Add dictionary entry if necessary.
            if (!monitoredProperties.ContainsKey(resourceType))
            {
                // Create list for the monitored properties of this resource type.
                monitoredProperties[resourceType] = new List<string>();

                // Implicitely add all "primary key" properties of the resource type.
                Tools.AddPrimaryKeyProperties(resourceType, monitoredProperties[resourceType], null);
            }

            // Add property if monitored.
            if (isMonitored && !monitoredProperties[resourceType].Contains(propertyName))
            {
                monitoredProperties[resourceType].Add(propertyName);
            }

            // Find the value type.
            Type propertyType = ModelSpecificTools.GetResourceType(this.resourceType).GetProperty(this.propertyName).PropertyType;
            if (propertyType.FullName.Contains("System.Boolean"))
            {
                this.propertyValueType = ExpressionValueType.BOOL;
            }
            else if (propertyType.FullName.Contains("System.String"))
            {
                this.propertyValueType = ExpressionValueType.STRING;
            }
            else if (propertyType.FullName.Contains("System.Double") || propertyType.FullName.Contains("System.Int") || propertyType.FullName.Contains("System.UInt"))
            {
                this.propertyValueType = ExpressionValueType.NUMERICAL;
            }
            else
            {
                this.propertyValueType = ExpressionValueType.NONSCALAR;
            }
        }

        // All built-in variables are numerical values.
        else
        {
            this.propertyValueType = ExpressionValueType.NUMERICAL;
        }
    }

    // The value type of a resource property or built-in variable.
    public override ExpressionValueType ValueType()
    {
        return this.propertyValueType;
    }

    // Return the modifiability of the property.
    public modifiability GetModifiability()
    {
        return Tools.GetModifiability(this.resourceType, this.propertyName);
    }

    // Parser.
    static public PropertyExpr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties, bool isMonitored)
    {
        // Validate the resource property (more tests to add).
        if (tokens.Count < 3 || !tokens[1].Equals("."))
        {
            throw new Exception("Invalid resource property in policy");
        }

        // Build and return resource property.
        string res = tokens[0];
        string prop = tokens[2];
        tokens.RemoveRange(0, 3);
        return new PropertyExpr(res, prop, isMonitored, monitoredProperties);
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        object val;

        // Handle build-in resource type.
        if (Tools.IsBuiltInVariable(this.resourceType))
        {
            val = Tools.EvalBuiltIn(this.resourceType, this.propertyName);
        }

        // Handle other resource types.
        else
        {
            val = o.GetType().GetProperty(this.propertyName).GetGetMethod().Invoke(o, null);
        }

        // Convert all numerical values to a double.
        if (!val.GetType().Name.ToLower().Contains("string") && !val.GetType().Name.ToLower().Contains("bool"))
        {
            double d = Double.Parse(val.ToString());
            val = d;
        }

        // Return value.
        return val;
    }

    // Setter method selector.
    private System.Reflection.MethodInfo GetSetter(object o)
    {
        // Return the property setter method.
        return o.GetType().GetProperty(this.propertyName).GetSetMethod();
    }

    // Property type getter.
    private string GetPropertyType(object o)
    {
        // Return the property setter method.
        return o.GetType().GetProperty(this.propertyName).PropertyType.FullName;
    }

    // Given an object of the right type and an instance of this property, set its value.
    public void SetProperty(object o, object propertyValue)
    {
        // Check the resource type.
        if (!o.GetType().Name.Equals(this.resourceType))
        {
            throw new Exception("Invalid resource type '" + o.GetType().Name + "' instead of '" + this.resourceType + "'");
        }

        // Set the property.
        this.GetSetter(o).Invoke(o, new object[] { Tools.Convert(propertyValue, this.GetPropertyType(o)) });
    }

    // Property name getter.
    public string PropertyName
    {
        get { return this.propertyName; }
    }

    // Resource type getter.
    public string ResourceType
    {
        get { return this.resourceType; }
    }

    // Print the resource property to a string.
    public override string ToString()
    {
        return this.resourceType + "." + this.propertyName;
    }
}

// A resource property selector of the form
//
//   ARGMIN(set-comprehension, expr)
//   ARGMAX(set-comprehension, expr)
//
public class PropertySelectorExpr : Expr
{
    // The types of property selectors supported.
    private enum SelectorType { ARGMIN, ARGMAX };

    // The property selector.
    private SelectorType selectorType;

    // The set comprehension.
    private SetComprehensionExpr scope;

    // The actual expression to minimise/maximise.
    private Expr expr;

    // The name of the property.
    private string property;

    // The value type of the property.
    private ExpressionValueType propertyValueType;

    // Basic constructor.
    private PropertySelectorExpr(SelectorType selectorType, SetComprehensionExpr scope, Expr expr, string property)
    {
        this.selectorType = selectorType;
        this.scope = scope;
        this.expr = expr;
        this.property = property;

        // Check that the scope and expression refer to the same type of resource.
        if (scope.exprType != expr.exprType)
        {
            throw new Exception("Different types for the scope ('" + scope.exprType +
                "') and expression ('" + expr.exprType + "') elements of ARGMIN/ARGMAX resource selector.");
        }

        // Check that the expression is of arithmetic type.
        if (expr.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Expression in ARGMIN/ARGMAX resource selector is not an arithmetic expression");
        }

        // Intialise expression type.
        this.exprType = scope.exprType;

        // Find the value type.
        Type propertyType = ModelSpecificTools.GetResourceType(this.exprType).GetProperty(this.property).PropertyType;
        if (propertyType.FullName.Contains("System.Boolean"))
        {
            this.propertyValueType = ExpressionValueType.BOOL;
        }
        else if (propertyType.FullName.Contains("System.String"))
        {
            this.propertyValueType = ExpressionValueType.STRING;
        }
        else if (propertyType.FullName.Contains("System.Double") || propertyType.FullName.Contains("System.Int") || propertyType.FullName.Contains("System.UInt"))
        {
            this.propertyValueType = ExpressionValueType.NUMERICAL;
        }
        else
        {
            this.propertyValueType = ExpressionValueType.NONSCALAR;
        }
    }

    // The value type of a selected resource property.
    public override ExpressionValueType ValueType()
    {
        return this.propertyValueType;
    }

    // Check selector name.
    static public bool IsSelectorFunction(string s)
    {
        return s.Equals("ARGMIN") || s.Equals("ARGMAX");
    }

    // Parser.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // The type of the selector.
        SelectorType st;

        // Parse function type.
        if (tokens.Count > 0)
        {
            switch (tokens[0])
            {
                case "ARGMIN":
                    st = SelectorType.ARGMIN;
                    break;
                case "ARGMAX":
                    st = SelectorType.ARGMAX;
                    break;
                default:
                    throw new Exception("Unsupported resource selector '" + tokens[0] + "' in policy expression");
            }
        }
        else
        {
            throw new Exception("Invalid function expression in policy");
        }

        // Check that the function name is followed by a left paranthesis.
        if (tokens.Count < 2 || !tokens[1].Equals("("))
        {
            throw new Exception("Missing paranthesis after " + tokens[0] + " in policy expression");
        }
        tokens.RemoveRange(0, 2);

        // Parse the scope.
        SetComprehensionExpr functionScope = SetComprehensionExpr.Parse(tokens, monitoredProperties);

        // Check that the scope is followed by a comma.
        if (tokens.Count == 0 || !tokens[0].Equals(","))
        {
            throw new Exception("Invalid function arguments in policy expression");
        }
        tokens.RemoveAt(0);

        // Parse the expression.
        Expr e = ArithmExpr.Parse(tokens, monitoredProperties);

        // Check that the arguments are followed by a right paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals(")"))
        {
            throw new Exception("Missing ')' after function arguments in policy expression");
        }
        tokens.RemoveAt(0);

        // Check that the selector is followed by '.'.
        if (tokens.Count == 0 || !tokens[0].Equals("."))
        {
            throw new Exception("Missing '.' after resource selector expression in policy");
        }
        tokens.RemoveAt(0);

        // Extract the property name.
        string propertyName;
        if (tokens.Count == 0)
        {
            throw new Exception("Missing property name after resource selector expression in policy");
        }
        propertyName = tokens[0];
        tokens.RemoveAt(0);

        // Add property if necessary.
        if (!monitoredProperties[functionScope.ResourceType].Contains(propertyName))
        {
            monitoredProperties[functionScope.ResourceType].Add(propertyName);
        }

        // Return the combined expression.
        return new PropertySelectorExpr(st, functionScope, e, propertyName);
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        // Filter the resources, selecting those in scope.
        Dictionary<string, List<object>> filteredResources = this.scope.Filter(resources);

        // Check that the result is not the empty list.
        if (filteredResources[this.scope.ResourceType].Count == 0)
        {
            throw new Exception("Empty resource list in resource selector");
        }

        // Select the resource whose property we are after.
        object selectedResource = filteredResources[this.scope.ResourceType][0];
        double value = Double.Parse((this.expr.eval(filteredResources[this.scope.ResourceType][0], filteredResources)).ToString());
        for (int i = 1; i < filteredResources[this.scope.ResourceType].Count; i++)
        {
            double nextValue = Double.Parse((this.expr.eval(filteredResources[this.scope.ResourceType][i], filteredResources)).ToString());

            if ((this.selectorType == SelectorType.ARGMAX && nextValue > value) ||
                (this.selectorType == SelectorType.ARGMIN && nextValue < value))
            {
                selectedResource = filteredResources[this.scope.ResourceType][i];
                value = nextValue;
            }
        }

        // Obtain the value of the property for the selected resource.
        object val = selectedResource.GetType().GetProperty(this.property).GetGetMethod().Invoke(selectedResource, null);

        // Convert all numerical values to a double.
        if (!val.GetType().Name.ToLower().Contains("string") && !val.GetType().Name.ToLower().Contains("bool"))
        {
            double d = Double.Parse(val.ToString());
            val = d;
        }

        // Return the value.
        return val;
    }
}

// A relational expression has the form
//
//   expr rel-operator expr
//
// where rel-operator is one of '<', '>', '=' , '!=', '<=', '>=', '=~', '!~'.
//
public class RelExpr : BooleanExpr
{
    // Relational operator.
    enum RelOperator { NA, LT, GT, EQ, NE, LE, GE, RegEx };
    private RelOperator op;

    // Left and right arithmetic expressions.
    private Expr left, right;

    // Basic constructor.
    private RelExpr(Expr left, RelOperator op, Expr right)
    {
        this.left = left;
        this.op = op;
        this.right = right;

        // Check that the left and right terms of the expression refer to the same resource type, or one has no type.
        if (!left.exprType.Equals(right.exprType) && !left.exprType.Equals("") && !right.exprType.Equals(""))
        {
            throw new Exception("Relational expression terms refer to resources of different types ('" + left.exprType + "', '" + right.exprType + "')");
        }

        // Check that the types of the terms are the same.
        switch (left.ValueType())
        {
            case ExpressionValueType.NUMERICAL:
                if (right.ValueType() != ExpressionValueType.NUMERICAL)
                {
                    throw new Exception("Terms of mismatched types in relational expression (numerical and non-numerical)");
                }
                break;

            case ExpressionValueType.BOOL:
                if (right.ValueType() != ExpressionValueType.BOOL)
                {
                    throw new Exception("Terms of mismatched types in relational expression (Boolean and non-Boolean)");
                }
                break;

            case ExpressionValueType.STRING:
                if (right.ValueType() != ExpressionValueType.STRING)
                {
                    throw new Exception("Terms of mismatched types in relational expression (string and non-string)");
                }
                break;

            default:
                throw new Exception("Term of invalid type in relational expression (only Boolean, numerical and string expressions are allowed)");
        }

        // This is a (resource) type-free expression.
        this.exprType = left.exprType.Equals("") ? right.exprType : left.exprType;
    }

    static public bool IsRelOperator(List<string> tokens)
    {
        return tokens.Count >= 1 && (tokens[0].Equals("=") || tokens[0].Equals(">") || tokens[0].Equals("<")) ||
               tokens.Count >= 2 && tokens[0].Equals("!") && (tokens[1].Equals("=") || tokens[1].Equals("~"));
    }

    // Expression parser.
    static public BooleanExpr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Parse the left-hand-side expression, then use the other parser.
        return RelExpr.Parse(RelTerm.Parse(tokens, monitoredProperties), tokens, monitoredProperties);

    }

    // Parser for the case when the left-hand-side term is known.
    static public BooleanExpr Parse(Expr leftExpr, List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Check if one of the relational operators follows.
        RelOperator o = ParseRelOperator(tokens);
        if (o == RelOperator.NA)
        {
            throw new Exception("Invalid relational expression in policy");
        }

        // Check for regular expressions.
        if (o == RelOperator.RegEx)
        {
            return RegularExpr.Parse(leftExpr, tokens, monitoredProperties);
        }

        // Parse the right arithmetic expression.
        Expr rightExpr = RelTerm.Parse(tokens, monitoredProperties);

        // Build and return the relational expression.
        return new RelExpr(leftExpr, o, rightExpr);
    }

    // Relational operator parser.
    static private RelOperator ParseRelOperator(List<string> tokens)
    {
        RelOperator o;

        // Handle empty token list.
        if (tokens.Count == 0)
        {
            throw new Exception("Expression ends before expected relational operator");
        }

        // Handle '<' and '<='.
        else if (tokens[0].Equals("<"))
        {
            if (tokens.Count > 1 && tokens[1].Equals("="))
            {
                tokens.RemoveRange(0, 2);
                o = RelOperator.LE;
            }
            else
            {
                tokens.RemoveAt(0);
                o = RelOperator.LT;
            }
        }

        // Handle '>' and '>='.
        else if (tokens[0].Equals(">"))
        {
            if (tokens.Count > 1 && tokens[1].Equals("="))
            {
                tokens.RemoveRange(0, 2);
                o = RelOperator.GE;
            }
            else
            {
                tokens.RemoveAt(0);
                o = RelOperator.GT;
            }
        }

        // Handle '=' and "=~".
        else if (tokens[0].Equals("="))
        {
            if (tokens.Count > 1 && tokens[1].Equals("~"))
            {
                o = RelOperator.RegEx;
            }
            else
            {
                tokens.RemoveAt(0);
                o = RelOperator.EQ;
            }
        }

        // Handle '!='.
        else if (tokens[0].Equals("!") && tokens.Count > 1 && tokens[1].Equals("="))
        {
            tokens.RemoveRange(0, 2);
            o = RelOperator.NE;
        }

        // Handle "!~".
        else if (tokens[0].Equals("!") && tokens.Count > 1 && tokens[1].Equals("~"))
        {
            o = RelOperator.RegEx;
        }

        // Not a relational operator.
        else
        {
            o = RelOperator.NA;
        }

        // Return operator.
        return o;
    }

    // Evaluator.
    public override bool evalBool(object o, Dictionary<string, List<object>> resources)
    {
        try
        {
            // Evaluate the operands.
            object l = this.left.eval(o, resources);
            object r = this.right.eval(o, resources);

            // Extract the operand types.
            string ltype = l.GetType().Name.ToLower();
            string rtype = r.GetType().Name.ToLower();

            // Handle strings.
            if (ltype.Contains("string"))
            {
                string ls = l.ToString();
                string rs = r.ToString();

                switch (this.op)
                {
                    case RelOperator.EQ:
                        return string.Compare(ls, rs) == 0;
                    case RelOperator.GE:
                        return string.Compare(ls, rs) >= 0;
                    case RelOperator.GT:
                        return string.Compare(ls, rs) > 0;
                    case RelOperator.LE:
                        return string.Compare(ls, rs) <= 0;
                    case RelOperator.LT:
                        return string.Compare(ls, rs) < 0;
                    case RelOperator.NE:
                        return string.Compare(ls, rs) != 0;
                    default:
                        return false;
                }
            }

            // Handle booleans.
            if (ltype.Contains("bool"))
            {
                switch (this.op)
                {
                    case RelOperator.EQ:
                        return l == r;
                    case RelOperator.NE:
                        return l != r;
                    default:
                        return false;
                }
            }

            // Handle numerical operands.
            double ld = Double.Parse(l.ToString());
            double rd = Double.Parse(r.ToString());

            switch (this.op)
            {
                case RelOperator.EQ:
                    return ld == rd;
                case RelOperator.GE:
                    return ld >= rd;
                case RelOperator.GT:
                    return ld > rd;
                case RelOperator.LE:
                    return ld <= rd;
                case RelOperator.LT:
                    return ld < rd;
                case RelOperator.NE:
                    return ld != rd;
                default:
                    return false;
            }
        }
        catch
        {
            return false;
        }
    }
}

// A term in a relational expression, of the form:
//
//   (expression)
//   arithmetic-expression
//   boolean-constant
//   string-constant
//
public abstract class RelTerm : Expr
{
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {

        Expr e;

        // Handle string constant.
        if (tokens.Count > 0 && tokens[0].Equals("\""))
        {
            e = StringConstant.Parse(tokens, monitoredProperties);
        }

        // Handle bool constant.
        else if (tokens.Count > 0 && (tokens[0].ToLower().Equals("true") || tokens[0].ToLower().Equals("false")))
        {
            e = new BoolConstant(tokens[0].ToLower().Equals("true"));
            tokens.RemoveAt(0);
        }

        // Handle paranthesised expression.
        else if (tokens.Count > 0 && tokens[0].Equals("("))
        {
            // Remove left paranthesis.
            tokens.RemoveAt(0);

            // Parse expression.
            e = RelTerm.Parse(tokens, monitoredProperties);

            // Check for matching right paranthesis.
            if (tokens.Count == 0 || !tokens[0].Equals(")"))
            {
                throw new Exception("Missing right paranthesis in policy expression");
            }
            tokens.RemoveAt(0);
        }

        // Handle arithmetic expression.
        else
        {
            e = ArithmExpr.Parse(tokens, monitoredProperties);
        }

        // Return the expression.
        return e;
    }
}

// Regular expression of the form
//
//   resource-property regex-operator "pattern"
//
// where regex-operator is one of '=~' or '!~', with the semantics from Perl.
//
public class RegularExpr : BooleanExpr
{
    // The expression that is compared with a regular expression.
    private Expr leftExpr;

    // The regular expression.
    private Regex regularEx;

    // Flag specifying if the operator is '=~' (if false, the operator is '!~').
    private bool matchesOp;

    // Basic constructor.
    private RegularExpr(Expr leftExpr, bool matchesOp, Regex regularEx)
    {
        this.leftExpr = leftExpr;
        this.matchesOp = matchesOp;
        this.regularEx = regularEx;

        // Check that the left term is a string-valued resoruce property.
        if (leftExpr.ValueType() != ExpressionValueType.STRING)
        {
            throw new Exception("Invalid term in regular expression (must be string-valued resource property)");
        }

        // This is a (resource) type-free expression.
        this.exprType = leftExpr.exprType;
    }

    // Parser.
    static public RegularExpr Parse(Expr leftExpr, List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Parse the operator.
        if (tokens.Count < 3 || (!tokens[0].Equals("=") && !tokens[0].Equals("!")) ||
            !tokens[1].Equals("~") || !tokens[2].Equals("\""))
        {
            throw new Exception("Invalid regular expression in policy");
        }
        bool op = tokens[0].Equals("=");
        tokens.RemoveRange(0, 2);

        // Check that the pattern starts with '"'.
        if (tokens.Count == 0 || !tokens[0].Equals("\""))
        {
            throw new Exception("Invalid pattern in regular expression");
        }
        tokens.RemoveAt(0);

        // Parse the regex pattern.
        string pattern = "";
        bool escapedChar = false;
        while (tokens.Count > 0 && (escapedChar || !tokens[0].Equals("\"")))
        {
            pattern += tokens[0];
            escapedChar = tokens[0].Equals("\\");
            tokens.RemoveAt(0);
        }

        // Check that pattern ends with '"'.
        if (tokens.Count == 0 || !tokens[0].Equals("\""))
        {
            throw new Exception("Invalid regular expression in policy");
        }
        tokens.RemoveAt(0);

        // Build and return the regular expression.
        return new RegularExpr(leftExpr, op, new Regex(pattern));
    }

    // Evaluator.
    public override bool evalBool(object o, Dictionary<string, List<object>> resources)
    {
        try
        {
            string val = (string)this.leftExpr.eval(o, resources);
            return this.matchesOp ? this.regularEx.IsMatch(val) : !this.regularEx.IsMatch(val);
        }
        catch
        {
            return false;
        }
    }
}

// An arithmetic expression of the form
//
//   term +/- term +/- ... +/- term
//
public class ArithmExpr : ArithmeticExpr
{
    // The two operands.
    private Expr left, right;

    // The operator.
    enum AddSubOp { ADD, SUB };
    private AddSubOp op;

    // Basic constructor.
    private ArithmExpr(Expr left, AddSubOp op, Expr right)
    {
        this.left = left;
        this.op = op;
        this.right = right;

        // Check that the left and right terms of the expression refer to the same resource type, or at least one has no type.
        if (!left.exprType.Equals(right.exprType) && !left.exprType.Equals("") && !right.exprType.Equals(""))
        {
            throw new Exception("Arithmetic expression terms refer to resources of different types ('" + left.exprType + "', '" + right.exprType + "')");
        }

        // Check that the types of the terms are numerical.
        if (left.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Left term in arithmetic expression is not of numerical type");
        }
        if (right.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Right term in arithmetic expression is not of numerical type");
        }

        // Initialise type of expression.
        this.exprType = left.exprType.Equals("") ? right.exprType : left.exprType;
    }

    // Parser.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        Expr left = null;
        AddSubOp op = AddSubOp.ADD;

        bool moreTerms = true;

        while (moreTerms)
        {
            // Parse the next arithmetic term.
            Expr e = ArithmTerm.Parse(tokens, monitoredProperties);

            // Combine with the left term if any.
            if (left != null)
            {
                left = new ArithmExpr(left, op, e);
            }
            else
            {
                left = e;
            }

            // Check if there are more terms in the expression.
            if ((moreTerms = tokens.Count > 0 && (tokens[0].Equals("+") || tokens[0].Equals("-"))) == true)
            {
                op = tokens[0].Equals("+") ? AddSubOp.ADD : AddSubOp.SUB;
                tokens.RemoveAt(0);
            }
        }

        // Return the expression.
        return left;
    }

    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        double l = (double)this.left.eval(o, resources);
        double r = (double)this.right.eval(o, resources);
        return (this.op == AddSubOp.ADD) ? (l + r) : (l - r);
    }
}

// An arithmetic term of the form
//
//   factor *// factor *// ... *// factor
//
public class ArithmTerm : ArithmeticExpr
{
    // The two operands.
    private Expr left, right;

    // The operator.
    enum MulDivOp { MUL, DIV };
    private MulDivOp op;

    // Basic constructor.
    private ArithmTerm(Expr left, MulDivOp op, Expr right)
    {
        this.left = left;
        this.op = op;
        this.right = right;

        // Check that the left and right factors of the term refer to the same resource type, or at least one has no type.
        if (!left.exprType.Equals(right.exprType) && !left.exprType.Equals("") && !right.exprType.Equals(""))
        {
            throw new Exception("Arithmetic term factors refer to resources of different types ('" + left.exprType + "', '" + right.exprType + "')");
        }

        // Check that the types of the terms are numerical.
        if (left.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Left factor in arithmetic term is not of numerical type");
        }
        if (right.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Right factor in arithmetic term is not of numerical type");
        }

        // Initialise type of term.
        this.exprType = left.exprType.Equals("") ? right.exprType : left.exprType;
    }

    // Parser.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        Expr left = null;
        MulDivOp op = MulDivOp.MUL;

        bool moreFactors = true;

        while (moreFactors)
        {
            // Parse the next arithmetic factor.
            Expr e = ArithmFactor.Parse(tokens, monitoredProperties);

            // Combine with the left factor if any.
            if (left != null)
            {
                left = new ArithmTerm(left, op, e);
            }
            else
            {
                left = e;
            }

            // Check if there are more factors in the expression.
            if ((moreFactors = tokens.Count > 0 && (tokens[0].Equals("*") || tokens[0].Equals("/"))) == true)
            {
                op = tokens[0].Equals("*") ? MulDivOp.MUL : MulDivOp.DIV;
                tokens.RemoveAt(0);
            }
        }

        // Return the expression.
        return left;
    }

    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        double l = (double)this.left.eval(o, resources);
        double r = (double)this.right.eval(o, resources);
        return (this.op == MulDivOp.MUL) ? (l * r) : (l / r);
    }
}

// Arithmetic factor, which can be one of the following:
//
//   numerical-constant
//   string-constant
//   resource property
//   paranthesised arithmetic expression
// 
public class ArithmFactor : ArithmeticExpr
{
    // Parser.
    static public Expr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        Expr e;

        // Check that the token list is not empty.
        if (tokens.Count == 0)
        {
            throw new Exception("Empty expression in policy");
        }

        // Handle numerical constant.
        if (tokens[0].Equals(".") || Char.IsDigit((tokens[0].ToCharArray())[0]))
        {
            e = NumericalConstant.Parse(tokens, monitoredProperties);
        }

        // Handle paranthesised expression.
        else if (tokens[0].Equals("("))
        {
            // Remove left paranthesis.
            tokens.RemoveAt(0);

            // Parse expression.
            e = ArithmExpr.Parse(tokens, monitoredProperties);

            // Check for matching right paranthesis.
            if (tokens.Count == 0 || !tokens[0].Equals(")"))
            {
                throw new Exception("Missing right paranthesis in policy expression");
            }
            tokens.RemoveAt(0);
        }

        // Handle function.
        else if (MathsFunction.IsMathsFunction(tokens[0]))
        {
            e = MathsFunction.Parse(tokens, monitoredProperties);
        }

        // Handle set function.
        else if (SetArithmFunction.IsSetFunction(tokens[0]))
        {
            e = SetArithmFunction.Parse(tokens, monitoredProperties);
        }

        // Handle built-in property.
        else if (Tools.IsBuiltInVariable(tokens[0]))
        {
            e = PropertyExpr.Parse(tokens, monitoredProperties, true);
        }

        // Handle resource property.
        else if (Tools.IsResourceID(tokens[0]))
        {
            e = PropertyExpr.Parse(tokens, monitoredProperties, true);
        }

        // Handle selected-resource property.
        else if (PropertySelectorExpr.IsSelectorFunction(tokens[0]))
        {
            e = PropertySelectorExpr.Parse(tokens, monitoredProperties);
        }

        // Handle string constant.
        else if (tokens.Count > 0 && tokens[0].Equals("\""))
        {
            e = StringConstant.Parse(tokens, monitoredProperties);
        }

        // Error.
        else
        {
            throw new Exception("Error in policy expression at token '" + tokens[0] + "'");
        }

        // Return the expression.
        return e;
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        throw new Exception("The method or operation is not implemented.");
    }
}

// Numerical constant of the form
//
//   digits
//   digits.
//   digits.digits
//   .digits
//
public class NumericalConstant : ArithmeticExpr
{
    // The numerical constant.
    private object numConst;

    // Basic constructor.
    private NumericalConstant(object numConst)
    {
        this.numConst = numConst;
        this.exprType = "";
    }

    // Parser.
    static public NumericalConstant Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // String to bring the tokens of the constant together.
        string s;

        // The constant.
        object c;

        // Flag recording if the value is an integer (otherwise it is a double).
        bool isInt = true;

        // Identify the type of the numerical constant.
        // Handle the case when the constant starts with '.'.
        if (tokens.Count > 0 && tokens[0].Equals("."))
        {
            if (tokens.Count >= 2)
            {
                isInt = false;
                s = tokens[0] + tokens[1];
                tokens.RemoveRange(0, 2);
            }
            else
            {
                throw new Exception("Invalid numerical constant in policy expression");
            }
        }

        // Handle the case when the constant starts with a sequence of digits.
        else if (tokens.Count > 0)
        {
            s = tokens[0];
            tokens.RemoveAt(0);

            // Add decimal point if present.
            if (tokens.Count > 0 && tokens[0].Equals("."))
            {
                isInt = false;
                s += tokens[0];
                tokens.RemoveAt(0);

                // Add fractional part if applicable.
                if (tokens.Count > 0 && Char.IsDigit((tokens[0].ToCharArray())[0]))
                {
                    s += tokens[0];
                    tokens.RemoveAt(0);
                }
            }
        }

        // Handle the case when the tokens list is empty.
        else  // tokens.Count == 0
        {
            throw new Exception("Missing numerical constant in policy expression");
        }

        // Parse  the numerical constant.
        if (isInt)
        {
            int i = Int32.Parse(s);
            c = i;
        }
        else
        {
            double d = Double.Parse(s);
            c = d;
        }

        // c = isInt ? Int32.Parse(s) : Double.Parse(s);

        // Return the constant.
        return new NumericalConstant(c);
    }

    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        return Double.Parse(this.numConst.ToString());
    }
}

// A function of the form
//
//   min(expr1, expr2, ...)
//   max(expr1, expr2, ...)
//
public class MathsFunction : ArithmeticExpr
{
    // Supported functions.
    private enum FunctionType { MIN, MAX };

    // The function type.
    private FunctionType functionType;

    // The arguments.
    List<Expr> args;

    // Basic constructor.
    private MathsFunction(FunctionType functionType, List<Expr> args)
    {
        this.functionType = functionType;
        this.args = args;

        // Calculate and validate expression type.
        this.exprType = "";
        foreach (Expr e in args)
        {
            if (!e.exprType.Equals(this.exprType))
            {
                if (this.exprType.Equals(""))
                {
                    this.exprType = e.exprType;
                }
                else
                {
                    throw new Exception("Arguments of MIN/MAX function refer to different types of resources ('" + this.exprType + "', '" + e.exprType + "')");
                }
            }

            // Check that the argument is numerical.
            if (e.ValueType() != ExpressionValueType.NUMERICAL)
            {
                throw new Exception("Argument of MIN/MAX function is not of numerical type");
            }
        }
    }

    // Function type validator.
    public static bool IsMathsFunction(string s)
    {
        return s.Equals("min") || s.Equals("max");
    }

    // Parser.
    public static MathsFunction Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // The type of the function.
        FunctionType ft;

        // Parse function type.
        if (tokens.Count > 0 && tokens[0].Equals("min"))
        {
            ft = FunctionType.MIN;
        }
        else if (tokens.Count > 0 && tokens[0].Equals("max"))
        {
            ft = FunctionType.MAX;
        }
        else
        {
            throw new Exception("Faulty function in policy expression");
        }

        // Check that the function name is followed by a left paranthesis.
        if (tokens.Count < 2 || !tokens[1].Equals("("))
        {
            throw new Exception("Missing paranthesis after " + tokens[0] + " in policy expression");
        }
        tokens.RemoveRange(0, 2);

        // The list of arguments.
        List<Expr> args = new List<Expr>();
        bool moreArgs = true;
        while (moreArgs)
        {
            args.Add(ArithmExpr.Parse(tokens, monitoredProperties));
            if ((moreArgs = (tokens.Count > 0 && tokens[0].Equals(","))) == true)
            {
                tokens.RemoveAt(0);
            }
        }

        // Check that the function ends with a right paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals(")"))
        {
            throw new Exception("Missing right paranthesis after " + ((ft == FunctionType.MAX) ? "max" : "min") + " expression in policy");
        }
        tokens.RemoveAt(0);

        // Check that function has at least two arguments.
        if (args.Count < 2)
        {
            throw new Exception("Function must have at least two arguments in policy expression");
        }

        // Return the result.
        return new MathsFunction(ft, args);
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        double result = (double)this.args[0].eval(o, resources);

        for (int i = 1; i < args.Count; i++)
        {
            double x = (double)this.args[i].eval(o, resources);
            switch (this.functionType)
            {
                case FunctionType.MIN:
                    if (x < result)
                    {
                        result = x;
                    }
                    break;
                case FunctionType.MAX:
                    if (x > result)
                    {
                        result = x;
                    }
                    break;
            }
        }

        return result;
    }
}

// A set function of the form
//
//   function-type(set-comprehension, arithmetic-expression)
//
public class SetArithmFunction : ArithmeticExpr
{
    // Function type.
    private enum FunctionType { SUM, PROD, MAX, MIN, MEAN, COUNT };

    // The function type.
    private FunctionType functionType;

    // The scope for the function.
    private SetComprehensionExpr scope;

    // The expression.
    private Expr expr;

    // Basic constructor.
    private SetArithmFunction(FunctionType functionType, SetComprehensionExpr scope, Expr expr)
    {
        this.functionType = functionType;
        this.scope = scope;
        this.expr = expr;

        // Validate the types of the components for functions other than COUNT.
        if (functionType != FunctionType.COUNT)
        {
            // The scope and expression must refer to the same type of resource.
            if (!scope.exprType.Equals(expr.exprType))
            {
                throw new Exception("The scope and expression of set arithmetic function refer to different types of resources ('" + scope.exprType + "', '" + expr.exprType + "')");
            }

            // The expression must be numerical.
            if (expr.ValueType() != ExpressionValueType.NUMERICAL)
            {
                throw new Exception("Non-numerical expression in set arithmetic function");
            }
        }

        // Set arithmetic function are (resource) type-free!
        this.exprType = "";
    }

    // Parser.
    public static SetArithmFunction Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // The type of the function.
        FunctionType ft;

        // Parse function type.
        if (tokens.Count > 0)
        {
            switch (tokens[0])
            {
                case "SUM":
                    ft = FunctionType.SUM;
                    break;
                case "PROD":
                    ft = FunctionType.PROD;
                    break;
                case "MAX":
                    ft = FunctionType.MAX;
                    break;
                case "MIN":
                    ft = FunctionType.MIN;
                    break;
                case "MEAN":
                    ft = FunctionType.MEAN;
                    break;
                case "COUNT":
                    ft = FunctionType.COUNT;
                    break;
                default:
                    throw new Exception("Unsupported function '" + tokens[0] + "' in policy expression");
            }
        }
        else
        {
            throw new Exception("Invalid function expression in policy");
        }

        // Check that the function name is followed by a left paranthesis.
        if (tokens.Count < 2 || !tokens[1].Equals("("))
        {
            throw new Exception("Missing paranthesis after " + tokens[0] + " in policy expression");
        }
        tokens.RemoveRange(0, 2);

        // Parse the scope.
        SetComprehensionExpr functionScope = SetComprehensionExpr.Parse(tokens, monitoredProperties);

        // Parse the expression unless the function is 'COUNT'.
        Expr e;
        if (ft != FunctionType.COUNT)
        {
            // Check that the scope is followed by a comma.
            if (tokens.Count == 0 || !tokens[0].Equals(","))
            {
                throw new Exception("Invalid function arguments in policy expression");
            }
            tokens.RemoveAt(0);

            // Parse the expression.
            e = ArithmExpr.Parse(tokens, monitoredProperties);
        }

        // No expression required for 'COUNT'.
        else
        {
            e = null;
        }

        // Check that the arguments are followed by a right paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals(")"))
        {
            throw new Exception("Missing ')' after function arguments in policy expression");
        }
        tokens.RemoveAt(0);

        // Return the combined expression.
        return new SetArithmFunction(ft, functionScope, e);
    }

    // Function name validator.
    public static bool IsSetFunction(string s)
    {
        switch (s)
        {
            case "SUM":
            case "PROD":
            case "MAX":
            case "MIN":
            case "MEAN":
            case "COUNT":
                return true;
            default:
                return false;
        }
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        // Filter the resources, selecting those in scope.
        Dictionary<string, List<object>> filteredResources = this.scope.Filter(resources);

        // Handle the COUNT function.
        if (this.functionType == FunctionType.COUNT)
        {
            return filteredResources[this.scope.ResourceType].Count;
        }

        // For all other functions, check that the result is not the empty list.
        if (filteredResources[this.scope.ResourceType].Count == 0)
        {
            throw new Exception("Empty resource list in resource selector");
        }

        // Evaluate the value of the expression for the first resource.
        double value = Double.Parse((this.expr.eval(filteredResources[this.scope.ResourceType][0], filteredResources)).ToString());

        // Integrate the expression value for the other resources.
        for (int i = 1; i < filteredResources[this.scope.ResourceType].Count; i++)
        {
            double nextValue = Double.Parse((this.expr.eval(filteredResources[this.scope.ResourceType][i], filteredResources)).ToString());

            switch (this.functionType)
            {
                case FunctionType.MAX:
                    if (nextValue > value)
                    {
                        value = nextValue;
                    }
                    break;

                case FunctionType.MEAN:
                case FunctionType.SUM:
                    value += nextValue;
                    break;

                case FunctionType.MIN:
                    if (nextValue < value)
                    {
                        value = nextValue;
                    }
                    break;

                case FunctionType.PROD:
                    value *= nextValue;
                    break;
            }
        }

        // Complete calculation for MEAN.
        if (this.functionType == FunctionType.MEAN)
        {
            value /= filteredResources[this.scope.ResourceType].Count;
        }

        // Return the value.
        return value;
    }
}

// String constant of the form "string".
//
public class StringConstant : Expr
{
    // The string.
    private string strConst;

    // Basic constructor.
    private StringConstant(string strConst)
    {
        this.strConst = strConst;
        this.exprType = "";
    }

    // This expression is always string-valued.
    public override ExpressionValueType ValueType()
    {
        return ExpressionValueType.STRING;
    }

    // Parser.
    static public StringConstant Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Make sure the token list starts with a string.
        if (tokens.Count < 2 || !tokens[0].Equals("\""))
        {
            throw new Exception("Invalid string constant in policy expression");
        }
        tokens.RemoveAt(0);

        // The string.
        string s = "";
        bool escapedChar = false;
        while (tokens.Count > 0 && (escapedChar || !tokens[0].Equals("\"")))
        {
            s += tokens[0];
            escapedChar = !escapedChar && tokens[0].Equals("\\");
            tokens.RemoveAt(0);
        }

        // Check that string ends with '"'.
        if (tokens.Count == 0 || !tokens[0].Equals("\""))
        {
            throw new Exception("Invalid string constant in policy expression");
        }
        tokens.RemoveAt(0);

        // Build and return string constant.
        return new StringConstant(s);
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        return this.strConst;
    }
}

// A policy action.
//
public class PolicyAction
{
    static public PolicyAction Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Policy action.
        PolicyAction action;

        // Handle utility-function policy action.
        if (tokens.Count > 0 && UtilityPA.IsUtilityFunction(tokens[0]))
        {
            action = UtilityPA.Parse(tokens, monitoredProperties);
        }

        // Handle action policy action.
        else
        {
            action = ActionPA.Parse(tokens, monitoredProperties);
        }

        // Return the parsed policy action.
        return action;
    }
}

// An action policy action, of the form
//
//   resource.property = expression; ...; resource.property = expression
//
public class ActionPA : PolicyAction
{
    // List of assignments.
    private List<AssignExpr> assignments;

    // Basic constructor.
    private ActionPA(List<AssignExpr> assignments)
    {
        this.assignments = assignments;
    }

    // Parser.
    static new public ActionPA Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // List for storing the assignments of the policy action.
        List<AssignExpr> assigns = new List<AssignExpr>();

        // Flag that stays true while more assignmentss need to be parsed.
        bool moreAssigns = true;

        // Parse the scope terms.
        while (moreAssigns)
        {
            // The scope term must be non-empty.
            if (tokens.Count == 0)
            {
                throw new Exception("Empty assignment in policy action");
            }

            // Parse the next term.
            assigns.Add(AssignExpr.Parse(tokens, monitoredProperties));

            // Check whether the scope contains more terms.
            if ((moreAssigns = tokens.Count > 0 && tokens[0].Equals(";")) == true)
            {
                tokens.RemoveAt(0);
            }
        }

        // All the tokens must have been consumed.
        if (tokens.Count > 0)
        {
            throw new Exception("Garbage at the end of policy action");
        }

        // Instantiate and return scope object.
        return new ActionPA(assigns);
    }

    // Implement policy action.
    public void Implement(object o, Dictionary<string, List<object>> resources)
    {
        foreach (AssignExpr e in this.assignments)
        {
            e.Implement(o, resources);
        }
    }
}

// An assignment expression of the form
// 
//   resource.property = expression
//
public class AssignExpr : Expr
{
    // Left term of the assignment.
    private PropertyExpr left;

    // Right-hand-side expression for the assignment.
    private Expr right;

    // Basic constructor.
    private AssignExpr(PropertyExpr left, Expr right)
    {
        this.left = left;
        this.right = right;
        this.exprType = left.exprType;

        // Check the property type and the value type of the left and right terms.
        if (!right.exprType.Equals("") && !right.exprType.Equals(left.exprType))
        {
            throw new Exception("The property assigned to in action policy belongs to a different type of resource ('" + left.exprType +
                "' than the type of the expression on the right ('" + right.exprType + "')");
        }

        // Left property cannot be a built-in variable.
        if (Tools.IsBuiltInVariable(left.ResourceType))
        {
            throw new Exception("Cannot modify built-in variable in action policy");
        }

        // Left property must be writeable.
        modifiability modif = left.GetModifiability();
        if (modif != modifiability.readwrite && modif != modifiability.writeonly)
        {
            throw new Exception("Cannot modify non-writeable resource property '" + left.ToString() + "' in action policy");
        }

        // The left resource property and the right expression must have the same type.
        switch (left.ValueType())
        {
            case ExpressionValueType.BOOL:
                if (right.ValueType() != ExpressionValueType.BOOL)
                {
                    throw new Exception("Cannot assign non-Boolean value to Boolean resource property in action policy");
                }
                break;

            case ExpressionValueType.NUMERICAL:
                if (right.ValueType() != ExpressionValueType.NUMERICAL)
                {
                    throw new Exception("Cannot assign non-numerical value to numerical resource property in action policy");
                }
                break;

            case ExpressionValueType.STRING:
                if (right.ValueType() != ExpressionValueType.STRING)
                {
                    throw new Exception("Cannot assign non-string value to string-valued resource property in action policy");
                }
                break;

            default:
                throw new Exception("Cannot assign to non-scalar resource property in action policy");
        }
    }

    // This is always a non-scalar expression.
    public override ExpressionValueType ValueType()
    {
        return ExpressionValueType.NONSCALAR;
    }

    // Parser.
    static public AssignExpr Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Parse the resource property assigned to.
        PropertyExpr prop = PropertyExpr.Parse(tokens, monitoredProperties, false);

        // Check that the property is not a built-in variable.
        if (Tools.IsBuiltInVariable(prop.ResourceType))
        {
            throw new Exception("Cannot assign to built-in variable " + prop.ToString());
        }

        // Check that the property is writeable.
        modifiability modif = prop.GetModifiability();
        if (modif != modifiability.readwrite && modif != modifiability.writeonly)
        {
            throw new Exception("Cannot assign to non-writeable resource property " + prop.ToString());
        }

        // Parse the '='.
        if (tokens.Count == 0 || !tokens[0].Equals("="))
        {
            throw new Exception("Invalid policy action");
        }
        tokens.RemoveAt(0);

        // Parse the expression.
        Expr expr;
        if (prop.ValueType() == ExpressionValueType.BOOL)
        {
            expr = BoolExpr.Parse(tokens, monitoredProperties);
        }
        else
        {
            expr = ArithmExpr.Parse(tokens, monitoredProperties);
        }

        // Build and return the assignment.
        return new AssignExpr(prop, expr);
    }

    // Evaluator.
    public override object eval(object o, Dictionary<string, List<object>> resources)
    {
        throw new Exception("The method or operation is not implemented.");
    }

    // Implement policy action.
    public void Implement(object o, Dictionary<string, List<object>> resources)
    {
        // Get logger.
        ILog policyActionLog = LogManager.GetLogger("PolicyEvaluator.Policy.Action");

        // Check if policy action applies to this resource type.
        if (this.left.ResourceType.Equals(o.GetType().Name))
        {

            // Calculate the value of the right-hand side expression.
            object r = this.right.eval(o, resources);

            // Use reflection to set resource property to this value.
            this.left.SetProperty(o, r); // this.left.GetSetter(o).Invoke(o, new object[] { Tools.Convert(r, this.left.GetPropertyType(o)) });

            // Update audit-trail log.
            policyActionLog.Info("Setting read-write resource property " + this.left.PropertyName + "=" + r.ToString());
        }

        // Update audit-trail log.
        else
        {
            policyActionLog.Info("Skipping resource because policy action does not apply to " + o.GetType().Name);
        }
    }
}

// A goal policy action, to be added.
//
public class GoalPA : PolicyAction
{

}

// A utility-function policy action, of the form:
//
//  MAXIMISE(set-comprehension, set-arithmetic-expression, config-property-list, constraint-expression, operational-model-expression)
//
//  or 
//
//  PRISM-MAXIMISE(set-comprehension, set-arithmetic-expression, config-property-list, constraint-expression)
//
public class UtilityPA : PolicyAction
{
    // The types of utility functions supported.
    private enum UtilityType { MAXIMISE, PRISM_MAXIMISE };

    // The property selector.
    private UtilityType utilityType;

    // The scope for the function.
    private SetComprehensionExpr scope;

    // The utility function:
    // - of type SetArithmFunction if the policy is "global" (optimising over a set of resources)
    // - of type ArithmExpr if the policy is "local" (optimising at the level of each resource in its scope)
    //
    private Expr utility;

    // The list of resource properties to configure.
    private List<string> cfgProperties;

    // The list of primary key properties for the configured resource.
    private List<string> primaryKeyProperties;

    // The constraints.
    private Expr constraint;

    // The operational model expression.
    private OperationalModel opModel;

    // Basic constructor.
    private UtilityPA(UtilityType utilityType, SetComprehensionExpr scope, Expr utility, List<string> cfgProperties, Expr constraint, OperationalModel opModel,
        List<string> primaryKeyProperties)
    {
        this.utilityType = utilityType;
        this.scope = scope;
        this.utility = utility;
        this.cfgProperties = cfgProperties;
        this.constraint = constraint;
        this.opModel = opModel;
        this.primaryKeyProperties = primaryKeyProperties;

        // Validation.
        if (utility.ValueType() != ExpressionValueType.NUMERICAL)
        {
            throw new Exception("Utility function must be an arithmetic expression in policy action");
        }

        if (constraint.ValueType() != ExpressionValueType.BOOL)
        {
            throw new Exception("Constraint in utility-function policy action must be a Boolean expression");
        }
    }

    // Check for utility function.
    static public bool IsUtilityFunction(string keyword)
    {
        return keyword.Equals("MAXIMISE") || keyword.Equals("PRISM_MAXIMISE");
    }

    // Parser.
    static new public UtilityPA Parse(List<string> tokens, Dictionary<string, List<string>> monitoredProperties)
    {
        // Parse the type of utility function.
        UtilityType uType = UtilityType.MAXIMISE;
        switch (tokens[0])
        {
            case "MAXIMISE":
                uType = UtilityType.MAXIMISE;
                break;
            case "PRISM_MAXIMISE":
                uType = UtilityType.PRISM_MAXIMISE;
                break;
        }

        // Check that the keyword is followed by left paranthesis.
        if (tokens.Count < 2 || !tokens[1].Equals("("))
        {
            throw new Exception("Expecting '(' after '" + tokens[0] + "' in policy action");
        }

        // Remove parsed tokens.
        tokens.RemoveRange(0, 2);

        // Parse the scope.
        SetComprehensionExpr utilityScope = SetComprehensionExpr.Parse(tokens, monitoredProperties);

        // Shallow check that the resource in the utility-function scope has an "operationalModel" property.
        // Additional checks are required to check that the operational model has the correct type, etc.
        if (!Tools.State.ValidateResourceProperty(utilityScope.ResourceType, "operationalModel"))
        {
            throw new Exception("Cannot use utility-function policy for resources of type '" + utilityScope.ResourceType + 
                "' because they have no 'operationalModel' property");
        }

        // List of primary key properties for the configured resource.
        List<string> primaryKeyProps = new List<string>();

        // Ensure that the resoruces involved in the policy are monitored.
        if (!monitoredProperties.ContainsKey(utilityScope.ResourceType))
        {
            monitoredProperties[utilityScope.ResourceType] = new List<string>();
        }

        // Implicitely add all "primary key" properties of the resource type.
        Tools.AddPrimaryKeyProperties(utilityScope.ResourceType, monitoredProperties[utilityScope.ResourceType], primaryKeyProps);

        // Add 'operationalModel' property to those monitored for this resource type.
        monitoredProperties[utilityScope.ResourceType].Add("operationalModel");

        // Skip the comma.
        if (tokens.Count < 1 || !tokens[0].Equals(","))
        {
            throw new Exception("Expecting ',' after the first argument in utility-function policy action");
        }
        tokens.RemoveAt(0);

        // Parse the set arithmetic function, recording necessary properties in a separate Dictionary.
        Dictionary<string, List<string>> utilityProperties = new Dictionary<string, List<string>>();
        Expr utilityFunction = ArithmExpr.Parse(tokens, utilityProperties);

        // Skip the comma.
        if (tokens.Count < 1 || !tokens[0].Equals(","))
        {
            throw new Exception("Expecting ',' after the second argument in utility-function policy action");
        }
        tokens.RemoveAt(0);

        // The config property list must start with a left paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals("("))
        {
            throw new Exception("Configuration property list must start with '('");
        }
        tokens.RemoveAt(0);

        // Parse the list of properties to configure.
        List<CfgProperty> configProperties = new List<CfgProperty>();
        bool moreCfgProperties = true;
        while (moreCfgProperties)
        {
            configProperties.Add(CfgProperty.Parser(tokens, utilityScope.ResourceType));

            if ((moreCfgProperties = tokens.Count > 0 && tokens[0].Equals(",")) == true)
            {
                tokens.RemoveAt(0);
            }
        }
        List<string> cfgPropertyIds = CfgProperty.GetIds(configProperties);

        // Add any further properties that require monitoring.
        for (int i = 0; i < utilityProperties[utilityScope.ResourceType].Count; )
        {
            string p = utilityProperties[utilityScope.ResourceType][i];

            if (Tools.GetModifiability(utilityScope.ResourceType, p) != modifiability.derived && !cfgPropertyIds.Contains(p))
            {
                if (!monitoredProperties[utilityScope.ResourceType].Contains(p))
                {
                    monitoredProperties[utilityScope.ResourceType].Add(p);
                }
                utilityProperties[utilityScope.ResourceType].Remove(p);
            }
            else
            {
                i++;
            }
        }

        // The config property list must end with a right paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals(")"))
        {
            throw new Exception("Configuration property list must end with ')'");
        }
        tokens.RemoveAt(0);

        // Skip the comma.
        if (tokens.Count < 1 || !tokens[0].Equals(","))
        {
            throw new Exception("Expecting ',' after the third argument in utility-function policy action");
        }
        tokens.RemoveAt(0);

        // Parse the constraints, making sure configured properties are not added to the monitored properties (unless there).
        // Remember which configurable properties will be set by the policy engine and are not already monitored.
        List<string> unmonitoredProperties = new List<string>();
        foreach (CfgProperty p in configProperties)
        {
            if (!monitoredProperties[utilityScope.ResourceType].Contains(p.Id))
            {
                unmonitoredProperties.Add(p.Id);
            }
        }

        // Parse the constraints.
        Expr constraints = BoolExpr.Parse(tokens, monitoredProperties);

        // Eliminate 'unmonitored' properties from the list if applicable.
        foreach (string p in unmonitoredProperties)
        {
            if (monitoredProperties[utilityScope.ResourceType].Contains(p))
            {
                monitoredProperties[utilityScope.ResourceType].Remove(p);
            }
        }

        // Skip the comma.
        if (tokens.Count < 1 || !tokens[0].Equals(","))
        {
            throw new Exception("Expecting ',' after the fourth argument in utility-function policy action");
        }
        tokens.RemoveAt(0);

        // Parse operational model.
        OperationalModel operationalModel =
            (uType == UtilityType.MAXIMISE) ?
            SimpleOperationalModel.Parse(utilityScope.ResourceType, tokens, configProperties) :
            PrismOperationalModel.Parse(utilityScope.ResourceType, tokens, configProperties);

        // Make sure that "known" properties for the operational model are acquired.
        foreach (string p in operationalModel.KnownProperties)
        {
            if (!cfgPropertyIds.Contains(p))
            {
                if (!monitoredProperties[utilityScope.ResourceType].Contains(p))
                {
                    monitoredProperties[utilityScope.ResourceType].Add(p);
                }
            }
        }

        // Make sure that property definitions are acquired for PRISM-based utility-function policies.
        if (uType == UtilityType.PRISM_MAXIMISE)
        {
            foreach (string propertyID in operationalModel.NewProperties)
            {
                monitoredProperties[utilityScope.ResourceType].Add(propertyID + "Definition");
            }
        }

        // Make sure that the expression ends with right paranthesis.
        if (tokens.Count != 1 || !tokens[0].Equals(")"))
        {
            throw new Exception("Utility-function policy action must end with ')'");
        }

        // Build and return the parsed policy action.
        return new UtilityPA(uType, utilityScope, utilityFunction, cfgPropertyIds, constraints, operationalModel, primaryKeyProps);
    }

    // Implement policy action.
    public List<object> Implement(Dictionary<string, List<object>> allResources)
    {
        // Get policy action logger.
        ILog policyActionLog = LogManager.GetLogger("PolicyEvaluator.Policy.Action");

        // Build list of resources in the scope of this policy.
        List<object> utilityResources = this.scope.Filter2(allResources);

        // Nothing to do if the policy scope does not contain resource instances of the right type.
        if (utilityResources.Count == 0)
        {
            policyActionLog.Info("Nothing to do because the scope of the policy action is empty");
            return null;
        }

        // Build array of matched operation model element indices. Initialise current and best indices arrays.
        this.opModel.InitialiseStructures(utilityResources.Count);
        int[] numIndices = new int[utilityResources.Count];
        int[] crtIndices = new int[utilityResources.Count];
        int[] bestIndices = new int[utilityResources.Count];
        for (int i = 0; i < utilityResources.Count; i++)
        {
            if ((numIndices[i] = this.opModel.GetMatchingIndices(utilityResources[i], i)) == 0)
            {
                if (policyActionLog.IsInfoEnabled)
                {
                    StringBuilder msg = new StringBuilder(
                        "Cannot do anything " + (IsGlobal() ? "because the state of " : "for "));
                    Tools.PrintResourceInstance(utilityResources[i], msg);
                    msg.Append((IsGlobal() ? "" : " because its state") + " is in an area not covered by its operational model");
                    policyActionLog.Info(msg.ToString());
                }

                if (IsGlobal())
                {
                    return null;
                }
                else
                {
                    utilityResources[i] = null;
                }
            }

            crtIndices[i] = bestIndices[i] = 0;
        }

        // Handle global policy.
        if (IsGlobal())
        {
            // Examine all index combinations.
            double bestUtility = 0;
            bool hasSolution = false;
            bool moreCombinations = true;
            while (moreCombinations)
            {
                // Instantiate the current combination (Optimisation possible: after the first go,
                // only re-instantiate resources with changed instances).
                for (int i = 0; i < utilityResources.Count; i++)
                {
                    this.opModel.Instantiate(utilityResources[i], i, crtIndices[i]);
                }

                // Check that the current combination satisfies the constraints.
                if (this.constraint.evalBool(null, allResources))
                {
                    // Calculate the utility level of the current combination
                    double utilityValue = Double.Parse(this.utility.eval(null, allResources).ToString());

                    // Check if the current combination betters the previous best.
                    if (!hasSolution || utilityValue > bestUtility)
                    {
                        hasSolution = true;
                        bestUtility = utilityValue;
                        System.Array.Copy(crtIndices, bestIndices, crtIndices.Length);
                    }
                }

                // Move on to the next combination.
                moreCombinations = false;
                for (int i = 0; !moreCombinations && i < crtIndices.Length; i++)
                {
                    if (++crtIndices[i] < numIndices[i])
                    {
                        moreCombinations = true;
                    }
                    else
                    {
                        crtIndices[i] = 0;
                    }
                }
            }

            // Check that the optimisation problem has a solution.
            if (hasSolution)
            {
                // Instantiate the optimal configuration.
                for (int i = 0; i < bestIndices.Length; i++)
                {
                    this.opModel.Instantiate(utilityResources[i], i, bestIndices[i]);
                }

                // Clear non-config properties.
                Tools.ClearAllNonConfigProperties(utilityResources, this.cfgProperties, this.primaryKeyProperties);

                // Update the audit trail.
                if (policyActionLog.IsInfoEnabled)
                {
                    policyActionLog.Info("An expected system utility of " + bestUtility + " calculated for the resource instance configurations below");

                    foreach (object resource in utilityResources)
                    {
                        StringBuilder msg = new StringBuilder();
                        Tools.PrintResourceInstance(resource, msg);
                        policyActionLog.Info(msg);
                    }
                }
            }

            // Handle the case when no solution was found.
            else
            {
                // Remove resources from the list.
                utilityResources.RemoveRange(0, utilityResources.Count);

                // Update the log.
                policyActionLog.Info("Nothing to do because the no system configuration satisfies the policy constraint(s)");
            }
        }

        // Handle local policy.
        else
        {
            // Iterate over all resources in the scope of the policy.
            for (int i = 0; i < utilityResources.Count; i++)
            {
                // Skip nullified resources.
                if (utilityResources[i] == null)
                {
                    continue;
                }

                // Initialise local variables.
                double bestUtility = 0;
                bool hasSolution = false;

                // Iterate over all possible configurations for the resource.
                for (int j = 0; j < numIndices[i]; j++)
                {
                    // Instantiate the resource.
                    this.opModel.Instantiate(utilityResources[i], i, j);

                    // Check that the examined configuration satisfies the constraints.
                    if (this.constraint.evalBool(utilityResources[i], allResources))
                    {
                        // Calculate the utility level of the current combination
                        double utilityValue = Double.Parse(this.utility.eval(utilityResources[i], allResources).ToString());

                        // Check if the current combination betters the previous best.
                        if (!hasSolution || utilityValue > bestUtility)
                        {
                            hasSolution = true;
                            bestUtility = utilityValue;
                            bestIndices[i] = j;
                        }
                    }
                }

                // Check that a solution was found.
                if (hasSolution)
                {
                    // Instantiate the optimal configuration.
                    this.opModel.Instantiate(utilityResources[i], i, bestIndices[i]);

                    // Clear non-config properties.
                    Tools.ClearNonConfigProperties(utilityResources[i], this.cfgProperties, this.primaryKeyProperties);

                    // Update the audit trail.
                    if (policyActionLog.IsInfoEnabled)
                    {
                        StringBuilder msg = new StringBuilder("An expected utility of " + bestUtility + " calculated for the ");
                        Tools.PrintResourceInstance(utilityResources[i], msg);
                        policyActionLog.Info(msg);
                    }
                }
                else
                {
                    // Update the audit trail.
                    if (policyActionLog.IsInfoEnabled)
                    {
                        StringBuilder msg = new StringBuilder("Configuration not changed for the following resource because the examined configurations don't satisfy policy constraints ");
                        Tools.PrintResourceInstance(utilityResources[i], msg);
                        policyActionLog.Info(msg);
                    }

                    // Eliminate the resource from the list.
                    utilityResources[i] = null;
                }
            }

            // Filter out nullified resources.
            List<object> nonNullUtilityResources = new List<object>();
            for (int i = 0; i < utilityResources.Count; i++)
            {
                if (utilityResources[i] != null)
                {
                    nonNullUtilityResources.Add(utilityResources[i]);
                }
            }
            utilityResources = nonNullUtilityResources;
        }

        // Return the result of the policy evaluation.
        return (utilityResources.Count > 0) ? utilityResources : null;
    }

    // Internal function to check if the policy is local or global.
    public bool IsGlobal()
    {
        return this.utility is SetArithmFunction;
    }
}


// Configuration property that may include a range.
//
public class CfgProperty
{
    // ID of property.
    private string id;

    // Start value for the property.
    private double first;

    // Step value.
    private double step;

    // End value.
    private double last;

    // Number of values.
    private int numValues;

    // Flag to record if the property has an associated range.
    bool hasRange;

    // Basic constructor.
    private CfgProperty(string id, bool hasRange, double first, double step, double last)
    {
        this.id = id;
        this.hasRange = hasRange;
        this.first = first;
        this.step = step;
        this.last = last;
        if (hasRange)
        {
            // Range validation.
            if (step <= 0.0)
            {
                throw new Exception("Non-positive value (" + step + ") for the range step of configuration property " + id);
            }
            if (first > last)
            {
                throw new Exception("Range start value (" + first + ") cannot be larger than range end value (" + last + ") for configuration property " + id);
            }

            // Calculate the number of steps.
            this.numValues = (int)((last - first) / step + 1.0);
        }
    }

    // ID getter method.
    public string Id { get { return this.id; } }

    // hasRange getter method.
    public bool HasRange { get { return this.hasRange; } }

    // numValues getter method.
    public int NumValues { get { return this.numValues; } }

    // Parser.
    public static CfgProperty Parser(List<string> tokens, string resourceType)
    {
        // Make sure the property specification starts with the resoruce ID followed by '.'.
        if (tokens.Count < 2 || !tokens[0].Equals(resourceType) || !tokens[1].Equals("."))
        {
            throw new Exception("Resource property must start with '" + resourceType + ".' in configuration property list");
        }
        tokens.RemoveRange(0, 2);

        // Get the property ID.
        if (tokens.Count == 0)
        {
            throw new Exception("Missing property name in configuration property list");
        }
        string propId = tokens[0];
        tokens.RemoveAt(0);

        // Check if a range is specified.
        if (tokens.Count > 0 && tokens[0].Equals("("))
        {
            // Parse range values.
            double[] values = new double[3];

            try
            {
                // Skip paranthesis.
                tokens.RemoveAt(0);

                // Read three colon-separated double values.
                for (int i = 0; i < 3; i++)
                {
                    if (tokens.Count > 1 && tokens[1].Equals("."))
                    {
                        values[i] = Double.Parse(tokens[0] + "." + tokens[2]);
                        tokens.RemoveRange(0, 3);
                    }
                    else
                    {
                        values[i] = Double.Parse(tokens[0]);
                        tokens.RemoveAt(0);
                    }

                    // Skip ':'.
                    if (i < 2)
                    {
                        if (tokens.Count == 0 || !tokens[0].Equals(":"))
                        {
                            throw new Exception();
                        }
                        tokens.RemoveAt(0);
                    }
                }

                // Skip paranthesis.
                if (tokens.Count == 0 || !tokens[0].Equals(")"))
                {
                    throw new Exception();
                }
                tokens.RemoveAt(0);
            }

            // Handle exceptions.
            catch
            {
                throw new Exception("Faulty range in configuration property list");
            }

            // Return cfg property with range.
            return new CfgProperty(propId, true, values[0], values[1], values[2]);
        }

        // Handle the case when no range is specified.
        else
        {
            return new CfgProperty(propId, false, 0.0, 0.0, 0.0);
        }
    }

    // Build list of property Ids.
    public static List<string> GetIds(List<CfgProperty> cfgProperties)
    {
        List<string> ids = new List<string>();

        foreach (CfgProperty p in cfgProperties)
        {
            ids.Add(p.Id);
        }

        return ids;
    }

    // Generate value.
    public double GetValue(int index)
    {
        if (index < 0 || index >= this.numValues)
        {
            throw new Exception("Out of range index in GetValue(" + index + ") for property " + this.id);
        }

        return this.first + this.step * index;
    }

    // Override ToString method.
    public override string ToString()
    {
        return this.id + "=" + this.first + ":" + this.step + ":" + this.last;
    }
}

// Abstract class for the operational model to use with a goal or utility-function 
// policy action. This is described as "(derived_prop_1, ...)(known_prop_1, ...)"
// in the policy.
//
abstract public class OperationalModel
{
    // Names of parameters whose values are deduced from the operational model.
    protected List<string> newProperties;

    // List of configuration properties.
    protected List<CfgProperty> configProperties;

    // Lsit of configuration property ids.
    protected List<string> configPropertyIds;

    // Names of parameters whose values are either obtained from the adaptibility manager or set.
    protected List<string> knownProperties;

    // List of non-configured known properties, i.e., state = known - config.
    protected List<string> stateProperties;

    // Indices of configuration properties that do NOT appear in the operational model.
    protected List<int> independentConfigProperties;

    // Number of different combinations of independent configuration properties.
    protected int numIndependentCombinations;

    // Basic constructor.
    protected OperationalModel(List<string> newProperties, List<string> knownProperties, List<CfgProperty> configProperties)
    {
        this.newProperties = newProperties;
        this.configProperties = configProperties;
        this.configPropertyIds = CfgProperty.GetIds(configProperties);
        this.knownProperties = knownProperties;
        this.stateProperties = new List<string>();
        foreach (string p in knownProperties)
        {
            if (!this.configPropertyIds.Contains(p))
            {
                this.stateProperties.Add(p);
            }
        }
        this.independentConfigProperties = new List<int>();
        this.numIndependentCombinations = 1;
        for (int i = 0; i < configProperties.Count; i++)
        {
            if (!knownProperties.Contains(configProperties[i].Id))
            {
                if (!configProperties[i].HasRange)
                {
                    throw new Exception("Range must be specified for configuration property " + configProperties[i].Id + " that does not appear in the operational model");
                }
                this.independentConfigProperties.Add(i);
                this.numIndependentCombinations *= configProperties[i].NumValues;
            }
        }
    }

    // Getter for the known property list.
    public List<string> KnownProperties
    {
        get { return this.knownProperties; }
    }

    // Getter for the new property list.
    public List<string> NewProperties
    {
        get { return this.newProperties; }
    }

    // Initialise operational model structures for the number of resources
    // in the utility function scope.
    abstract public void InitialiseStructures(int nresources);

    // Build list of operational model element instances to be considered.
    abstract public int GetMatchingIndices(object resource, int pos);

    // Instantiate a resource with one of its possible configurations.
    abstract public void Instantiate(object resource, int pos, int index);
}

// Operational model obtained from multiple experiments.
//
public class SimpleOperationalModel : OperationalModel
{
    // Array of integer lists that hold the indices of the considered elements from the operational model,
    // one list per resource in the scope of the utility function.
    private List<int>[] indices;

    // Basic constructor.
    private SimpleOperationalModel(List<string> newProperties, List<string> knownProperties, List<CfgProperty> configProperties)
        :
        base(newProperties, knownProperties, configProperties)
    { }

    // Parser.
    static public OperationalModel Parse(string ResourceType, List<string> tokens, List<CfgProperty> configProperties)
    {
        return new SimpleOperationalModel(Tools.BuildPropertyList(ResourceType, tokens), Tools.BuildPropertyList(ResourceType, tokens), configProperties);
    }

    // Initialise structures.
    public override void InitialiseStructures(int nresources)
    {
        this.indices = new List<int>[nresources];
    }

    // Calculates the Euclidian distance between a resource instance state and an operational model element.
    private double CalculateDistance(Dictionary<string, object> coordinates, object modelElement)
    {
        double distance = 0; // Note: the distance is always 0 if the 'coordinates' dictionary is empty.

        foreach (string propertyId in coordinates.Keys)
        {
            object modelPropertyValue = modelElement.GetType().GetProperty(propertyId).GetGetMethod().Invoke(modelElement, null);

            string propertyType = modelPropertyValue.GetType().Name;
            switch (propertyType)
            {
                case "string":
                    if (!((string)modelPropertyValue).Equals((string)coordinates[propertyId]))
                    {
                        return Double.MaxValue;
                    }
                    break;

                case "bool":
                    if (((bool)modelPropertyValue) != ((bool)coordinates[propertyId]))
                    {
                        return Double.MaxValue;
                    }
                    break;

                default: // a numerical value
                    double val1 = Double.Parse(modelPropertyValue.ToString());
                    double val2 = Double.Parse(coordinates[propertyId].ToString());
                    double diff = val1 - val2;
                    distance += diff * diff;
                    break;
            }
        }

        return distance;
    }

    // Build list of operational model element instances whose state is closest to that of the resource instance.
    public override int GetMatchingIndices(object resource, int pos)
    {
        // Instantiate list of indices.
        this.indices[pos] = new List<int>();

        // Extract operational model array.
        object[] model = (object[])resource.GetType().GetProperty("operationalModel").GetGetMethod().Invoke(resource, null);

        // Find the "coordinates" for this resource.
        Dictionary<string, object> coordinates = new Dictionary<string, object>();
        foreach (string propertyId in this.stateProperties)
        {
            coordinates.Add(propertyId, resource.GetType().GetProperty(propertyId).GetGetMethod().Invoke(resource, null));
        }

        // Find the minimum Euclidean distance.
        double minDistance = Double.MaxValue;
        for (int i = 0; i < model.Length; i++)
        {
            double distance = CalculateDistance(coordinates, model[i]);

            if (distance < minDistance)
            {
                minDistance = distance;
            }
        }

        // Find the list of model elements at the minimum distance.
        if (minDistance != Double.MaxValue)
        {
            for (int i = 0; i < model.Length; i++)
            {
                if (CalculateDistance(coordinates, model[i]) == minDistance)
                {
                    this.indices[pos].Add(i);
                }
            }
        }

        // Return the result.
        return indices[pos].Count * this.numIndependentCombinations;
    }

    // Given a resource instance and the index of the index of an operational model element, 
    // enforce the associated configuration.
    public override void Instantiate(object resource, int pos, int index)
    {
        // Process independent configuration properties (i.e., those not in the operational model).
        // This is a DUPLICATE block of code; to be abstracted out into base class method.
        int[] indepCfgPropIndices = new int[this.independentConfigProperties.Count];
        for (int i = 0; i < this.independentConfigProperties.Count; i++)
        {
            indepCfgPropIndices[i] = index % this.configProperties[this.independentConfigProperties[i]].NumValues;
            index /= this.configProperties[this.independentConfigProperties[i]].NumValues;
        }

        // Extract operational model array.
        object[] model = (object[])resource.GetType().GetProperty("operationalModel").GetGetMethod().Invoke(resource, null);

        // Extract the operational model element.
        object modelElement = model[(this.indices[pos])[index]];

        // Set configuration as per operational model element. This may overwrite some "known" properties, which is OK.
        foreach (string propertyId in this.configPropertyIds)
        {
            if (this.knownProperties.Contains(propertyId))
            {
                resource.GetType().GetProperty(propertyId).GetSetMethod().Invoke(resource, new object[] {
                       modelElement.GetType().GetProperty(propertyId).GetGetMethod().Invoke(modelElement,null)
                   });
            }
        }

        // Set configuration as per range of independent configuration properties.
        // This is a DUPLICATE block of code; to be abstracted out into base class method.
        for (int i = 0; i < this.independentConfigProperties.Count; i++)
        {
            int idx = this.independentConfigProperties[i];
            object value = this.configProperties[idx].GetValue(indepCfgPropIndices[i]);

            System.Reflection.PropertyInfo propInfo = resource.GetType().GetProperty(this.configPropertyIds[idx]);
            propInfo.GetSetMethod().Invoke(resource, new object[] { Tools.Convert(value, propInfo.PropertyType.FullName) });
        }

        // Set new state as per operational model element. This may overwrite some of the "known" properties, which is OK.
        foreach (string propertyId in this.newProperties)
        {
            resource.GetType().GetProperty(propertyId).GetSetMethod().Invoke(resource, new object[] {
                modelElement.GetType().GetProperty(propertyId).GetGetMethod().Invoke(modelElement,null)
            });
        }
    }
}

// PRISM operational model.
//
public class PrismOperationalModel : OperationalModel
{
    // An array comprising an element for each resource in the utility function scope. 
    // This element is a dictionary that maps resource property names to the lists of 
    // their possible values.
    private Dictionary<string, List<object>>[] options;

    // Constructor - called once, during policy parsing.
    private PrismOperationalModel(List<string> newProperties, List<string> knownProperties, List<CfgProperty> configProperties)
        : base(newProperties, knownProperties, configProperties)
    { }

    // Parser.
    static public OperationalModel Parse(string ResourceType, List<string> tokens, List<CfgProperty> configProperties)
    {
        return new PrismOperationalModel(Tools.BuildPropertyList(ResourceType, tokens), Tools.BuildPropertyList(ResourceType, tokens), configProperties);
    }

    // Initialiser - called each time the policy is evaluated.
    public override void InitialiseStructures(int nresources)
    {
        this.options = new Dictionary<string, List<object>>[nresources];
    }

    // Build the dictionary of possible property values for this resource.
    public override int GetMatchingIndices(object resource, int pos)
    {
        // Create property value dictionary for this resource.
        this.options[pos] = new Dictionary<string, List<object>>();

        // Create temporary model, property and PRISM result files.
        string tempModelFile = System.IO.Path.GetTempFileName();
        string tempPropertyFile = System.IO.Path.GetTempFileName();
        string tempResultFile = System.IO.Path.GetTempFileName();

        // Generate PRISM model file.
        using (System.IO.StreamWriter sw = new System.IO.StreamWriter(tempModelFile))
        {
            System.Reflection.PropertyInfo propertyProperty = resource.GetType().GetProperty("operationalModel");
            object modelStr = propertyProperty.GetGetMethod().Invoke(resource, null);
            sw.Write(modelStr);
        }

        // PRISM constants.
        string prismConstants = "";
        foreach (CfgProperty cfgProperty in this.configProperties)
        {
            // Check that the config property is mentioned in the operational model specifier.
            if (this.knownProperties.Contains(cfgProperty.Id))
            {
                if (!prismConstants.Equals(""))
                {
                    prismConstants += ",";
                }
                prismConstants += cfgProperty.ToString();
            }
        }
        foreach (string propertyId in this.stateProperties)
        {
            if (!prismConstants.Equals(""))
            {
                prismConstants += ",";
            }
            prismConstants += propertyId + "=" + resource.GetType().GetProperty(propertyId).GetGetMethod().Invoke(resource, null);
        }

        // Generate PRISM property file.
        using (System.IO.StreamWriter sw = new System.IO.StreamWriter(tempPropertyFile))
        {
            foreach (string propertyId in this.newProperties)
            {
                System.Reflection.PropertyInfo propertyProperty = resource.GetType().GetProperty(propertyId + "Definition");
                object propStr = propertyProperty.GetGetMethod().Invoke(resource, null);
                sw.WriteLine(propStr);
            }
        }

        // Run PRISM experiments.
        System.Diagnostics.Process prism = new System.Diagnostics.Process();
        prism.StartInfo.FileName = System.Configuration.ConfigurationManager.AppSettings["prismPath"];
        prism.StartInfo.WorkingDirectory = System.IO.Path.GetDirectoryName(prism.StartInfo.FileName);
        prism.StartInfo.Arguments = "\"" + tempModelFile + "\" \"" + tempPropertyFile + "\" -const " + prismConstants + " -exportresults \"" + tempResultFile + "\"";
        prism.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Hidden;
        prism.Start();
        prism.WaitForExit();

        // Parse the experiment results.
        string derivedProperty = "";
        using (System.IO.StreamReader sr = new System.IO.StreamReader(tempResultFile))
        {
            // Flag that ensures a single parsing of configuration property values.
            bool cfgValuesParsed = false;

            // Handle each derived resource property.
            while (!sr.EndOfStream)
            {
                // Look for the first line containing a PRISM result report.
                string[] tokens = sr.ReadLine().Split('"');
                if (tokens.Length > 1)
                {
                    // Record name of analysed property.
                    derivedProperty = tokens[1];

                    // Handle the list of configuration variables. Start from the first config property 
                    // when the first set of results is examined; and from the last property - i.e., 
                    // the result of the analysis - otherwise.
                    // (NB: PRISM may change the order from the constant list).
                    string[] propertyNames = sr.ReadLine().Replace("Result", derivedProperty).Split(' ');
                    for (int i = cfgValuesParsed ? propertyNames.Length - 1 : 0; i < propertyNames.Length; i++)
                    {
                        this.options[pos][propertyNames[i]] = new List<object>();
                    }

                    // Parse result until end of file or empty line.
                    bool moreEntries = true;
                    while (!sr.EndOfStream && moreEntries)
                    {
                        tokens = sr.ReadLine().Split(' ');
                        if ((moreEntries = (tokens.Length >= 1 && !tokens[0].Equals(""))) == true)
                        {
                            // Start from the first config property when the first set of results is examined;
                            // and from the last property - i.e., the result of the analysis - otherwise.
                            for (int i = cfgValuesParsed ? propertyNames.Length - 1 : 0; i < propertyNames.Length; i++)
                            {
                                double value = Double.Parse(tokens[i]);
                                this.options[pos][propertyNames[i]].Add(value);
                            }
                        }
                    }

                    // Make sure the configuration property values are not parsed more than once.
                    if (!cfgValuesParsed)
                    {
                        cfgValuesParsed = true;
                    }
                }
            }
        }

        // Remove the temporary files.
        System.IO.File.Delete(tempModelFile);
        System.IO.File.Delete(tempPropertyFile);
        System.IO.File.Delete(tempResultFile);

        //Temporary.
        return this.options[pos][derivedProperty].Count * this.numIndependentCombinations;
    }

    // Given a resource instance its position among all the resources in the utility function scope,
    // and the index of the values that should be assigned to its derived and configured properties, 
    // enforce the associated configuration.
    public override void Instantiate(object resource, int pos, int index)
    {
        // Process independent configuration properties (i.e., those not in the operational model).
        // This is a DUPLICATE block of code; to be abstracted out into base class method.
        int[] indepCfgPropIndices = new int[this.independentConfigProperties.Count];
        for (int i = 0; i < this.independentConfigProperties.Count; i++)
        {
            indepCfgPropIndices[i] = index % this.configProperties[this.independentConfigProperties[i]].NumValues;
            index /= this.configProperties[this.independentConfigProperties[i]].NumValues;
        }

        // Set configuration as per operational model element. This may overwrite some "known" properties, which is OK.
        foreach (string propId in this.configPropertyIds)
        {
            if (this.knownProperties.Contains(propId))
            {
                object value = this.options[pos][propId][index];

                System.Reflection.PropertyInfo propInfo = resource.GetType().GetProperty(propId);
                propInfo.GetSetMethod().Invoke(resource, new object[] { Tools.Convert(value, propInfo.PropertyType.FullName) });
            }
        }

        // Set configuration as per range of independent configuration properties.
        // This is a DUPLICATE block of code; to be abstracted out into base class method.
        for (int i = 0; i < this.independentConfigProperties.Count; i++)
        {
            int idx = this.independentConfigProperties[i];
            object value = this.configProperties[idx].GetValue(indepCfgPropIndices[i]);

            System.Reflection.PropertyInfo propInfo = resource.GetType().GetProperty(this.configPropertyIds[idx]);
            propInfo.GetSetMethod().Invoke(resource, new object[] { Tools.Convert(value, propInfo.PropertyType.FullName) });
        }

        // Set new state as per operational model element. This may overwrite some of the "known" properties, which is OK.
        foreach (string propertyId in this.newProperties)
        {
            object value = this.options[pos][propertyId][index];

            System.Reflection.PropertyInfo propInfo = resource.GetType().GetProperty(propertyId);
            propInfo.GetSetMethod().Invoke(resource, new object[] { Tools.Convert(value, propInfo.PropertyType.FullName) });
        }
    }

}

// Parsed autonomic computing policy.
//
class Policy
{
    private ScopeExpr scope;
    private Expr trigger;
    private PolicyAction action;

    public ScopeExpr Scope
    {
        get
        {
            return this.scope;
        }
        set
        {
            this.scope = value;
        }
    }

    public Expr Trigger
    {
        get
        {
            return this.trigger;
        }
        set
        {
            this.trigger = value;
        }
    }

    public PolicyAction Action
    {
        get
        {
            return this.action;
        }
        set
        {
            this.action = value;
        }
    }

    static public Policy Parse(autonomicComputingPolicy p, Dictionary<string, List<string>> monitoredProperties, int policyIdx)
    {
        // Name of parsed component.
        string parsedComponent = "";

        // List of tokens.
        List<string> tokens = new List<string>();

        // Parse the policy.
        try
        {
            // Parse the policy scope.
            parsedComponent = "scope";
            tokens = Tools.Tokenise(p.scope, new Regex("([\\+\\-\\*/<>=!|&:;,\\(\\)\\.\\s\\\"])"));
            ScopeExpr scope = ScopeExpr.Parse(tokens, monitoredProperties);

            // Parse the polic condition.
            parsedComponent = "condition";
            tokens = Tools.Tokenise(p.condition, new Regex("([\\+\\-\\*/<>=!|&:;,\\(\\)\\.\\s\\\"])"));
            Expr condition = BoolExpr.Parse(tokens, monitoredProperties);
            if (!condition.exprType.Equals(""))
            {
                throw new Exception("policy condition must be global, not specific to a resource (use policy scope for resource-specific constraints)");
            }

            // Parse the policy action.
            parsedComponent = "action";
            tokens = Tools.Tokenise(p.action, new Regex("([\\+\\-\\*/<>=!|&:;,\\(\\)\\.\\s\\\"])"));
            PolicyAction action = PolicyAction.Parse(tokens, monitoredProperties);

            // Return the parsed policy.
            return new Policy(scope, condition, action);
        }

        // Handle exceptions.
        catch (Exception e)
        {
            string where = (tokens.Count == 0) ? " the end" : ("'" + Tools.Concatenate(tokens) + "'"); 
            throw new Exception("Error in the " + parsedComponent + " component of policy " + policyIdx + " at or before " + where + ": " + e.Message);
        }
    }

    public Policy(ScopeExpr scope, Expr trigger, PolicyAction action)
    {
        this.scope = scope;
        this.trigger = trigger;
        this.action = action;
    }
}
