//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Data;
using System.Configuration;
using System.Xml;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;

/// <summary>
///    Maintains and supports updates to the policy engine state.
/// </summary>
public class PolicyEngineState
{
    // The state of the policy engine.
    private policyEngine state;

    // Flag that records whether the policy engine state 
    // has changed since the last 'save' operation.
    private bool modified;

    // Getter/setter methods for the system model.
    public system SystemModel
    {
        get
        {
            return state.system;
        }
        set
        {
            // Mark the state as modified.
            this.modified = true;

            // Update the policy engine state.
            this.state.system = value;

            // Cancel all policies in place - use the setter method 
            // to propagate the change to the policy engine processor.
            this.PolicySet = new autonomicComputingPolicy[0];
        }
    }

    // Getter/setter methods for the policy evaluation period.
    public uint Period
    {
        get
        {
            return this.state.period.Value;
        }
        set
        {
            // Update is required only if the period changed.
            if (this.state.period != value)
            {
                // Mark the state as modified.
                this.modified = true;

                // Remember the new value.
                this.state.period = value;

                // Update the timer.
                Tools.PolicyEngineTimer.Change(0, 1000 * value);
            }
        }
    }

    // Getter/setter methods for the resource URLs.
    public string[] ResourceUrls
    {
        get
        {
            return state.resourceUrls;
        }
        set
        {
            // Mark the state as modified.
            this.modified = true;

            // Record the new set of manageability adaptor URLs.
            this.state.resourceUrls = value;

            // Parse the resurce URLs.
            Tools.Processor.ParseURLs(value);
        }
    }   

    // Getter/setter methods for the policy set.
    public autonomicComputingPolicy[] PolicySet
    {
        get
        {
            return state.policySet;
        }
        set
        {
            // Parse policies.
            Tools.Processor.ParsePolicies(value);

            // Mark the state as modified.
            this.modified = true;

            // Record the new policy set.
            this.state.policySet = value;
        }
    }

    // Constructor: obtain the current state of the policy engine.
	public PolicyEngineState() 
    {
        // Find the state file.
        string stateFile = System.Configuration.ConfigurationManager.AppSettings["servicePath"] + "\\App_Data\\State.xml";

        // Read the state from the state file.
        if (System.IO.File.Exists(stateFile))
        {
            XmlDocument stateDoc = new XmlDocument();
            stateDoc.Load(stateFile);

            // Deserialise the policy engine state.
            System.Text.ASCIIEncoding encoding = new System.Text.ASCIIEncoding();
            System.IO.Stream stream = new System.IO.MemoryStream(encoding.GetBytes(stateDoc.OuterXml));
            System.Xml.Serialization.XmlSerializer xmlSerializer =
                new System.Xml.Serialization.XmlSerializer(typeof(policyEngine));
            this.state = (policyEngine)(xmlSerializer.Deserialize(stream));
        }

        // Or start with the default state if the state file does not exist yet.
        else
        {
            this.state = new policyEngine();
            this.state.period = 10;
            this.state.policySet = new autonomicComputingPolicy[0];
            this.state.resourceUrls = new string[0];
            this.state.system = null;
        }

        // Mark the fact that the state of the policy engine is synchronised.
        this.modified = false;
    }

    // Save the state of the policy engine if required.
    public void Save()
    {
        // Proceed only if the state has changed since the last save operation.
        if (this.modified)
        {
            // Find the state file.
            string stateFile = System.Configuration.ConfigurationManager.AppSettings["servicePath"] + "\\App_Data\\State.xml";

            // Serialise and save the policy engine state.
            System.Xml.Serialization.XmlSerializer xmlSerializer =
                new System.Xml.Serialization.XmlSerializer(typeof(policyEngine));
            System.IO.Stream stream = new System.IO.MemoryStream();
            xmlSerializer.Serialize(stream, this.state);
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();
            stream.Seek(0, System.IO.SeekOrigin.Begin);
            doc.Load(stream);
            doc.Save(stateFile);

            // The state is now saved.
            this.modified = false;
        }
    }

    // Resource property validation.
    public bool ValidateResourceProperty(string resourceID, string propertyID)
    {
        // Look the resource up in the system model.
        foreach (resource res in this.state.system.resource)
        {
            if (res.ID.Equals(resourceID))
            {
                // Look the property up in the system model.
                foreach (property prop in res.property)
                {
                    if (prop.ID.Equals(propertyID))
                    {
                        return true;
                    }
                }

                // The property was not found although the resource exists.
                return false;
            }
        }

        // The resource type was not found.
        return false;
    }

    // Resource type validation.
    public bool ValidateResourceType(string resourceID)
    {
        // Look the resource up in the system model.
        foreach (resource res in this.state.system.resource)
        {
            if (res.ID.Equals(resourceID))
            {
                return true;
            }
        }

        // The resource type was not found.
        return false;
    }
}
