//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the 
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as 
//    published by the Free Software Foundation, either version 3 of 
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public 
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.Data;
using System.Configuration;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using System.Collections;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Threading;
using System.IO;
using System.Text;

/// <summary>
/// Summary description for Tools
/// </summary>
public class Tools
{
    // Getter method for the policy engine state.
    public static PolicyEngineState State
    {
        get
        {
            return (PolicyEngineState)HttpContext.Current.Application["state"];
        }
    }

    // Getter method for the policy engine processor.
    public static PolicyEngineProcessor Processor
    {
        get
        {
            return (PolicyEngineProcessor)HttpContext.Current.Application["processor"];
        }
    }

    // Getter method for the timer.
    public static Timer PolicyEngineTimer
    {
        get
        {
            return (Timer)HttpContext.Current.Application["timer"];
        }
    }

    // Getter method for the high-level resources.
    public static Dictionary<string, List<object>> HighLevelResources
    {
        get
        {
            return (Dictionary<string, List<object>>)HttpContext.Current.Application["highLevelResources"];
        }
    }

    // Validate a system model, throwing an exception if the model is invalid.
    public static void ValidateModel(system s)
    {
        // Resource types and resoruce property ids must be alphanumerical, 
        // starting with a letter.
        Regex RE = new Regex(@"^[a-zA-Z]\w*$");

        // Check that the model defines any types of resources.
        if (s.resource == null || s.resource.Length == 0)
        {
            throw new Exception("No resources defined in the system model");
        }

        // Validate every resource definition.
        List<string> resourceTypes = new List<string>();
        foreach (resource r in s.resource)
        {
            // Validate the resource ID.
            if (!RE.IsMatch(r.ID))
            {
                throw new Exception("Invalid resource name '" + r.ID + "'");
            }

            // Make sure the resource type is not specified twice.
            if (resourceTypes.Contains(r.ID))
            {
                throw new Exception("Multiply defined resource '" + r.ID);
            }
            resourceTypes.Add(r.ID);

            // Check that at least one property is defined for the resource.
            if (r.property == null || r.property.Length == 0)
            {
                throw new Exception("No properties defined for resource type '" + r.ID + "'");
            }

            // Validate every resource property.
            List<string> propertyNames = new List<string>();
            foreach (property p in r.property)
            {
                // Validate the property name.
                if (!RE.IsMatch(p.ID))
                {
                    throw new Exception("Invalid property name '" + p.ID + "' for resource '" + r.ID + "'");
                }

                // Make sure that the property is not defined multiple times.
                if (propertyNames.Contains(p.ID))
                {
                    throw new Exception("Multiply defined resource property '" + r.ID + "." + p.ID + "'");
                }
                propertyNames.Add(p.ID);
            }
        }
    }

    // Check if string argument is the type of a resource.
    public static bool IsResourceID(string s)
    {
        resource[] rs = State.SystemModel.resource;
        if (rs != null)
        {
            foreach (resource r in rs)
            {
                if (r.ID.Equals(s))
                {
                    return true;
                }
            }
        }
        return false;
    }

    // Convert between numerical types.
    class SupportedTypes
    {
        public System.Nullable<float> f; 
        public System.Nullable<double> d;

        public System.Nullable<ushort> us;
        public System.Nullable<uint> ui;
        public System.Nullable<ulong> ul;

        public System.Nullable<short> s;        
        public System.Nullable<int> i;
        public System.Nullable<long> l;        
    }
    public static object Convert(object value, string typeName)
    {
        Type t = new SupportedTypes().GetType();

        if (typeName.Equals(t.GetField("d").FieldType.FullName))
        {
            double d = Double.Parse(value.ToString());
            return d;
        }
        else if (typeName.Equals(t.GetField("f").FieldType.FullName))
        {
            float f = (float) Double.Parse(value.ToString());
            return f;
        }

        else if (typeName.Equals(t.GetField("i").FieldType.FullName))
        {
            int i = Int32.Parse(value.ToString());
            return i;
        }
        else if (typeName.Equals(t.GetField("s").FieldType.FullName))
        {
            short s = Int16.Parse(value.ToString());
            return s;
        }    
        else if (typeName.Equals(t.GetField("l").FieldType.FullName))
        {
            long l = Int64.Parse(value.ToString());
            return l;
        }

        else if (typeName.Equals(t.GetField("ui").FieldType.FullName))
        {
            uint i = UInt32.Parse(value.ToString());
            return i;
        }
        else if (typeName.Equals(t.GetField("us").FieldType.FullName))
        {
            ushort s = UInt16.Parse(value.ToString());
            return s;
        }
        else if (typeName.Equals(t.GetField("ul").FieldType.FullName))
        {
            ulong l = UInt64.Parse(value.ToString());
            return l;
        }

        return value;
    }

    public static object CreateGenericObject(string name, params Type[] types)
    {
        string t = name + "`" + types.Length;
        Type generic = Type.GetType(t).MakeGenericType(types);
        return Activator.CreateInstance(generic);
    }

    public static string NormaliseTypeName(string typeName)
    {
        string[] typeNameParts = typeName.Split(new char[] { ' ' });
        string normalisedTypeName = "";
        foreach (string s in typeNameParts)
        {
            normalisedTypeName = normalisedTypeName + (normalisedTypeName.Equals("") ? s : s.Substring(0, 1).ToUpper() + s.Substring(1));
        }
        return normalisedTypeName;
    }

    // Checks if a string name denotes a built-in resource.
    //
    public static bool IsBuiltInVariable(string s)
    {
        return s.ToLower().Equals("time");
    }

    // Add copyright-licence notice to generated code.
    static public void AddCopyrightNotice(StreamWriter f)
    {
        f.WriteLine("//========================================================================");
        f.WriteLine("//");
        f.WriteLine("// Copyright (C) 2009");
        f.WriteLine("// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)");
        f.WriteLine("//");
        f.WriteLine("//========================================================================");
        f.WriteLine("//");
        f.WriteLine("// This file is part of the GPAC general-purpose framework for the");
        f.WriteLine("// development of autonomic computing applications.");
        f.WriteLine("//");
        f.WriteLine("//    GPAC is free software: you can redistribute it and/or modify");
        f.WriteLine("//    it under the terms of the GNU Affero General Public License as");
        f.WriteLine("//    published by the Free Software Foundation, either version 3 of");
        f.WriteLine("//    the License, or (at your option) any later version.");
        f.WriteLine("//");
        f.WriteLine("//    GPAC is distributed in the hope that it will be useful,");
        f.WriteLine("//    but WITHOUT ANY WARRANTY; without even the implied warranty of");
        f.WriteLine("//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
        f.WriteLine("//    GNU Affero General Public License for more details.");
        f.WriteLine("//");
        f.WriteLine("//    You should have received a copy of the GNU Affero General Public");
        f.WriteLine("//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.");
        f.WriteLine("//");
        f.WriteLine("//========================================================================");
        f.WriteLine("");
    }

    // Validates properties of built-in variables.
    //
    public static bool IsBuiltInVariableProperty(string var, string prop)
    {
        if (var.ToLower().Equals("time"))
        {
            Regex RE = new Regex("^(year|month|day|hour|minute|second)$");
            return RE.IsMatch(prop.ToLower());
        }

        return false;
    }

    // Evaluates built-in variable.
    public static object EvalBuiltIn(string var, string prop)
    {
        if (var.ToLower().Equals("time"))
        {
            DateTime now = DateTime.Now;

            switch (prop.ToLower())
            {
                case "year":
                    return now.Year;
                case "month":
                    return now.Month;
                case "day":
                    return now.Day;
                case "hour":
                    return now.Hour;
                case "minute":
                    return now.Minute;
                case "second":
                    return now.Second;
                default:
                    break;
            }
        }

        // Unknown built-in variable.
        throw new Exception("Cannot evaluate unknown built-in variable '" + var + "." + prop + "'");
    }

    // Splits a string into tokens, using the provided regular expression.
    //
    public static List<string> Tokenise(string s, Regex rex)
    {
        // List of tokens extracted from the string.
        List<string> tokens = new List<string>(rex.Split(s));

        // Remove all empty tokens outside speech marks.
        bool betweenSpeechMarks = false;
        bool escapedChar = false;
        for (int i = 0; i < tokens.Count; )
        {
            if (tokens[i].Equals("") || (!betweenSpeechMarks && (tokens[i].Equals(" ") || tokens[i].Equals("") || tokens[i].Equals("\n") || tokens[i].Equals("\t"))))
            {
                tokens.RemoveAt(i);
            }
            else
            {
                if (tokens[i].Equals("\"") && !escapedChar)
                {
                    betweenSpeechMarks = !betweenSpeechMarks;
                }
                escapedChar = !escapedChar && tokens[0].Equals("\\");
                i++;
            }
        }

        // Return the token list.
        return tokens;
    }

    // Concatenate list of tokens.
    public static string Concatenate(List<string> tokens)
    {
        StringBuilder msg = new StringBuilder();

        foreach (string token in tokens)
        {
            msg.Append(token);
        }

        return msg.ToString();
    }

    // Separates a string into tokens using the provided separator(s), but taking into account paranthesis.
    public static string[] ParseTokens(string s, char[] sep)
    {
        List<string> tokens = new List<string>();

        while (!s.Equals(""))
        {
            string token = "";

            bool finished = false;
            do
            {
                string[] tmp = s.Split(sep);
                int firstSep = (tmp.Length > 1) ? tmp[0].Length : -1;
                int firstBracket = s.IndexOf('(');
                int nbrackets = 0;

                if (firstSep < 0)
                {
                    token += s;
                    s = "";
                    finished = true;
                }
                else if (firstBracket < 0 || firstSep < firstBracket)
                {
                    token += s.Substring(0, firstSep);
                    s = s.Substring(firstSep + 1);
                    finished = true;
                }
                else
                {
                    token += s.Substring(0, firstBracket + 1);
                    s = s.Substring(firstBracket + 1);
                    nbrackets++;
                    do
                    {
                        firstBracket = s.IndexOf('(');
                        int firstClosedBracket = s.IndexOf(')');

                        if (firstBracket > 0 && firstBracket < firstClosedBracket)
                        {
                            token += s.Substring(0, firstBracket + 1);
                            s = s.Substring(firstBracket + 1);
                            nbrackets++;
                        }
                        else
                        {
                            token += s.Substring(0, firstClosedBracket + 1);
                            s = s.Substring(firstClosedBracket + 1);
                            nbrackets--;
                        }
                    } while (nbrackets > 0);
                }
            } while (!finished);

            tokens.Add(token);
        }

        return tokens.ToArray();
    }

    // Get resource definition from the system model.
    static public resource GetResourceModel(string resourceType)
    {
        system systemModel = Tools.State.SystemModel;
        foreach (resource r in systemModel.resource)
        {
            if (r.ID.Equals(resourceType))
            {
                return r;
            }
        }
        return null;
    }

    // Get resource property definition from the system model.
    static public property GetResourcePropertyModel(string resourceType, string propertyName)
    {
        resource r = Tools.GetResourceModel(resourceType);
        if (r != null)
        {
            foreach (property p in r.property)
            {
                if (p.ID.Equals(propertyName))
                {
                    return p;
                }
            }
        }
        return null;
    }

    // Add "public key" properties of a resource type to its list of monitored properties.
    static public void AddPrimaryKeyProperties(string resourceType, List<string> monitoredProperties, List<string> primaryKeyProperties)
    {
        resource r = Tools.GetResourceModel(resourceType);
        if (r != null)
        {
            foreach (property p in r.property)
            {
                if (p.primaryKey)
                {
                    if (!monitoredProperties.Contains(p.ID))
                    {
                        monitoredProperties.Add(p.ID);
                    }

                    if (primaryKeyProperties != null)
                    {
                        primaryKeyProperties.Add(p.ID);
                    }
                }
            }
        }
    }

    // Clear non-configuration properties of a resource instance.
    static public void ClearNonConfigProperties(object resource, List<string> configProperties, List<string> primaryKeyProperties)
    {
        // Get array of all public properties for the resource instance.
        System.Reflection.PropertyInfo[] props = resource.GetType().GetProperties();

        // Examine each property, clearing those that are neither key properties nor configured ones.
        foreach (System.Reflection.PropertyInfo p in props)
        {
            if (!primaryKeyProperties.Contains(p.Name) && !configProperties.Contains(p.Name))
            {
                p.GetSetMethod().Invoke(resource, new object[] { null });
            }
        }
    }

    // Clear non-configuration properties of all resources in a list.
    static public void ClearAllNonConfigProperties(List<object> resources, List<string> configProperties, List<string> primaryKeyProperties)
    {
        foreach (object resource in resources)
        {
            Tools.ClearNonConfigProperties(resource, configProperties, primaryKeyProperties);
        }
    }

    // Convert resource instance to string for audit trail.
    static public void PrintResourceInstance(object resource, StringBuilder msg)
    {
        // Add prefix.
        msg.Append(" '" + resource.GetType().Name + "' resource with");

        // Get array of all public properties for the resource instance.
        System.Reflection.PropertyInfo[] props = resource.GetType().GetProperties();

        // Examine each property, clearing those that are neither key properties nor configured ones.
        foreach (System.Reflection.PropertyInfo p in props)
        {
            object value = p.GetGetMethod().Invoke(resource, null);

            if (value != null)
            {
                msg.Append(" " + p.Name + "=" + value.ToString());
            }
        }
    }

    // Builds a resourceType -> resource list dictionary from a resourceTypeUrlPair -> resource list one.
    public static Dictionary<string, List<object>> BuildResourceDictionary(Dictionary<ResourceTypeUrlPair, IList> locatedResources)
    {
        Dictionary<string, List<object>> resources = new Dictionary<string, List<object>>();

        foreach (ResourceTypeUrlPair ru in locatedResources.Keys)
        {
            if (!resources.ContainsKey(ru.ResourceType))
            {
                resources[ru.ResourceType] = new List<object>();
            }

            foreach (object r in locatedResources[ru])
            {
                resources[ru.ResourceType].Add(r);
            }
        }

        return resources;
    }

    // Return modifiability of a resource property.
    public static modifiability GetModifiability(string resourceType, string propertyName)
    {
        return Tools.GetResourcePropertyModel(resourceType, propertyName).modifiability;
    }

    // Builds list of resource property names from tokenised list of the form
    //
    //  (resource.property1,resource.property2,...)
    //
    public static List<string> BuildPropertyList(string resourceType, List<string> tokens/*, bool includeRange*/)
    {
        // The list of property names.
        List<string> propertyList = new List<string>();

        // The list must start with a left paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals("("))
        {
            throw new Exception("Property list must start with '('");
        }
        tokens.RemoveAt(0);

        // Parse properties, one at a time.
        bool notFinished = true;
        while (notFinished)
        {
            if (tokens.Count < 2 || !tokens[0].Equals(resourceType) || !tokens[1].Equals("."))
            {
                throw new Exception("Resource property must start with '" + resourceType + ".' in property list");
            }
            tokens.RemoveRange(0, 2);

            if (tokens.Count == 0)
            {
                throw new Exception("Missing property name in property list");
            }
            propertyList.Add(tokens[0]);
            tokens.RemoveAt(0);

            if ((notFinished = (tokens.Count > 0 && tokens[0].Equals(","))))
            {
                tokens.RemoveAt(0);
            }
        }

        // The list must end with a right paranthesis.
        if (tokens.Count == 0 || !tokens[0].Equals(")"))
        {
            throw new Exception("Property list must end with ')'");
        }
        tokens.RemoveAt(0);

        // Return  list.
        return propertyList;
    }
}

public class ResourceTypeUrlPair
{
    private string resourceType;
    private string url;

    public ResourceTypeUrlPair(string resourceType, string url)
    {
        this.resourceType = resourceType;
        this.url = url;
    }

    public string ResourceType { get { return this.resourceType; } }
    public string Url { get { return this.url; } }
}

public class ResourcePair
{
    private object resource;
    private object enablingResource;
    private bool scheduled;

    public ResourcePair(object resource, object enablingResource)
    {
        this.resource = resource;
        this.enablingResource = enablingResource;
        this.scheduled = false;
    }

    public object Resource { get { return this.resource; } }
    public object EnablingResource { get { return this.enablingResource; } set { this.enablingResource = value; } }
    public bool Scheduled { set { this.scheduled = value; } get { return this.scheduled; } }
}