//========================================================================
//
// Copyright (C) 2009
// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)
//
//========================================================================
//
// This file is part of the GPAC general-purpose framework for the
// development of autonomic computing applications.
//
//    GPAC is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Affero General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    GPAC is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Affero General Public License for more details.
//
//    You should have received a copy of the GNU Affero General Public
//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.
//
//========================================================================

using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.CodeDom;
using System.CodeDom.Compiler;
using Microsoft.CSharp;


namespace AdaptorGenerator
{
    public partial class Form1 : Form
    {
        public Form1()
        {
            InitializeComponent();
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            About about = new About();

            if (about.ShowDialog() == DialogResult.OK)
            {
                ;
            }
        }

        private void toolStripMenuItem1_Click(object sender, EventArgs e)
        {
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void button1_Click(object sender, EventArgs e)
        {
            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                textBox1.Text = openFileDialog1.FileName;
            }
        }

        private void button2_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialog1.ShowDialog() == DialogResult.OK)
            {
                textBox2.Text = folderBrowserDialog1.SelectedPath;

                if (textBox1.Text.Length > 0 && textBox2.Text.Length > 0)
                {
                    button3.Enabled = true;
                }
            }
        }

        private void button3_Click(object sender, EventArgs e)
        {
            try
            {
                GenerateAdaptors();
                textBox1.Text = textBox2.Text = "";
            }
            catch (Exception ex)
            {
                Error errDialog = new Error(ex.Message);

                errDialog.ShowDialog();
            }
        }

        private void GenerateAdaptors()
        {
            // Given a system model for system X that defines resources 
            // Y1, Y2, ..., Yn, n>0, create the software artefacts below:
            //
            //    YiManageabilityAdaptor/
            //       App_Code/
            //         ManagedResource.cs
            //         XTypes.cs
            //         YiManageabilityAdaptor.cs
            //       App_Data/
            //       YiManageabilityAdaptor.asmx
            //       Web.Config
            //         
            // for all 1 <= i <= n.
            //

            // Load the XML system model.
            XmlDocument modelDoc = new XmlDocument();
            modelDoc.Load(textBox1.Text);

            // Get the system name.
            string systemName = getTextValue((XmlElement)modelDoc.GetElementsByTagName("system")[0], "name");

            // Validate the model.
            if (!modelDoc.DocumentElement.NamespaceURI.Equals("http://www.rcc.com/system") || !ModelValidator.Validate(modelDoc))
            {
                throw new Exception(textBox1.Text + "\nis not an instance of the GPAC system meta-model.");
            }

            // Generate the XML schema in the output folder.
            StringBuilder schemaStr = new StringBuilder();
            schemaStr.Append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n\n" +
                             "<xs:schema xmlns=\"http://www.rcc.com/system\" xmlns:xs=\"http://www.w3.org/2001/XMLSchema\" targetNamespace=\"http://www.rcc.com/system\" elementFormDefault=\"qualified\">\n\n");

            // Identify the template file directory.
            string templateDir =
                System.IO.Path.GetDirectoryName(
                System.Reflection.Assembly.GetExecutingAssembly().GetName().CodeBase).Substring(6) + "\\App_Data\\template\\";

            // Get the set of resources.
            XmlNodeList nl = modelDoc.GetElementsByTagName("resource");
            if (nl == null || nl.Count == 0)
            {
                throw new Exception("The specified system model contains\nno resource definitions.");
            }

            List<string> generateTypeFiles = new List<string>();

            for (int i = 0; i < nl.Count; i++)
            {
                XmlElement el = (XmlElement)nl[i];
                string resourceName = getTextValue(el, "ID");

                StringBuilder getterStr = new StringBuilder();
                StringBuilder setterStr = new StringBuilder();

                schemaStr.Append("  <xs:element name=\"" + resourceName + "\" type=\"" + resourceName + "\"/>\n" +
                                 "  <xs:complexType name=\"" + resourceName + "\">\n" +
                                 "    <xs:sequence>\n");

                XmlNodeList pl = el.GetElementsByTagName("property");
                if (pl == null || pl.Count == 0)
                {
                    throw new Exception("The system model doesn't define any property\nfor resource " + resourceName);
                }

                for (int j = 0; j < pl.Count; j++)
                {
                    XmlElement prop = (XmlElement)pl[j];
                    string propName = getTextValue(prop, "ID");

                    schemaStr.Append("      <xs:element name=\"" + propName + "\" type=\"" + 
                        MakePropertyType(resourceName, propName) + "\" nillable=\"true\"/>\n");

                    string fieldName = propName;
                    string propType = getTextValue(prop, "modifiability");
                    if (propType.Equals("derived"))
                    {
                        schemaStr.Append("      <xs:element name=\"" + propName + "Definition\" type=\"xs:string\" nillable=\"true\"/>\n");
                        fieldName += "Definition";
                    }
                    else if (propType.Equals("read-write") || propType.Equals("write-only"))
                    {
                        setterStr.Append("        if (resource." + propName + (IsString(prop)? " != null" : ".HasValue") + ")\n" +
                                        "        {\n" +
                                        "            //\n" +
                                        "            // Add code to set the new value for the " + propName + " property here\n" +
                                        "            //\n" +
                                        "            \n" +
                                        "        }\n");
                    }

                    getterStr.Append("            case \"" + fieldName + "\":\n" +
                                     "                //\n" +
                                     "                // Add code to return the value of the " + fieldName + " property here\n" +
                                     "                //\n");
                }

                schemaStr.Append("    </xs:sequence>\n" +
                                 "  </xs:complexType>\n");

                // Create the directory structure for the 'resourceName' manageability adaptor.
                string adaptorName = resourceName + "ManageabilityAdaptor";
                string maPath = textBox2.Text + "\\" + adaptorName;
                if (!Directory.Exists(maPath))
                {
                    Directory.CreateDirectory(maPath);
                }
                if (!Directory.Exists(maPath + "\\App_Code"))
                {
                    Directory.CreateDirectory(maPath + "\\App_Code");
                }
                if (!Directory.Exists(maPath + "\\App_Data"))
                {
                    Directory.CreateDirectory(maPath + "\\App_Data");
                }

                // Create the .asmx file.
                using (StreamWriter sw = new StreamWriter(maPath + "\\" + adaptorName + ".asmx"))
                {
                    sw.WriteLine("<%@ WebService Language=\"C#\" CodeBehind=\"~/App_Code/" + adaptorName + ".cs\" Class=\"" + adaptorName + "\" %>");
                }

                // Copy the template Web.Config file.
                File.Copy(templateDir + "Web.Config", maPath + "\\Web.Config", true);

                // Copy the base adaptor class.
                File.Copy(templateDir + "ManagedResource.cs", maPath + "\\App_Code\\ManagedResource.cs", true);

                // Add name of type file to the list.
                generateTypeFiles.Add(maPath + "\\App_Code\\" + systemName + "Types.cs");

                // Create the resource-specific adaptor class.
                using (StreamWriter sw = new StreamWriter(maPath + "\\App_Code\\" + adaptorName + ".cs"))
                {
                   sw.Write("//========================================================================\n" +
                            "//\n" +
                            "// Copyright (C) 2009\n" +
                            "// Radu Calinescu <Radu.Calinescu@comlab.ox.ac.uk> (University of Oxford)\n" +
                            "//\n" +
                            "//========================================================================\n" +
                            "//\n" +
                            "// This file is part of the GPAC general-purpose framework for the\n" +
                            "// development of autonomic computing applications.\n" +
                            "//\n" +
                            "//    GPAC is free software: you can redistribute it and/or modify\n" +
                            "//    it under the terms of the GNU Affero General Public License as\n" +
                            "//    published by the Free Software Foundation, either version 3 of\n" +
                            "//    the License, or (at your option) any later version.\n" +
                            "//\n" +
                            "//    GPAC is distributed in the hope that it will be useful,\n" +
                            "//    but WITHOUT ANY WARRANTY; without even the implied warranty of\n" +
                            "//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" +
                            "//    GNU Affero General Public License for more details.\n" +
                            "//\n" +
                            "//    You should have received a copy of the GNU Affero General Public\n" +
                            "//    License along with GPAC. If not, see <http://www.gnu.org/licenses/>.\n" +
                            "//\n" +
                            "//========================================================================\n\n");

                   sw.Write("using System;\n" +
                            "using System.Web;\n" +
                            "using System.Web.Services;\n" +
                            "using System.Web.Services.Protocols;\n" +
                            "\n" +
                            "[WebService(Namespace = \"http://www.yupe.com/\")]\n" +
                            "[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]\n" +
                            "public class " + adaptorName +  ": ManagedResource<" + resourceName + ">\n" +
                            "{\n" +
                            "    protected override object[] GetRawResources()\n" +
                            "    {\n" +
                            "        " + resourceName + "[] " + resourceName + "Array = null;\n" +
                            "        \n" +
                            "        //\n" +
                            "        // Add code to build managed resource array here\n" +
                            "        //\n" +
                            "        \n" +
                            "        \n" +
                            "        return " + resourceName + "Array;\n" +
                            "    }\n" +
                            "    \n" +
                            "    protected override object GetResourceProperty(object rawResource, string property)\n" +
                            "    {\n" +
                            "        " + resourceName + " typedResource = (" + resourceName + ")rawResource;\n" +
                            "        \n" +
                            "        switch (property)\n" +
                            "        {\n");

                    sw.Write(getterStr.ToString());

                    sw.Write("            default:\n" +
                            "                throw new Exception(\"unknown property (\" + property + \")\");\n" +
                            "        }\n" +
                            "    }\n" +
                            "    \n" +
                            "    protected override ManagedResourceOpResult SetResourceProperties(" + resourceName + " resource)\n" +
                            "    {\n" +
                            "        ManagedResourceOpResult result = new ManagedResourceOpResult();\n" +
                            "        \n" +
                            "        // Set the value of read-write " + resourceName + " properties\n");

                    sw.Write(setterStr.ToString());

                    sw.Write("        \n" +
                             "        return result;\n" +
                             "    }\n" +
                             "}\n");
                }
            }

            String tmp = modelDoc.OuterXml;
            do
            {
                int startIdx = tmp.IndexOf("<propertyDataType>");
                if (startIdx >= 0)
                {
                    tmp = tmp.Substring(startIdx + 18);
                    startIdx = tmp.IndexOf("/>");
                    if (startIdx >= 0)
                    {
                        tmp = tmp.Substring(startIdx + 2);
                        int endIdx = tmp.IndexOf("</propertyDataType>");
                        if (endIdx >= 0)
                        {
                            String content = tmp.Substring(0, endIdx);
                            schemaStr.Append(content);
                            tmp = tmp.Substring(endIdx + 19);
                        }
                        else
                        {
                            break;
                        }
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    break;
                }
            } while (true);

            schemaStr.Append("\n</xs:schema>\n");

            // Write schema into temporary file.
            string schemaFile = textBox2.Text + "\\systemSchema.xsd";
            using (StreamWriter schemaStream = new StreamWriter(schemaFile))
            {
                schemaStream.Write(schemaStr.ToString());
            }

            // Load as a schema.
            XmlSchema xsd;
            using (FileStream stream = new FileStream(schemaFile, FileMode.Open, FileAccess.Read))
            {
                xsd = XmlSchema.Read(stream, null);
            }

            // Compile the schema.
            XmlSchemas xsds = new XmlSchemas();
            xsds.Add(xsd);
            xsds.Compile(null, true);
            XmlSchemaImporter schemaImporter = new XmlSchemaImporter(xsds);

            // create the codedom
            CodeNamespace codeNamespace = new CodeNamespace(); // ("Generated");
            XmlCodeExporter codeExporter = new XmlCodeExporter(codeNamespace);

            // Create the mappings.
            List<XmlTypeMapping> maps = new List<XmlTypeMapping>();
            foreach (XmlSchemaType schemaType in xsd.SchemaTypes.Values)
            {
                maps.Add(schemaImporter.ImportSchemaType(schemaType.QualifiedName));
            }
            foreach (XmlSchemaElement schemaElement in xsd.Elements.Values)
            {
                maps.Add(schemaImporter.ImportTypeMapping(schemaElement.QualifiedName));
            }
            foreach (XmlTypeMapping map in maps)
            {
                codeExporter.ExportTypeMapping(map);
            }

            // Check for invalid characters in identifiers
            CodeGenerator.ValidateIdentifiers(codeNamespace);

            // output the C# code
            CSharpCodeProvider codeProvider = new CSharpCodeProvider();

            string tmpTypeFile = textBox2.Text + "\\system.cs";
            using (StreamWriter writer = new StreamWriter(tmpTypeFile))
            {
                codeProvider.GenerateCodeFromNamespace(codeNamespace, writer, new CodeGeneratorOptions());
            }

            foreach (string f in generateTypeFiles)
            {
                File.Copy(tmpTypeFile, f);
            }

            // Delete the temporary files.
            File.Delete(tmpTypeFile);
            File.Delete(schemaFile);

            // Inform the user.
            Success successDialog = 
                new Success("Manageability adaptor stubs generated successfully under\n" +
                textBox2.Text + ".\n\n\nUse the tool to generate more adaptors or exit.");
            successDialog.ShowDialog();
        }

        private string MakePropertyType(string resource, string property)
        {
            return resource + property.Substring(0, 1).ToUpper() + property.Substring(1);
        }

        private String getTextValue(XmlElement ele, String tagName)
        {
            String textVal = null;
            XmlNodeList nl = ele.GetElementsByTagName(tagName);
            if (nl != null && nl.Count > 0)
            {
                XmlElement el = (XmlElement)nl[0];
                textVal = el.FirstChild.Value;
            }

            return textVal;
        }

        private bool ValidateModel(XmlDocument model)
        {
            return true;
        }

        private void textBox2_TextChanged(object sender, EventArgs e)
        {
            if (textBox1.Text.Length > 0 && textBox2.Text.Length > 0)
            {
                button3.Enabled = true;
            }
        }

        private void textBox1_TextChanged(object sender, EventArgs e)
        {
            if (textBox1.Text.Length > 0 && textBox2.Text.Length > 0)
            {
                button3.Enabled = true;
            }
        }

        private bool IsString(XmlElement property)
        {
            return property.InnerXml.Contains("base=\"xs:string\"");
        }
    }
}