close all

clear FV

% YOU MUST set this to the base directory of the Basel Face Model
BFMbasedir = '/Users/williamsmith/Documents/arnaud/dbs/BaselFaceModel/';

% Load morphable model
load(strcat(BFMbasedir,'01_MorphableModel.mat'));

% ADJUSTABLE PARAMETERS

% Number of model dimensions to use - this must be <= 2*number of landmarks
% otherwise linear system of equations will be underdetermined
ndims = 70;

% Number of standard deviations that each parameter is allowed to deviate
% from the mean
numsd = 2;

% Use these to set the size of the rendered images
oglp.width=400;
oglp.height=400;
eyewidth = 100;

% Distance of the original and fitted faces from the camera (units of
% micrometres):
dist_target = -30*10^4; % i.e. 30cm
dist_fit = -90*10^4;

% If true, then use the same texture for both faces, otherwise use a white
% texture and diffuse shading
usetex = true;

% If true, generate a random face as the target, otherwise load BFM scan
userandom = false;
% If loading a BFM scan, this is the ID (between 1 and 10) of the face
faceID = 1;

% If true, use the Farkas feature points, otherwise use whole mesh
usefarkas = false;

% Number of ALS iterations
niter = 5;

% END ADJUSTABLE PARAMETERS

% Colour of the ambient and diffuse light
if usetex
    oglp.i_amb_light=[1; 1; 1];
    oglp.i_dir_light=[0; 0; 0];
else
    oglp.i_amb_light=[0; 0; 0];
    oglp.i_dir_light=[1; 1; 1];
end

nverts = size(shapeMU,1)./3;

if usefarkas
    % Either use landmarks, e.g. Farkas:
    fileID = fopen(strcat(BFMbasedir,'11_feature_points/Farkas_face05.fp'));
    C = textscan(fileID,'%f %f %f %f %f %f %s','Delimiter',' ','CommentStyle','#');
    fclose(fileID);
    landmarks = C{1};
    clear C fileID
else
    % Or use all vertices:
    landmarks=1:53490;
end

% BFM indices for eye centres, used for normalising interocular distance
% in projected images
lefteye = 12150;
righteye = 4410;

FV.faces = tl;

clear pts

if userandom
% Either:
% 1. Generate a random target face
b = randn(ndims,1).*shapeEV(1:ndims);
pts = reshape(shapePC(:,1:ndims)*b+shapeMU,3,53490)';
if usetex
    % random texture
    c = randn(ndims,1).*texEV(1:ndims);
    FV.facevertexcdata = min(1,max(0,reshape(texPC*c+texMU,3,53490)'./255));
    clear c
else
    % white texture
    FV.facevertexcdata = ones(size(FV.vertices));
end

else
% Or:
% 2. Load a target face from a ply file
files = dir(strcat(BFMbasedir,'03_scans_ply/*.ply'));

% Remove mystery 11th face!
if faceID>=6
    faceID=faceID+1;
end

[data,~]=plyread(strcat(BFMbasedir,'03_scans_ply/',files(faceID).name));
pts(:,1)=data.vertex.x;
pts(:,2)=data.vertex.y;
pts(:,3)=data.vertex.z;
FV.vertices = pts;
if usetex
    % use actual texture
    FV.facevertexcdata(:,1)=double(data.vertex.red)./255;
    FV.facevertexcdata(:,2)=double(data.vertex.green)./255;
    FV.facevertexcdata(:,3)=double(data.vertex.blue)./255;
else
    % white texture
    FV.facevertexcdata = ones(size(FV.vertices));
end
end

dist_target = dist_target-max(pts(:,3)); % Make sure translation is from nose tip
dist_fit = dist_fit-max(pts(:,3));

% Subselect landmark pts and make homogeneous:
pts = double(pts(landmarks,:));
pts(:,4)=1;
pts = pts';

% Render an orthographic image of target face
scale = eyewidth/norm(FV.vertices(lefteye,:)-FV.vertices(righteye,:));
T = [scale 0 0 oglp.width/2; 0 scale 0 oglp.height/2; 0 0 scale 0];
target_ortho = render_face_ortho(FV,T,oglp);
R = [cosd(45) 0 sind(45) 0; 0 1 0 0; -sind(45) 0 cosd(45) 0; 0 0 0 1];
T = [scale 0 0 oglp.width/2; 0 scale 0 oglp.height/2; 0 0 scale 0; 0 0 0 1];
T = R*T;
T = T(1:3,:);
target_ortho_rot = render_face_ortho(FV,T,oglp);

% Perspective projection of target face
phi=1;
K = [phi 0 0; 0 phi 0; 0 0 1];
T = [1 0 0 0; 0 1 0 0; 0 0 1 dist_target];

% Calculate correct scale for perspective rendering
eyel = K*T*[squeeze(FV.vertices(lefteye,:)) 1]';
eyel = [eyel(1)/eyel(3) eyel(2)/eyel(3)];
eyer = K*T*[squeeze(FV.vertices(righteye,:)) 1]';
eyer = [eyer(1)/eyer(3) eyer(2)/eyer(3)];
targeteyedist = norm(eyel-eyer);
scale = eyewidth/targeteyedist;
clear eyel eyer

% Render a perspective image of target face
K2 = [phi*scale 0 oglp.width/2; 0 phi*scale oglp.width/2; 0 0 1]; 
target_persp = render_face(FV,K2,T,oglp);

target_scale = scale;

% Perspective projection of target points
homo2d = K*T*(pts);
x=homo2d(1,:)./homo2d(3,:);
y=homo2d(2,:)./homo2d(3,:);

% PERSPECTIVE FITTING ALGORITHM STARTS HERE

% Create reduced versions of PCs and average:
sortedfps = reshape(1:nverts*3,3,nverts)';
fps_sel = sortedfps(landmarks,1:3)';
fps_sel = fps_sel(:);
sortedfps = fps_sel;
P = double(shapePC(sortedfps,1:ndims));
mu = double(shapeMU(sortedfps,:));
ev = double(shapeEV(1:ndims));
P = reshape(P,3,size(P,1)/3,ndims);
mu = reshape(mu,3,length(landmarks))';

% Preallocate space for matrix:
C = zeros(3*length(landmarks),ndims);

% Construct linear inequality constraint matrix:
A = [eye(ndims); -eye(ndims)];    
e = [numsd.*ev; numsd.*ev];

options = optimset('LargeScale','off','Display','off');

% Initialise phi using mean shape
pts = reshape(shapeMU,3,53490)';
pts = double(pts(landmarks,:));
pts(:,4)=1;
pts = pts';
K = [1 0 0; 0 1 0; 0 0 1];
T = [1 0 0 0; 0 1 0 0; 0 0 1 dist_fit];
tauz=dist_fit;
homo2d = K*T*(pts);
x2=homo2d(1,:)./homo2d(3,:);
y2=homo2d(2,:)./homo2d(3,:);
phi = ([x2'; y2']\[x'; y']);

for iter=1:niter
    % Construct linear system of equations
    for i=1:length(landmarks)
        C(3*i-2,:)=y(i).*squeeze(P(3,i,:)) - phi.*squeeze(P(2,i,:));
        C(3*i-1,:)=phi.*squeeze(P(1,i,:)) - x(i).*squeeze(P(3,i,:));
        C(3*i,:)=x(i).*squeeze(P(2,i,:)) - y(i).*squeeze(P(1,i,:));
    
        d(3*i-2,1)=phi*mu(i,2) - y(i)*mu(i,3) - tauz*y(i);
        d(3*i-1,1)=x(i)*mu(i,3) - phi*mu(i,1) + tauz*x(i);
        d(3*i,1)=y(i)*mu(i,1)-x(i)*mu(i,2);
    end

    % Solve constrained linear system
    a = lsqlin(C,d,A,e,[],[],[],[],[],options);

    % Update estimate of phi
    pts = reshape(shapePC(:,1:ndims)*a+shapeMU,3,53490)';
    pts = double(pts(landmarks,:));
    pts(:,4)=1;
    pts = pts';
    K = [phi 0 0; 0 phi 0; 0 0 1];
    T = [1 0 0 0; 0 1 0 0; 0 0 1 tauz];
    homo2d = K*T*(pts);
    x2=homo2d(1,:)./homo2d(3,:);
    y2=homo2d(2,:)./homo2d(3,:);
    phi = ([x2'; y2']\[x'; y'])*phi;
    disp(['Iteration ' num2str(iter) ': Error=' num2str(sum((x-x2).^2+(y-y2).^2))]);
end
clear C d

FV2.faces = tl;
FV2.facevertexcdata = FV.facevertexcdata;
FV2.vertices = reshape(shapePC(:,1:ndims)*a+shapeMU,3,53490)';

% Calculate correct scale for perspective rendering
eyel = K*T*[squeeze(FV2.vertices(lefteye,:)) 1]';
eyel = [eyel(1)/eyel(3) eyel(2)/eyel(3)];
eyer = K*T*[squeeze(FV2.vertices(righteye,:)) 1]';
eyer = [eyer(1)/eyer(3) eyer(2)/eyer(3)];
scale = eyewidth/norm(eyel-eyer);
clear eyel eyer

% Render a perspective image of fitted face
K2 = [phi*scale 0 oglp.width/2; 0 phi*scale oglp.width/2; 0 0 1]; 
fitted_persp = render_face(FV2,K2,T,oglp);

% Render an orthographic image of fitted face
scale = eyewidth/norm(FV2.vertices(lefteye,:)-FV2.vertices(righteye,:));
T = [scale 0 0 oglp.width/2; 0 scale 0 oglp.height/2; 0 0 scale 0];
fitted_ortho = render_face_ortho(FV2,T,oglp);
R = [cosd(45) 0 sind(45) 0; 0 1 0 0; -sind(45) 0 cosd(45) 0; 0 0 0 1];
T = [scale 0 0 oglp.width/2; 0 scale 0 oglp.height/2; 0 0 scale 0; 0 0 0 1];
T = R*T;
T = T(1:3,:);
fitted_ortho_rot = render_face_ortho(FV2,T,oglp);

% Compute mean Euclidian distance between surfaces
err3D = mean(sqrt((FV2.vertices(:,1)-FV.vertices(:,1)).^2+(FV2.vertices(:,2)-FV.vertices(:,2)).^2+(FV2.vertices(:,3)-FV.vertices(:,3)).^2));
disp(['Mean Euclidian distance between surfaces = ' num2str(err3D/10^3) 'mm']);

% Compute mean landmark error
err2D = mean(sqrt((x-x2).^2+(y-y2).^2));
disp(['Mean distance between landmarks = ' num2str((err2D/targeteyedist)*100) '% (percentage of interocular distance)']);

disp(['Mahalanobis distance of fitting from mean = ' num2str(sum((a./shapeEV(1:ndims)).^2))]);

figure; 
subplot(1,4,1)
imshow(target_persp)
title('Target perspective')
subplot(1,4,2)
imshow(fitted_persp)
title('Fitted perspective')
subplot(1,4,3)
imshow(target_ortho)
title('Target orthographic')
subplot(1,4,4)
imshow(fitted_ortho)
title('Fitted orthographic')

if usefarkas
    % Display target and fitted landmarks over the target rendering
    target_persp(isnan(target_persp))=1;
    figure; 
    imshow(target_persp)
    hold
    xt = -x*target_scale+oglp.width/2;
    yt = y*target_scale+oglp.height/2;
    x2t = -x2*target_scale+oglp.width/2;
    y2t = y2*target_scale+oglp.height/2;
    plot(xt,yt,'og','MarkerSize',10)
    plot(x2t,y2t,'xk','MarkerSize',10)
    % If you want to grab an image of this, do:
    % iptsetpref('ImshowBorder','tight');
    % plotted_landmarks = frame2im(getframe(gcf));
    clear xt yt x2t y2t
end


% 3D views
if usetex
    figure; patch(FV, 'FaceColor', 'interp', 'EdgeColor', 'none', 'FaceLighting', 'phong'); 
    light('Position',[0 0 1],'Style','infinite');
    axis equal; axis off;

    figure; patch(FV2, 'FaceColor', 'interp', 'EdgeColor', 'none', 'FaceLighting', 'phong'); 
    light('Position',[0 0 1],'Style','infinite');
    axis equal; axis off;
else
    figure; patch(FV, 'FaceColor', [1 1 1], 'EdgeColor', 'none', 'FaceLighting', 'phong'); 
    light('Position',[0 0 1],'Style','infinite');
    axis equal; axis off;

    figure; patch(FV2, 'FaceColor', [1 1 1], 'EdgeColor', 'none', 'FaceLighting', 'phong'); 
    light('Position',[0 0 1],'Style','infinite');
    axis equal; axis off;
end 