function Dsym_vertex = compute_dist_sym(FV, sym_indices, options)
%COMPUTE_DIST_SYM   Compute Distances Between Symmetric Vertices.
%   Dsym_vertex = compute_dist_sym(FV, sym_indices, options) outputs the
%   geodesic distances between symmetric vertices on a triangular mesh.
%
%   Input arguments:
%       * FV: structure representing a mesh with the following fields:
%           - faces: array of size #faces x 3 where rows contain the 3 vertex indices of the respective faces;
%           - vertices: array of size #vertices x 3 where rows contain the 3 spatial coordinates of the respective vertices.
%       * sym_indices: array of size #symmetric vertex pairs x 2 where rows represent the symmetric vertex indices for the respective symmetric vertex pairs.
%       * options: structure of options with the following fields:
%           - verbose: boolean to display the progress of calculation (default: false).
%
%   Output arguments:
%       * Dsym_vertex: vector of length #symmetric vertex pairs representing the geodesic distances between symmetric vertices.
%
%   Copyright (c) 2013, Arnaud Dessein (University of York)

% Options
if nargin < 3
    options = struct;
end

% Verbose
verbose = false;
if isfield(options, 'verbose')
    verbose = options.verbose;
end

% Initialization
Nsym_vertices   = size(sym_indices, 1);
Dsym_vertex     = inf(Nsym_vertices, 1);

if verbose
    h = waitbar(0, 'Computing the distances between symmetric vertices.');
end

% Compute the distances between symmetric vertices
for i = 1: Nsym_vertices   
     
    D               = fast_marching_mesh(FV, sym_indices(i, 1), sym_indices(i, 2), []);
    Dsym_vertex(i)  = D(sym_indices(i, 2));
    
    if verbose
        waitbar(i / Nsym_vertices, h)
    end
    
end

if verbose
    close(h);
end

end