function D = fast_marching_mesh(FV, start_vertices, end_vertices, L)
%FAST_MARCHING_MESH     Fast Marching on a Mesh.
%   D = fast_marching_mesh(FV, start_vertices, end_vertices, L)
%   outputs the geodesic distances from an initial front to the vertices of 
%   a triangular mesh. This is a wrapper to Gabriel Peyr's MEX-file for 
%   fast marching on a triangular mesh.
%
%   Input arguments:
%       * FV: structure representing a mesh with the following fields:
%           - faces: array of size #faces x 3 where rows contain the 3 vertex indices of the respective faces;
%           - vertices: array of size #vertices x 3 where rows contain the 3 spatial coordinates of the respective vertices.
%       * start_vertices: vector of vertex indices representing the initial front (can be set to a single point).
%       * end_vertices: vector of vertex indices representing the vertices to reach before stopping propagation (can be left empty to propagate over the whole mesh).
%       * L: vector of length #vertices representing a constraint map to restrict propagation to vertices whose distance is less than their respective entries (can be left empty for no constraint, or have entries set to +/- infinity for no constraint or exclusion of vertices).
%
%   Output arguments:
%       * D: vector of length #vertices representing the geodesic distances from the respective vertices to the initial front (infinity if propagation stopped before reaching vertex).
%
%   Copyright (c) 2013, Arnaud Dessein (University of York)

% Get the number of vertices
Nvertices = size(FV.vertices, 1);

% Build uniform vertex weights
W = ones(Nvertices, 1);

% Format the constraint map from MATLAB to C/C++
L(L == -inf) = -1e9;
L(L == +inf) = +1e9;

% Perform front propagation on the mesh
D = perform_front_propagation_mesh(FV.vertices.', FV.faces.' - 1, W, start_vertices - 1, end_vertices - 1, Nvertices, [], L, [], 1e9);

% Format the distances from C/C++ to MATLAB
D(D >= 1e9) = inf;

end