function FVC = synthesize_face_model(tl, shapeMU, shapePC, shapeEV, texMU, texPC, texEV, oglp)
%SYNTHESIZE_FACE_MODEL  Synthesize a Face from a Model.
%   FVC = synthesize_face_model(tl, shapeMU, shapePC, shapeEV, texMU, texPC, texEV, oglp)
%   outputs the textured mesh of a synthesized face by random generation 
%   from a morphable model.
%
%   Input arguments:
%       * tl: array of size #faces x 3 where rows contain the 3 vertex indices of the respective faces;
%       * shapeMU: vector of length 3#vertices where entries contain the 3 spatial coordinates of the respective vertices stacked;
%       * shapePC: array of size 3#vertices x #modes where columns contain the 3 spatial coordinates of the respective vertices stacked for the different principal components;
%       * shapeEV: vector of length #modes where entries contain the eigenvalues of the respective principal components for shape;
%       * texMU: vector of length 3#vertices where entries contain the 3 color coordinates of the respective vertices stacked;
%       * texPC: array of size 3#vertices x #modes where columns contain the 3 color coordinates of the respective vertices stacked for the different principal components;
%       * texEV: vector of length #modes where entries contain the eigenvalues of the respective principal components for texture;
%       * oglp: vector of length #lighting parameters where entries are structures representing the candidate lighting parameters in OpenGL format for rendering.
%
%   Output arguments:
%       * FVC: structure representing a textured mesh with the following fields:
%           - faces: array of size #faces x 3 where rows contain the 3 vertex indices of the respective faces;
%           - vertices: array of size #vertices x 3 where rows contain the 3 spatial coordinates of the respective vertices;
%           - facevertexcdata: array of size #vertices x 3 where rows contain the 3 color coordinates of the respective vertices.
%
%   Copyright (c) 2013, Arnaud Dessein (University of York)

% Build the faces
FVC.faces = tl;

% Generate a random shape
FVC.vertices = reshape(shapeMU + shapePC * (randn(length(shapeEV), 1) .* shapeEV), 3, length(shapeMU) / 3).';

% Generate a random texture
FVC.facevertexcdata	= reshape(texMU + texPC * (randn(length(texEV), 1) .* texEV), 3, length(texMU) / 3).';
FVC.facevertexcdata = max(0, min(1, FVC.facevertexcdata / 255));

% Rendering
if nargin > 7 && ~isempty(oglp)
    FVC = render_texture_mesh(FVC, oglp(randi(length(oglp))));
end

end